/* $Id: stringio.c 715 2006-05-29 21:02:00Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <assert.h>

#include "stringio.h"

struct stringio_out_data
{
  char *out_buffer;
  size_t *out_count;
  size_t out_buffer_len, out_buffer_pos;
  bool out_own_buffer, out_discard_excess;
};


static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct stringio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  const size_t space = data->out_buffer_len - data->out_buffer_pos;

  if (0 == space)
    {
      // Buffer is full - refuse or discard

      if (!data->out_discard_excess)
        return OUTPUT_ERR_REFUSED;

      iobuffer_mark_taken (buf, max);
      return OUTPUT_OK;
    }

  if (space < max)
    max = space;

  char *src = iobuffer_data_pointer (buf);
  memcpy (data->out_buffer + data->out_buffer_pos, src, max);
  iobuffer_mark_taken (buf, max);

  data->out_buffer_pos += max;

  if (data->out_count)
    *data->out_count += max;

  return OUTPUT_OK;
}

static void
my_release_out (void *uncast_data)
{
  struct stringio_out_data *data = uncast_data;
  assert (data);

  if (data->out_own_buffer)
    free (data->out_buffer);

  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = NULL,          // Unimplemented
  .flush = NULL,                // Unimplemented
  .close_out = NULL,            // Unimplemented
  .release_out = my_release_out
};

out_stream_t
stringio_open_out (char *buf, size_t buf_len, size_t * out_count,
                   bool discard_excess, bool own_buf)
{

  struct stringio_out_data *data = malloc (sizeof (struct stringio_out_data));
  if (!data)
    return NULL;

  data->out_buffer = buf;
  data->out_buffer_len = buf_len;
  data->out_buffer_pos = 0;
  data->out_own_buffer = own_buf;
  data->out_count = out_count;
  data->out_discard_excess = discard_excess;

  return open_out (&my_out_type, data);
}

struct stringio_in_data
{
  char *in_buffer;
  size_t in_buffer_len, in_buffer_pos;
  bool in_own_buffer;
};

static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct stringio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  const size_t left = data->in_buffer_len - data->in_buffer_pos;

  if (0 == left)
    return INPUT_EOF;

  if (left < max)
    max = left;

  char *dst = iobuffer_free_pointer (buf);
  memcpy (dst, data->in_buffer + data->in_buffer_pos, max);

  iobuffer_mark_added (buf, max);
  data->in_buffer_pos += max;

  return INPUT_OK;
}

static void
my_release_in (void *uncast_data)
{
  struct stringio_in_data *data = uncast_data;
  assert (data);

  if (data->in_own_buffer)
    free (data->in_buffer);

  free (data);
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = NULL,           // Unimplemented
  .close_in = NULL,             // Unimplemented
  .release_in = my_release_in
};

in_stream_t
stringio_open_in (char *buf, size_t buf_len, bool own_buf)
{
  struct stringio_in_data *data = malloc (sizeof (struct stringio_in_data));
  if (!data)
    return NULL;

  data->in_buffer = buf;
  data->in_buffer_len = buf_len;
  data->in_buffer_pos = 0;
  data->in_own_buffer = own_buf;

  return open_in (&my_in_type, data);
}
