
#include <assert.h>

#include "logging.h"
#include "list.h"

struct node
{
  void *node_data;
  struct node *node_next, *node_prev;
};

struct block
{
  struct block *block_next;
  struct node block_nodes[];
};

struct list
{
  struct node *list_start, *list_end;
  size_t list_len;
  int list_modifications;
  next_size_f list_next_size;
  void *list_next_size_data;
  struct block list_first_block; // Must be last
};

struct iterator
{
  struct node *iterator_next;
  list_t iterator_list;
  int iterator_list_modifications;
  bool iterator_has_more;
};

list_t
empty_list_full (size_t default_space, next_size_f next_size,
                 void *next_size_data)
{
  if (default_space < 1)
    default_space = 1;

  assert (next_size);

  list_t list = malloc (sizeof (struct list) +
                        (default_space * sizeof (struct node)));
  if (!list)
    return NULL;

  list->list_len = 0;
  list->list_next_size = next_size;
  list->list_next_size_data = next_size_data;
  list->list_modifications = 0;

  struct block *first_block = &list->list_first_block;
  first_block->block_next = NULL;

  for (int i = 0; i < default_space; i++)
    {
      first_block->block_nodes[i].node_prev =
        &first_block->block_nodes[(i - 1) % default_space];
      first_block->block_nodes[i].node_next =
        &first_block->block_nodes[(i + 1) % default_space];
    }

  list->list_start = list->list_end = &first_block->block_nodes[0];
  return list;
}

size_t DEFAULT_BLOCK = 8;

size_t
next_size_fixed (void *uncast_size, size_t list_size)
{
  return *((size_t *) uncast_size);
}

size_t
next_size_exp (void *uncast_size, size_t list_size)
{
  return list_size;
}

list_t
empty_list ()
{
  return empty_list_full (DEFAULT_BLOCK, next_size_exp, NULL);
}

bool
is_empty_list (list_t list)
{
  return 0 == list->list_len;
}

void
clear_list (list_t list)
{
  list->list_len = 0;
  list->list_end = list->list_start;
}

list_t
copy_list (list_t list)
{
  list_t result = empty_list();
  if (!result)
    return NULL;

  list_append_l (result, list);
  return result;
}

static void
allocate_space (list_t list)
{
  // Only need to call this when the list is empty
  assert (list->list_end->node_next == list->list_start);
  // Sanity check
  assert (list->list_start->node_prev == list->list_end);

  // Compute size of next block
  const size_t next_block_size =
    (list->list_next_size) (list->list_next_size_data, list->list_len);
  assert (next_block_size > 0);

  struct block *block =
    malloc (sizeof (struct block) + (next_block_size * sizeof (struct node)));

  if (!block)
    MEMFAILED ();

  // Add to the list of blocks
  block->block_next = list->list_first_block.block_next;
  list->list_first_block.block_next = block;

  // Initialise the nodes
  for (size_t i = 0; i < next_block_size;)
    {
      if (i > 0)
        block->block_nodes[i].node_prev = &block->block_nodes[i - 1];
      else
        block->block_nodes[i].node_prev = list->list_end;

      const size_t j = i + 1;
      if (j < next_block_size)
        block->block_nodes[i].node_next = &block->block_nodes[j];
      else
        block->block_nodes[i].node_next = list->list_start;

      i = j;
    }

  list->list_end->node_next = &block->block_nodes[0];
  list->list_start->node_prev = &block->block_nodes[next_block_size - 1];
}

void
list_snoc (list_t list, void *data)
{
  // Make end pointer point to a free node
  // When the list is empty, this is already true!
  if (list->list_len > 0)
    {
      if (list->list_end->node_next == list->list_start)
        {
          // There are no free nodes in the list
          allocate_space (list);
        }
      list->list_end = list->list_end->node_next;
    }

  list->list_end->node_data = data;
  list->list_len++;
  list->list_modifications++;
}

void *
list_head (list_t list)
{
  if (0 == list->list_len)
    {
      // The list is empty!
      return NULL;
    }

  return list->list_start->node_data;
}

void *
remove_list_head (list_t list)
{
  if (0 == list->list_len)
    {
      // The list is empty!
      return NULL;
    }

  void *data = list->list_start->node_data;
  list->list_len--;
  list->list_modifications++;
  if (list->list_len > 0)
    {
      list->list_start = list->list_start->node_next;
    }

  return data;
}

size_t
list_length (list_t list)
{
  return list->list_len;
}

void
release_list (list_t list)
{
  // Release all additional blocks
  struct block *block_tail = list->list_first_block.block_next;
  while (NULL != block_tail)
    {
      struct block *block_next = block_tail->block_next;
      free (block_tail);
      block_tail = block_next;
    }

  free (list);
}

/*
 * Iterators
 */

#define CHECK_ITERATOR(it) assert(it->iterator_list_modifications == it->iterator_list->list_modifications)

iterator_t
list_iterator (list_t list)
{
  iterator_t it = malloc (sizeof (struct iterator));
  if(!it)
    return NULL;

  it->iterator_list_modifications = list->list_modifications;
  it->iterator_list = list;
  it->iterator_next = list->list_start;
  it->iterator_has_more = list->list_len > 0;

  return it;
}

bool
iterator_has_next (iterator_t it)
{
  assert(it);
  CHECK_ITERATOR(it);

  return it->iterator_has_more;
}

void *
iterator_next (iterator_t it)
{
  assert (it);
  CHECK_ITERATOR (it);

  if (!it->iterator_has_more)
    return NULL;

  void *data = it->iterator_next->node_data;
  it->iterator_has_more = (it->iterator_next != it->iterator_list->list_end);
  it->iterator_next = it->iterator_next->node_next;

  return data;
}

void *
iterator_remove (iterator_t it)
{
  assert (it);
  CHECK_ITERATOR (it);

  struct node *removed_node = it->iterator_next->node_prev;
  // Update start/end, if necessary
  if ((--(it->iterator_list->list_len)) > 0)
    {
      if(it->iterator_list->list_start == removed_node)
        it->iterator_list->list_start = it->iterator_list->list_start->node_next;
      if(it->iterator_list->list_end == removed_node)
        it->iterator_list->list_end = it->iterator_list->list_end->node_prev;
    }

  // Remove from node
  removed_node->node_next->node_prev = removed_node->node_prev;
  removed_node->node_prev->node_next = removed_node->node_next;
  // Insert into free space
  it->iterator_list->list_end->node_next->node_prev = removed_node;
  removed_node->node_next = it->iterator_list->list_end->node_next;
  it->iterator_list->list_end->node_next = removed_node;
  removed_node->node_prev = it->iterator_list->list_end;

  it->iterator_list_modifications = ++(it->iterator_list->list_modifications);

  return removed_node->node_data;
}

void
release_iterator (iterator_t it)
{
  free(it);
}

void
list_append_l (list_t l0, list_t l1)
{
  iterator_t it = list_iterator (l1);
  while (iterator_has_next (it))
    list_snoc (l0, iterator_next (it));
  release_iterator (it);
}
