/* $Id: iobuffer.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>

#include "iobuffer.h"

bool
init_iobuffer (iobuffer_t * buf, size_t size)
{
  buf->data = malloc (size);
  if (!buf->data)
    return false;

  buf->size = size;
  buf->data_start = 0;
  buf->free_start = 0;

  return true;
}

void
init_iobuffer_with (iobuffer_t * buf, size_t size, size_t data_size,
                    char *str)
{
  assert (buf);
  assert (size > 0);
  assert (str);

  buf->data = str;
  buf->size = size;
  buf->data_start = 0;
  buf->free_start = data_size;
}

void
uninit_iobuffer (iobuffer_t * buf)
{
  if (buf->data)
    {
      free (buf->data);
      buf->data = NULL;
    }
}

iobuffer_t *
create_iobuffer (size_t size)
{
  iobuffer_t *r = malloc (sizeof (struct iobuffer));
  if (!r)
    return NULL;

  r->data = NULL;

  if (!init_iobuffer (r, size))
    {
      free (r);
      return NULL;
    }

  return r;
}

void
free_iobuffer (iobuffer_t * buf)
{
  uninit_iobuffer (buf);
  free (buf);
}

size_t
iobuffer_size (iobuffer_t * buf)
{
  return buf->size;
}

size_t
iobuffer_free_size (iobuffer_t * buf)
{
  return buf->size - buf->free_start;
}

size_t
iobuffer_data_size (iobuffer_t * buf)
{
  return buf->free_start - buf->data_start;
}

char *
iobuffer_free_pointer (iobuffer_t * buf)
{
  return buf->data + buf->free_start;
}

char *
iobuffer_data_pointer (iobuffer_t * buf)
{
  return buf->data + buf->data_start;
}

void
iobuffer_mark_added (iobuffer_t * buf, size_t bytes)
{
  buf->free_start += bytes;
  assert (buf->free_start <= buf->size);
}

void
iobuffer_mark_taken (iobuffer_t * buf, size_t bytes)
{
  buf->data_start += bytes;

  if (buf->data_start == buf->free_start)
    {
      buf->data_start = 0;
      buf->free_start = 0;
    }

  assert (buf->data_start <= buf->free_start);
  assert (buf->data_start <= buf->size);
}

bool
iobuffer_add_data (iobuffer_t * buf, const char *str, const size_t len)
{
  if (iobuffer_free_size (buf) < len)
    return false;

  char *dst = iobuffer_free_pointer (buf);
  memcpy (dst, str, len);
  iobuffer_mark_added (buf, len);

  return true;
}

char *
iobuffer_data_block (iobuffer_t * buf)
{
  return buf->data;
}

size_t
iobuffer_copy_limited (iobuffer_t * src, iobuffer_t * dst, size_t len)
{
  size_t src_size = iobuffer_data_size (src),
    dst_free = iobuffer_free_size (dst);

  if (src_size < len)
    len = src_size;

  if (dst_free < len)
    len = dst_free;

  if (0 == len)
    return 0;

  memcpy (iobuffer_free_pointer (dst), iobuffer_data_pointer (src), len);

  iobuffer_mark_taken (src, len);
  iobuffer_mark_added (dst, len);

  return len;
}

size_t
iobuffer_copy (iobuffer_t * src, iobuffer_t * dst)
{
  return iobuffer_copy_limited (src, dst, SIZE_MAX);
}

size_t
iobuffer_shunt (iobuffer_t * buf)
{
  size_t size = iobuffer_data_size (buf);
  size_t offset = buf->data_start;

  memmove (buf->data, (buf->data) + offset, size);
  buf->data_start = 0;
  buf->free_start -= offset;
  return offset;
}

void
iobuffer_mark_all_taken (iobuffer_t * buf)
{
  buf->data_start = 0;
  buf->free_start = 0;
}
