/* $Id: fixedlenio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>
#include <stdio.h>
#include <inttypes.h>

#include "logging.h"
#include "fixedlenio.h"

struct fixedio_out_data
{
  out_stream_t out_base;
  off_t out_remaining;
  bool out_marks;
};

static output_err_t
my_output_mark (void *uncast_data)
{
  struct fixedio_out_data *data = (struct fixedio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  if (!data->out_marks)
    return OUTPUT_ERR_UNSUPPORTED;

  return output_mark (data->out_base);
}

static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * buf, size_t amount)
{
  struct fixedio_out_data *data = (struct fixedio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  if (0 == data->out_remaining)
    return OUTPUT_ERR_REFUSED;

  if (data->out_remaining < amount)
    amount = data->out_remaining;

  size_t before_amount = iobuffer_data_size (buf);
  output_err_t err = output_limited (data->out_base, buf, amount);
  size_t after_amount = iobuffer_data_size (buf);
  data->out_remaining -= (before_amount - after_amount);

  return err;
}

static output_err_t
my_flush (void *uncast_data)
{
  struct fixedio_out_data *data = (struct fixedio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  return flush (data->out_base);
}

static output_err_t
my_close_out (void *uncast_data)
{
  struct fixedio_out_data *data = (struct fixedio_out_data *) uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  if (data->out_remaining > 0)
    {
      char buf = '\0';
      iobuffer_t iobuf;
      for (; data->out_remaining > 0; data->out_remaining--)
        {
          init_iobuffer_with (&iobuf, 1, 1, &buf);
          output_err_t err = output_all (data->out_base, &iobuf);
          if (OUTPUT_OK != err)
            return err;
        }
    }

  return OUTPUT_OK;
}

static void
my_release_out (void *data)
{
  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = my_output_mark,
  .flush = my_flush,
  .close_out = my_close_out,
  .release_out = my_release_out
};

out_stream_t
fixedlenio_open_out (out_stream_t out_base, off_t len)
{
  struct fixedio_out_data *data = malloc (sizeof (struct fixedio_out_data));
  if (!data)
    return NULL;

  data->out_base = out_base;
  data->out_remaining = len;
  data->out_marks = false;

  return open_out (&my_out_type, data);
}

struct fixedio_in_data
{
  in_stream_t in_base;
  off_t in_remaining;
};

input_err_t
my_input_limited (void *uncast_data, iobuffer_t * iobuffer, size_t len)
{
  struct fixedio_in_data *data = (struct fixedio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (0 == data->in_remaining)
    return INPUT_EOF;

  if (data->in_remaining < len)
    len = data->in_remaining;

  size_t size_before = iobuffer_data_size (iobuffer);
  input_err_t err = input_limited (data->in_base, iobuffer, len);
  size_t size_after = iobuffer_data_size (iobuffer);
  data->in_remaining -= (size_after - size_before);

  return err;
}

static input_err_t
my_close_in (void *uncast_data)
{
  struct fixedio_in_data *data = (struct fixedio_in_data *) uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (data->in_remaining > 0)
    {
      char buf;
      iobuffer_t iobuf;
      DEBUGF ("Skipping %" PRId64, data->in_remaining);
      for (; data->in_remaining > 0; data->in_remaining--)
        {
          init_iobuffer_with (&iobuf, 1, 0, &buf);
          input_err_t err = input_all (data->in_base, &iobuf);
          if (INPUT_OK != err)
            return err;
        }
      DEBUG ("Skipped");
    }

  return INPUT_OK;
}

void
my_release_in (void *uncast_data)
{
  struct fixedio_in_data *data = (struct fixedio_in_data *) uncast_data;
  assert (data);
  free (data);
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = NULL,           // Unimplemented
  .close_in = my_close_in,
  .release_in = my_release_in
};

in_stream_t
fixedlenio_open_in (in_stream_t base_ins, off_t len)
{
  struct fixedio_in_data *data = malloc (sizeof (struct fixedio_in_data));
  if (!data)
    return NULL;

  data->in_base = base_ins;
  data->in_remaining = len;

  return open_in (&my_in_type, data);
}
