/* $Id: filestore_common.c 716 2006-05-31 18:14:11Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <time.h>
#include <unistd.h>

#include "logging.h"
#include "filestore_common.h"

const char *TAGS[] = {
  "Filestore",
  "ArchiveDob",
  "ArchiveUid",
  "ArchiveLabel",
  "ChunkFilter",
  "ChecksumHint",
  "FileType",
  "Path",
  "Link",
  "FileData",
  "UnixPerm",
  "CreationTime",
  "ChangeTime",
  "Owner",
  "Group",
  "Checksum:MD5",
  "Checksum:SHA1",
  NULL
};

size_t *TAG_LENS = NULL;

size_t
init_tag_len (tag_t tag)
{
  int count = 0;
  while (TAGS[count])
    count++;

  TAG_LENS = malloc (sizeof (size_t) * count);

  if (!TAG_LENS)
    {
      LOG (FATAL, "Fatal: Could not allocate TAG_LENS lookup table");
      exit (100);
    }

  for (int i = 0; i < count; i++)
    TAG_LENS[i] = strlen (TAGS[i]);

  return TAG_LENS[tag];
}

/* Lookup a tag_t based on the given string.  Returns UNKNOWN_TAG is no tag
 * matches. */
tag_t
lookup_tag (const char *tag_text)
{
  // This function is algorithmically dumb, and may become a performance
  // bottleneck?
  tag_t tag = 0;
  while (TAGS[tag] && 0 != strcmp (tag_text, TAGS[tag]))
    tag++;
  return tag;
}

/* Format a file modification time into the given buffer, which must be at
 * least FORMATTED_TIME_BUFFER_SIZE characters long. */
bool
format_time(time_t t, char *buf, bool pretty)
{
  struct tm tm ;
  if (!gmtime_r (&t, &tm))
    return false;

  // eg "20060513 17:00:54"
  snprintf (buf,
            pretty ? PRETTY_FORMATTED_TIME_BUFFER_SIZE : FORMATTED_TIME_BUFFER_SIZE,
            pretty ? PRETTY_TIME_FORMAT : TIME_FORMAT,
            tm.tm_year+1900, tm.tm_mon+1, tm.tm_mday,
            tm.tm_hour, tm.tm_min, tm.tm_sec);
  return true;
}

/* Read a tm struct that is in UTC */
time_t
mktimegm (struct tm *tm)
{
  time_t ret;
  char *tz;

  tz = getenv ("TZ");
  setenv ("TZ", "", 1);
  tzset ();
  ret = mktime (tm);
  if (tz)
    setenv ("TZ", tz, 1);
  else
    unsetenv ("TZ");
  tzset();
  return ret;
}

/* Parse a modification time from the given buffer */
bool
parse_time(char *buf, time_t *result)
{
  struct tm tm;
  int n = sscanf (buf, TIME_FORMAT, &tm.tm_year, &tm.tm_mon, &tm.tm_mday,
                  &tm.tm_hour, &tm.tm_min, &tm.tm_sec) ;

  if (6 != n)
    {
      LOGF (ERROR, "Bad time: %s (Matched %d)", buf, n);
      return false;
    }

  tm.tm_year -= 1900;
  tm.tm_mon -= 1;
  tm.tm_isdst = false;

  *result = mktimegm (&tm);
  return true;
}

/* Read where a link points to.  The caller must free() the returned buffer. */
char *
read_link (path_t path)
{
  size_t linkbuflen = 512;      // Will be doubled in loop
  char *link = NULL;
  int link_len;

  // Keep doubling the size of the buffer until the contents of the  link fit
  // In reality, unlikely to need more than one go
  do
    {
      linkbuflen *= 2;
      link = realloc (link, linkbuflen);
      if (!link)
        {
          LOG (ERROR, "Path too long");
          return false;
        }
      link_len = readlink (path_str (path), link, linkbuflen);
    }
  while (link_len == linkbuflen);
  link[link_len] = '\0';

  return link;
}

