
#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <regex.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "filelib.h"
#include "logging.h"

bool
create_directories (path_t path, mode_t mode)
{
  assert (0 != mode);
  DIR *dir = opendir (path_str (path));
  if (dir)
    {
      // Sweet... job already done
      if (-1 == closedir (dir))
        {
          LOGF (ERROR, "Error closing directory: %s", path_str (path));
          return false;
        }
      return true;
    }

  // Couldn't open the directory - find out why

  if (ENOTDIR == errno)
    {
      LOGF (ERROR, "Couldn't create directory, file already exists: %s",
            path_str (path));
      return false;
    }

  if (ENOENT != errno)
    {
      LOGF (ERROR, "Couldn't create directory: %s", path_str (path));
      return false;
    }

  // No such directory
  // Create parents
  path_t parent = copy_path (path);
  remove_last_path_part (parent);
  bool parents_ok = create_directories (parent, mode);
  release_path (parent);
  if (!parents_ok)
    {
      DEBUGF ("Failed to create parent: %s", path_str (path));
      return false;
    }

  // Finally!  Create the actual directory
  if (-1 == mkdir (path_str (path), mode))
    {
      LOGF (ERROR, "Directory creation failed: %s", path_str (path));
      return false;
    }

  LOGF (VERBOSE, "Created directory: %s", path_str (path));
  return true;
}

bool
include_file (path_t path, list_t include_filters, list_t exclude_filters)
{
  bool include;

  if (!include_filters)
    include = true;
  else
    {
      include = false;
      iterator_t it = list_iterator (include_filters);

      while (iterator_has_next (it))
        {
          regex_t *re = iterator_next (it);
          assert (re);
          if (0 == regexec (re, path_str (path), 0, NULL, 0))
            {
              include = true;
              break;
            }
        }
      release_iterator (it);
    }

  if (include && exclude_filters)
    {
      iterator_t it = list_iterator (exclude_filters);
      while (iterator_has_next (it))
        {
          regex_t *re = iterator_next (it);
          assert (re);
          if (0 == regexec (re, path_str (path), 0, NULL, 0))
            {
              include = false;
              break;
            }
        }
      release_iterator (it);
    }

  return include;
}

