/* $Id: darray.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include <string.h>
#include <stdio.h>
#include <stdbool.h>
#include <assert.h>

#include "darray.h"

#ifdef NDEBUG
// If this is defined, we will skip the sanity check which (often) prevents
// accidentally passing non-darray's to function in this module.  This saves us
// some time and memory.
#define DARRAY_NO_MAGIC
#endif

/*
 * Dynamically resizeable arrays.  The array is in two parts: A header, and a
 * block of data.  A pointer inside the block of data is returned to the user,
 * and can be used like a regular pointer.  In the bytes BEFORE this user
 * returned pointer, we store a pointer to an array header.  This array header
 * contains the size of the array and other data. 
 */

#define POINTER_SIZE (sizeof(void *))
#define DARRAY_MAGIC ((int)0xC0DE)

struct darray_header
{
  size_t da_elem_size;
  size_t da_elem_total;
  size_t da_elem_used;
  size_t da_elem_pad;
#ifndef DARRAY_NO_MAGIC
  int da_magic;
#endif

  void *da_block;
};

static void *
init_darray (struct darray_header *header)
{
  // Allocate the elements, and the padding
  bool resizing = NULL != header->da_block;
  header->da_block =
    realloc (header->da_block,
             (header->da_elem_total +
              header->da_elem_pad) * header->da_elem_size);
  if (!header->da_block)
    {
      free (header);
      return NULL;
    }

  // Copy the pointer.  It's not safe to do this with an assignment, because
  // the destination address might not be correctly aligned to hold the
  // pointer.
  void *data =
    header->da_block + (header->da_elem_pad * header->da_elem_size);
  if (!resizing)
    memcpy (data - POINTER_SIZE, &header, POINTER_SIZE);

  return data;
}

void *
create_darray (size_t elem_size, size_t elem_total, size_t elem_used)
{
  struct darray_header *header = malloc (sizeof (struct darray_header));
  if (!header)
    return NULL;

  // We need to leave enough padding to store the pointer to the header
  // before the data
  header->da_elem_pad = 1;
  while ((header->da_elem_pad * elem_size) < POINTER_SIZE)
    header->da_elem_pad *= 2;

  header->da_elem_size = elem_size;
  header->da_elem_used = elem_used;
  header->da_elem_total = elem_total;

#ifndef DARRAY_NO_MAGIC
  // Store a special value so we can sanity check later
  header->da_magic = DARRAY_MAGIC;
#endif

  header->da_block = NULL;
  return init_darray (header);
}

/* Retrieve the header from a data pointer. */
static struct darray_header *
retrieve_header (void *data)
{
  struct darray_header *header;
  memcpy (&header, data - POINTER_SIZE, POINTER_SIZE);

#ifndef DARRAY_NO_MAGIC
  // Sanity check.  If the magic value isn't present, caller gave us
  // something they shouldn't have
  if (header->da_magic != DARRAY_MAGIC)
    {
      fputs
        ("Internal error: retrieve_header: darray magic not found, aborting\n",
         stderr);
      fflush (stderr);
      abort ();
    }
#endif

  return header;
}

void
release_darray (void *data)
{
  assert (data);
  struct darray_header *header = retrieve_header (data);
  free (header->da_block);
  free (header);
}


/* Return the number of elements used.
 * This is less than or equal to the number of elements allocated.
 */
size_t
darray_size (void *data)
{
  return retrieve_header (data)->da_elem_used;
}

size_t
darray_elem_size (void *data)
{
  return retrieve_header (data)->da_elem_size;
}

/* Set the size of the array.  */
void *
darray_set_size (void *data, size_t size)
{
  assert (data);
  assert (size >= 0);
  struct darray_header *header = retrieve_header (data);

  header->da_elem_used = size;
  if (size <= header->da_elem_total)
    return data;

  while (header->da_elem_total < size)
    header->da_elem_total *= 2;

  return init_darray (header);
}

void *
darray_adjust_size (void *data, ssize_t size)
{
  assert (data);
  struct darray_header *header = retrieve_header (data);

  header->da_elem_used += size;
  if (header->da_elem_used <= header->da_elem_total)
    return data;

  do
    {
      header->da_elem_total *= 2;
    }
  while (header->da_elem_total < header->da_elem_used);

  return init_darray (header);
}

void *
darray_add (void *data, void *new_elem)
{
  assert (data);
  assert (new_elem);

  size_t loc = darray_size (data);
  data = darray_adjust_size (data, 1);
  size_t elem_size = darray_elem_size (data);
  memcpy (data + (loc * elem_size), new_elem, elem_size);
  return data;
}
