/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <alloca.h>
#include <errno.h>

#include "peer.h"

static void callhooks (lua_State *L, void *key, int reference, char *message)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	    
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);

	    if (message) {
	    	lua_pushstring (L, message);
	    } else {
	    	lua_pushnil (L);
	    }
	    
	    luaX_call(L, 2, 0);
	} else if (lua_istable (L, -1)) {
	    int i, n;
	    
	    n = lua_objlen (L, -1);

	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (L, -2, i + 1);

		lua_pushlightuserdata (L, key);
		lua_gettable (L, -3);

		if (message) {
		    lua_pushstring (L, message);
		} else {
		    lua_pushnil (L);
		}
		
		luaX_call (L, 2, 0);
	    }
	    
	    lua_pop (L, 1);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Peer

-(Peer *) init
{
    [super init];

    self->incoming = LUA_REFNIL;
    self->port = 0;
    memset (&self->peer, 0, sizeof (struct sockaddr));
    
    return self;
}

-(Peer *) initWithSocket: (int) file
{
    [super init];

    self->socket = file;
    self->source = malloc (sizeof (*self->source));
    self->source->queried = 0;
    self->source->length = 0;

    return self;
}

-(void) freeSocket
{
    close (self->socket);
    free (self->source);
    
    [super free];
}

-(void) prepare
{
    socklen_t length = sizeof(self->source);

    /* Check each socket for a new packet once per frame. */
    
    if (!self->source->queried) {
	self->source->queried = 1;
	self->source->length = recvfrom (self->socket,
					 self->source->message, 1024, 0,
					 &self->source->address, &length);
    }

    /* Call the message hook if the packet originated from
       our peer. */
    
    if (self->source->length > 0 &&
	(!((struct sockaddr_in *)&self->peer)->sin_addr.s_addr || 
	 ((struct sockaddr_in *)&self->source->address)->sin_addr.s_addr ==
	 ((struct sockaddr_in *)&self->peer)->sin_addr.s_addr) &&
	(!((struct sockaddr_in *)&self->peer)->sin_port ||
	 ((struct sockaddr_in *)&self->source->address)->sin_port ==
	 ((struct sockaddr_in *)&self->peer)->sin_port)) {
	callhooks (_L, self, self->incoming, self->source->message);
    }
    
    [super prepare];
}

-(void) finish
{
    self->source->queried = 0;
    
    [super finish];
}

-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "incoming")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->incoming);
    } else if (!xstrcmp(k, "host")) {
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_rawget (_L, 1);
    } else if (!xstrcmp(k, "port")) {
	lua_pushnumber (_L, self->port);
    } else  {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "message")) {
	if (lua_tostring (_L, 3) &&
	    ((struct sockaddr_in *)&self->peer)->sin_addr.s_addr &&
	    ((struct sockaddr_in *)&self->peer)->sin_port) {
	    sendto (self->socket,
	    	    lua_tostring (_L, 3), lua_strlen (_L, 3), 0,
	    	    &self->peer, sizeof(struct sockaddr));
	}
    } else if (!xstrcmp(k, "incoming")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->incoming);
        self->incoming = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "port")) {
	self->port = lua_tonumber (_L, 3);
	((struct sockaddr_in *)&self->peer)->sin_port = htons(lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "host")) {
    	const struct addrinfo hints = {
    	    0,
    	    AF_INET,
    	    SOCK_DGRAM,
    	    0,
    	    0, NULL, NULL, NULL
    	};
	
	struct addrinfo *info;
	int resolved;

    	/* Resolve the hostname. */

	lua_pushnumber (_L, self->port);
	
	resolved = getaddrinfo (lua_tostring (_L, 3),
				lua_tostring (_L, 4),
				&hints, &info);

	lua_pop (_L, 1);

	if (resolved == 0) {
	    memcpy (&self->peer, info->ai_addr, sizeof (struct sockaddr));
	    
	    printf ("Resolved %s to %s.\n",
		    lua_tostring (_L, 3),
		    inet_ntoa(((struct sockaddr_in *)&self->peer)->sin_addr));

	    lua_getmetatable (_L, 1);
	    lua_replace (_L, 1);
	    lua_rawset (_L, 1);
	} else {
	    memset (&self->peer, 0, sizeof (struct sockaddr));

	    printf ("Could not resolve hostname %s (%s).\n",
		    lua_tostring (_L, 3), strerror(resolved));
	}

	freeaddrinfo (info);
    } else {
	[super set];	
    }
}

@end
