/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <AL/al.h>
#include <AL/alc.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <time.h>

#include "node.h"

Node *list;

static void callhooks (lua_State *L, void *key, int reference)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    luaX_call(L, 1, 0);
	} else if (lua_istable (L, -1)) {
	    int i, n;
	    
	    n = lua_objlen (L, -1);

	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (L, -2, i + 1);

		lua_pushlightuserdata (L, key);
		lua_gettable (L, -3);
		luaX_call (L, 1, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

static int ancestry_tostring(lua_State *L)
{
    lua_pushstring(L, "Ancestry");
   
    return 1;
}

static int ancestry_index(lua_State *L)
{
    int i, n;

    if (lua_type (_L, 2) == LUA_TNUMBER) {
	n = lua_tonumber (_L, 2);
	lua_getmetatable (_L, 1);
	lua_pushstring (_L, "__node");
	lua_gettable (_L, -2);

	for (i = 0 ; i < n ; i += 1) {
	    lua_getmetatable (_L, -1);
	    lua_pushstring (_L, "__parent");
	    lua_gettable (_L, -2);
	    lua_replace (_L, -3);
	    lua_pop (_L, 1);
	}
    } else {
	lua_pushnil (_L);
    }

    return 1;
}

static int ancestry_newindex(lua_State *L)
{
    int i, n;

    if (lua_type (_L, 2) == LUA_TNUMBER) {
    	n = lua_tonumber (_L, 2);
    	lua_getmetatable (_L, 1);
    	lua_pushstring (_L, "__node");
    	lua_gettable (_L, -2);

    	for (i = 0 ; i < n - 1 ; i += 1) {
    	    lua_getmetatable (_L, -1);
    	    lua_pushstring (_L, "__parent");
    	    lua_gettable (_L, -2);
    	    lua_replace (_L, -3);
    	    lua_pop (_L, 1);
    	}

    	lua_pushstring (_L, "parent");
    	lua_pushvalue (_L, 3);
    	lua_settable (_L, -3);
    }

    return 0;
}

@implementation Node

-(Node *) init
{
    char *list[] = {
	"ancestry", "finish", "input", "link", "parent", "prepare",
	"step", "tag", "traverse", "unlink", "begin"
    };
    
    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];
    
    self->link = LUA_REFNIL;
    self->unlink = LUA_REFNIL;
    self->input = LUA_REFNIL;
    self->step = LUA_REFNIL;
    self->traverse = LUA_REFNIL;
    self->prepare = LUA_REFNIL;
    self->finish = LUA_REFNIL;
    self->begin = LUA_REFNIL;
    
    self->tag = 0;
    self->linked = 0;

    return self;
}

-(int) linked
{
    return self->linked;
}

-(void) toggle
{
    id child;

    /* Call the hooks first. */
    
    if (!self->linked) {
	callhooks (_L, self, self->link);
    } else {
	callhooks (_L, self, self->unlink);
    }

    /* Toggle ourselves. */
    
    self->linked = !self->linked;

    /* Recurse on the node's children. */
    
    for(child = [self children] ; child ; child = [child sister]) {
        [child toggle];
    }
}

-(void) input
{
    id child;

    /* Call the node's input hook. */
    
    callhooks (_L, self, self->input);
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child input];
    }
}

-(void) begin
{
    id child;
    
    callhooks (_L, self, self->begin);
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child begin];
    }
}

-(void) stepBy: (double) h at: (double) t
{
    id child;

    callhooks (_L, self, self->step);

    for(child = [self children] ; child ; child = [child sister]) {
	[child stepBy: h at: t];
    }
}

-(void) prepare
{
    id child;
    
    callhooks (_L, self, self->prepare);
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child prepare];
    }
}

-(void) traversePass: (int)pass
{
    id child;

    callhooks (_L, self, self->traverse);

    for(child = [self children] ; child ; child = [child sister]) {
	[child traversePass: pass];
    }
}

-(void) finish
{
    id child;
    
    callhooks (_L, self, self->finish);
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child finish];
    }
}

-(void) get
{
    const char *k;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "ancestry")) {
	lua_newtable (_L);

	lua_newtable (_L);
	lua_pushstring(_L, "__node");
	lua_pushvalue (_L, 1);
	lua_settable(_L, -3);
	lua_pushstring(_L, "__index");
	lua_pushcfunction(_L, (lua_CFunction)ancestry_index);
	lua_settable(_L, -3);
	lua_pushstring(_L, "__newindex");
	lua_pushcfunction(_L, (lua_CFunction)ancestry_newindex);
	lua_settable(_L, -3);
	lua_pushstring(_L, "__tostring");
	lua_pushcfunction(_L, (lua_CFunction)ancestry_tostring);
	lua_settable(_L, -3);
	lua_setmetatable(_L, -2);
    } else if (!xstrcmp(k, "parent")) {
	lua_getmetatable (_L, 1);
	lua_pushstring (_L, "__parent");
	lua_gettable (_L, -2);
    } else if (!xstrcmp(k, "tag")) {
	 lua_pushinteger (_L, self->tag);
    } else if (!xstrcmp(k, "link")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->link);
    } else if (!xstrcmp(k, "unlink")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->unlink);
    } else if (!xstrcmp(k, "input")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->input);
    } else if (!xstrcmp(k, "step")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->step);
    } else if (!xstrcmp(k, "traverse")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->traverse);
    } else if (!xstrcmp(k, "prepare")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->prepare);
    } else if (!xstrcmp(k, "finish")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->finish);
    } else if (!xstrcmp(k, "begin")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->begin);
    } else  {
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_gettable (_L, 1);
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "parent")) {
	if (lua_isuserdata (_L, 3)) {
	    lua_pushlightuserdata(_L, (void *)lua_topointer (_L, 1));
	    lua_pushvalue (_L, 1);
	    lua_settable (_L, 3);
	} else {
	    if (luaL_getmetafield(_L, 1, "__parent")) {
		luaL_getmetafield(_L, 1, "__key");
		lua_pushnil (_L);
		lua_settable (_L, -3);
	    }
	}
    } else if (!xstrcmp(k, "tag")) {
	self->tag = lua_tointeger (_L, 3);
    } else if (!xstrcmp(k, "unlink")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->unlink);
        self->unlink = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "link")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->link);
        self->link = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "input")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->input);
        self->input = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "step")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->step);
        self->step = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "traverse")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->traverse);
        self->traverse = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "prepare")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->prepare);
        self->prepare = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "finish")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->finish);
        self->finish = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "begin")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->begin);
        self->begin = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	Node *child;
	const char *name;

        /* Check if there's a node linked with
	   the same key and if so free it. */

	lua_getmetatable (_L, 1);
	lua_pushvalue (_L, 2);
	lua_gettable (_L, -2);
        
	if(lua_isuserdata (_L, -1)) {
	    child = *(Node **)lua_touserdata (_L, -1);
            
	    if ([child isKindOf: [Node class]]) {
		if([child linked]) {
		    [child toggle];
		}
		
		[self renounce: child];
		[child adoptedBy: nil];
	    }

	    lua_getmetatable (_L, -1);
	    lua_pushnil (_L);
	    lua_setfield (_L, -2, "__parent");
	    lua_pushnil (_L);
	    lua_setfield (_L, -2, "__key");
	    lua_pop (_L, 1);
	}
	
	lua_pop (_L, 1);
	
	/* Link the new node. */
	
	if(lua_isuserdata (_L, 3)) {
	    child = *(Node **)lua_touserdata (_L, 3);

	    if ([child isKindOf: [Node class]]) {
		/* Unlink the child if it already has a parent... */

		if (luaL_getmetafield (_L, 3, "__parent")) {
		    luaL_getmetafield (_L, 3, "__key");
		    lua_pushnil (_L);
		    lua_settable (_L, -3);
		    lua_pop (_L, 1);
		}

		/* ...and link to the new parent. */

		lua_getmetatable (_L, 3);
		lua_pushvalue (_L, 1);
		lua_setfield (_L, -2, "__parent");
		lua_pushvalue (_L, 2);
		lua_setfield (_L, -2, "__key");
		lua_pop (_L, 1);
	    
		name = lua_tostring (_L, 2);

		[self adopt: child named: name];
		[child adoptedBy: self];
	
		if([self linked]) {
		    [child toggle];
		}
	    }
	}
        
	/* Update the children table. */

	lua_replace (_L, 1);
	lua_settable (_L, 1);
    }
}

@end
