/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "joint.h"
#include "body.h"

static void callhooks (lua_State *L, void *self, void *a, void *b, int reference)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");

	    lua_pushlightuserdata (L, a);
	    lua_gettable (L, -2);

	    lua_pushlightuserdata (L, b);
	    lua_gettable (L, -3);

	    lua_pushlightuserdata (L, self);
	    lua_gettable (L, -4);

	    lua_replace (L, -4);
	    
	    luaX_call(L, 3, 0);
	} else if (lua_istable (L, -1)) {
	    int i, n;
	    
	    n = lua_objlen (L, -1);

	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (L, -2, i + 1);

		lua_pushlightuserdata (L, self);
		lua_gettable (L, -3);
		lua_pushlightuserdata (L, a);
		lua_gettable (L, -4);
		lua_pushlightuserdata (L, b);
		lua_gettable (L, -5);

		luaX_call (L, 3, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Joint

-(Joint *) init
{
    char *list[] = {"attach", "forces", "inverted", "torques"};

    /* Get the configuration. */
    
    lua_getglobal (_L, "options");

    lua_getfield (_L, -1, "drawjoints");
    self->debug = lua_toboolean (_L, -1);
    lua_pop (_L, 2);

    /* Initialize the object. */
    
    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->inverted = 0;
    self->attach = LUA_REFNIL;
    
    dJointSetFeedback ([self joint], &self->feedback);
    
    return self;
}

-(void) free
{
    dJointDestroy (self->joint);
    
    [super free];
}

-(dJointID) joint
{
    return self->joint;
}

-(void) update
{
    if ([self linked]) {
	id p, q;
	dBodyID a, b;

	/* Resolve the parent body. */
    
	if ([self parent] && [[self parent] isKindOf: [Body class]]) {
	    p = [self parent];
	    a = [[self parent] body];
	} else {
	    p = nil;
	    a = NULL;
	}

	/* Resolve the child body. */

	for (q = [self children];
	     q && ![q isKindOf: [Body class]];
	     q = [q sister]);
    
	if (q) {
	    b = [q body];
	} else {
	    b = NULL;
	}

	if (self->inverted) {
	    dJointAttach ([self joint], b, a);
	    callhooks (_L, self, q, p, self->attach);
	} else {
	    dJointAttach ([self joint], a, b);
	    callhooks (_L, self, p, q, self->attach);
	}
    } else {
	dJointAttach ([self joint], NULL, NULL);
    }	
}

-(void) toggle
{
    [super toggle];
    [self update];
}

-(void) adoptedBy: (Item *)parent
{
    [super adoptedBy: parent];

    if ([parent isKindOf: [Body class]]) {
	[self update];
    }
}

-(void) adopt: (Item *)child named: (char *)name
{
    [super adopt: child named: name];
    
    if ([child isKindOf: [Body class]]) {
	[self update];
    }
}

-(void) renounce: (Item *)child
{
    [super renounce: child];
    
    if ([child isKindOf: [Body class]]) {
	[self update];
    }
}

-(void) get
{
    const char *k;
    int i;
    
    k = lua_tostring (_L, -1);

    if (!xstrcmp(k, "forces")) {
        lua_newtable (_L);

	/* The force applied on the first body. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->feedback.f1[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 1);

	/* The force applied on the second body. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->feedback.f2[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 2);
    } else if (!xstrcmp(k, "torques")) {
        lua_newtable (_L);

	/* The torque applied on the first body. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->feedback.t1[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 1);

	/* The torque applied on the second body. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->feedback.t2[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 2);	
    } else if (!xstrcmp(k, "inverted")) {
	lua_pushboolean (_L, self->inverted);
    } else if (!xstrcmp(k, "attach")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->attach);
    } else {
	[super get];
    }
}

-(void) set
{

    const char *k;

    k = lua_tostring (_L, -2);

    if (!xstrcmp(k, "inverted")) {
	self->inverted = lua_toboolean (_L, 3);

	[self update];
    } else if (!xstrcmp(k, "attach")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->attach);
        self->attach = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	[super set];
    }
}

@end
