/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "transform.h"
#include "bound.h"

@implementation Bound

-(Bound *) init
{
    char *list[] = {"size"};

    self = [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];
    
    self->size[0] = 1;
    self->size[1] = 1;
    self->size[2] = 1;

    return self;
}

-(void) get
{
    const char *k;
    int i;
    
    k = lua_tostring (_L, -1);

    if (!xstrcmp(k, "size")) {
	lua_newtable (_L);
        
	for(i = 0; i < 3; i += 1) {
	    lua_pushnumber (_L, self->size[i]);
	    lua_rawseti (_L, -2, i + 1);
	}
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i;

    k = lua_tostring (_L, -2);

    if (!xstrcmp(k, "size")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->size[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
	}
    } else {
	[super set];
    }
}

-(void) traversePass: (int)pass
{
    if (pass == 1 && self->debug) {
	id root;
	double *l;

    	glUseProgramObjectARB(0);

    	glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
	glEnable (GL_DEPTH_TEST);

    	glColor3d (0, 1, 0);
    	glLineWidth (1);

	l = self->size;

	for (root = self;
	     ![root isKindOf: [Transform class]];
	     root = [root parent]);

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([root matrix]);
	    
	glBegin (GL_QUAD_STRIP);
	glVertex3d(-0.5 * l[0], -0.5 * l[1], -0.5 * l[2]);
	glVertex3d(-0.5 * l[0], -0.5 * l[1], 0.5 * l[2]);
	glVertex3d(-0.5 * l[0], 0.5 * l[1], -0.5 * l[2]);
	glVertex3d(-0.5 * l[0], 0.5 * l[1], 0.5 * l[2]);

	glVertex3d(0.5 * l[0], 0.5 * l[1], -0.5 * l[2]);
	glVertex3d(0.5 * l[0], 0.5 * l[1], 0.5 * l[2]);
 
	glVertex3d(0.5 * l[0], -0.5 * l[1], -0.5 * l[2]);
	glVertex3d(0.5 * l[0], -0.5 * l[1], 0.5 * l[2]);
       
	glVertex3d(-0.5 * l[0], -0.5 * l[1], -0.5 * l[2]);
	glVertex3d(-0.5 * l[0], -0.5 * l[1], 0.5 * l[2]);
	glEnd();

	glPopMatrix();
	
	glDisable (GL_DEPTH_TEST);
    	glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
    }
    
    [super traversePass: pass];
}

-(void) finish
{
    id object;
    double n_b[3], v_n[3], *R, *r;
    int i;

    /* Find our parent for transformation purposes. */
    
    for (object = self;
	 ![object isKindOf: [Transform class]];
	 object = [object parent]);

    r = [object translation];
    R = [object rotation];

    /* Assume the target is inside. */
    
    self->target = 1;

    /* Loop through all six planes. */

    for (i = 0 ; i < 6 ; i += 1) {
	/* Project the plane normal to the bounds base. */
	
    	n_b[0] = R[0] * self->planes[i][0] +
    	         R[3] * self->planes[i][1] +
    	         R[6] * self->planes[i][2];

    	n_b[1] = R[1] * self->planes[i][0] +
    	         R[4] * self->planes[i][1] +
    	         R[7] * self->planes[i][2];

    	n_b[2] = R[2] * self->planes[i][0] +
    	         R[5] * self->planes[i][1] +
    	         R[8] * self->planes[i][2];

	/* Calculate the "positive" vertex. */
	
    	v_n[0] = r[0] + (n_b[0] >= 0 ? 1 : -1) * 0.5 * self->size[0];
    	v_n[1] = r[1] + (n_b[1] >= 0 ? 1 : -1) * 0.5 * self->size[1];
    	v_n[2] = r[2] + (n_b[2] >= 0 ? 1 : -1) * 0.5 * self->size[2];

	/* And test it against the plane. */
	
    	if (v_n[0] * self->planes[i][0] +
    	    v_n[1] * self->planes[i][1] +
    	    v_n[2] * self->planes[i][2] + self->planes[i][3] < 0) {
	    
    	    self->target = 0;
	    break;
    	}
    }

    [super finish];
}

@end
