/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <GL/glu.h>
#include <lua.h>
#include <lauxlib.h>
#include "annotation.h"

@implementation Annotation

-(Annotation *) init
{
    char *list[] = {
	"angle", "node", "radius", "size", "thickness"
    };

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->thickness = 1;
    self->radius = 1;
    self->angle = 0;

    return self;
}

-(void) transform
{
    /* Transformation will be deffered to the
       traversal phase. */
}

-(void) traversePass: (int)pass
{
    double x, y;

    if (pass == 2) {
	id child;
	double M[16], P[16], w[3], q[3], p[3];
	double I[9] = {1, 0, 0,  0, 1, 0,  0, 0, 1};
	int v[4];

	/* Transform the node and its subtree to its
	   final resting place. This can be done during
	   traversal only to be sure that the modelview
	   matrix is finalized. */

	[super transform];
	
	for (child = [self children];
	     child && ![child isKindOf:[Widget class]];
	     child = [child sister]);
  
	/* Calculate our own allocation based on our children. */
    
	self->minimum[0] = 2 * self->padding[0];
	self->minimum[1] = 2 * self->padding[1];
  
	if(child) {
	    float width, height;

	    width = [child measureWidth] + 2 * self->padding[0];
	    height = [child measureHeight] + 2 * self->padding[1];
	
	    self->minimum[0] = width > self->minimum[0] ?
		width : self->minimum[0];

	    self->minimum[1] = height > self->minimum[1] ?
		height : self->minimum[1];
	}

	self->allocated[0] = self->minimum[0] > self->requested[0] ?
	    self->minimum[0] : self->requested[0];
	self->allocated[1] = self->minimum[1] > self->requested[1] ?
	    self->minimum[1] : self->requested[1];
	
	/* Now reset the children. */
    
	if(child) {
	    /* Align the child properly. */
    
	    if (self->align[0] < 0) {
		[child position][0] = 0.5 * (self->minimum[0] -
					     self->allocated[0]);
	    } else if (self->align[0] > 0) {
		[child position][0] = 0.5 * (self->allocated[0] -
					     self->minimum[0]);
	    } else {
		[child position][0] = 0;
	    }

	    if (self->align[1] < 0) {
		[child position][1] = 0.5 * (self->minimum[1] -
					     self->allocated[1]);
	    } else if (self->align[1] > 0) {
		[child position][1] = 0.5 * (self->allocated[1] -
					     self->minimum[1]);
	    } else {
		[child position][1] = 0;
	    }
	}

	/* Get the projection matrix and viewport, multiply
	   the node's transform by the current modelview matrix
	   and project to screen space. */
	
	glGetDoublev (GL_PROJECTION_MATRIX, P);
	glGetIntegerv (GL_VIEWPORT, v);

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([self matrix]);
	glGetDoublev (GL_MODELVIEW_MATRIX, M);
	glPopMatrix ();

	gluProject (0, 0, 0, M, P, v, &w[0], &w[1], &w[2]);

	/* Set the translation and copy the orientation. */

	q[0] = (w[0] - 0.5 * v[2]) / v[3] + 0.5 * self->minimum[0] +
	    self->radius * cos (self->angle) + 0.01;
	q[1] = w[1] / v[3] - 0.5 +
	    self->radius * sin(self->angle);
	q[2] = -w[2];

	[super transformToTranslation: q andRotation: I];

	/* Draw the stem. */
	
	x = self->radius * cos (self->angle);
	y = self->radius * sin (self->angle);

	glMatrixMode (GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	
	glGetIntegerv (GL_VIEWPORT, v);
	glOrtho(-(double)v[2] / v[3] * 0.5,
		(double)v[2] / v[3] * 0.5,
		-0.5, 0.5,
		0, 1);

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	/* glTranslatef (q[0], q[1], q[2]); */
	glLoadMatrixd ([self matrix]);
    
	glUseProgramObjectARB(0);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_BLEND);

	glColor4dv (self->color);
	glLineWidth (self->thickness);
	glPointSize (3 * self->thickness);

	glBegin (GL_POINTS);
	glVertex2d (-0.5 * self->allocated[0], 0);
	glVertex2d (-0.5 * self->allocated[0] - x - 0.01, -y);
	glEnd();

	glBegin (GL_LINE_STRIP);
	glVertex2d (-0.5 * self->allocated[0], 0);
	glVertex2d (-0.5 * self->allocated[0] - 0.01, 0);
	glVertex2d (-0.5 * self->allocated[0] - x - 0.01, -y);
	glEnd();

	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_POINT_SMOOTH);
    
	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();
	
	[super traversePass: pass];
    
	glMatrixMode (GL_MODELVIEW);
	glPopMatrix();
    
	glMatrixMode (GL_PROJECTION);
	glPopMatrix();
    } else {
	[super traversePass: pass];
    }
}

-(void) get
{
    const char *k;
    int i;
    
    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "size")) {
	lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->requested[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "node")) {
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_gettable (_L, 1);
    } else if (!xstrcmp(k, "thickness")) {
	lua_pushnumber (_L, self->thickness);
    } else if (!xstrcmp(k, "radius")) {
	lua_pushnumber (_L, self->radius);
    } else if (!xstrcmp(k, "angle")) {
	lua_pushnumber (_L, self->angle * 180 / M_PI);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "size")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->requested[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "node")) {
	if (!lua_isnil (_L, 3)) {
	    self->node = *(id *)lua_touserdata (_L, 3);
	} else {
	    self->node = nil;
	}
	
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_settable (_L, 1);
    } else if (!xstrcmp(k, "thickness")) {
	self->thickness = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "radius")) {
	self->radius = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "angle")) {
	self->angle = lua_tonumber (_L, -1) * M_PI / 180;
   } else {
	[super set];
    }
}

@end
