-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "frames"
require "transforms"
require "switches"
require "dynamics"
require "physics"
require "morejoints"
require "moremath"
require "transitions"

director = {}

local function callhooks (hooks, ...)
   if type (hooks) == "table" then
      for _, hook in pairs(hooks) do
         if ... and type(...) == "table" then
            hook (unpack(...))
         else
            hook (...)
         end
      end
   elseif type (hooks) == "function" then
      if ... and type(...) == "table" then
         hooks(unpack(...))
      else
         hooks (...)
      end
   end
end

local function adjustorbit (self, radius, azimuth, elevation)
   local meta = getmetatable (self)

   self.torso.neck.preload = radius and
                             radius - meta.rest[1] or
			     self.torso.neck.preload

   self.preload = {elevation and
		   math.clamp (elevation - meta.rest[3], -math.pi, math.pi) or
		   self.preload[1],
		   0,
		   azimuth and
		   math.clamp (azimuth - meta.rest[2], -math.pi, math.pi) or
		   self.preload[3]}
end

local function adjustperson (self, azimuth, elevation)
   if not self.setup then
      if azimuth then
	 self.rest[1] = azimuth
      end

      if elevation then
	 self.rest[2] = elevation and math.clamp (elevation, 0, math.pi / 2)
      end

      -- Wait for the first full theta, phi specifcation
      -- and use it as the initial rig configuration
   
      if #self.rest == 2 then
	 self.head.orientation = transforms.relue (0,
						   math.deg(self.rest[2]),
						   math.deg(self.rest[1]))

	 -- Join the rig.
	 
	 self.neck.axes = {transforms.fromnode (self.head, {0, 1, 0}),
			   {1, 0, 0}, 
			   {0, 0, 1},}
	 self.neck.bodies = {self.head, nil}

	 self.state = {self.rest[1], self.rest[2]}
	 self.setup = true
      end
   else
      if azimuth then
	 self.state[1] = math.clamp (azimuth,
				     -math.pi + self.rest[1], 
				     math.pi + self.rest[1])
      end

      if elevation then
	 self.state[2] = math.clamp (elevation,
				     -math.pi / 2 + self.rest[2],
				     math.pi / 2 + self.rest[2])
      end

      self.neck.preload = {self.state[2] - self.rest[2],
			   0,
			   self.state[1] - self.rest[1]}
   end
end

function director.orbit (values)
   local self, meta, oldindex, oldnewindex

   self = springs.euler {
      axes = {{0, 1, 0}, {1, 0, 0}, {0, 0, 1}},

      stiffness = 3000,
      damping = 1000,

      torso = bodies.point {
	 position = {0, 0, 0},
	 mass = physics.spheremass (1, 0.1),

	 neck = springs.linear {
	    stiffness = 3000,
	    damping = 1000,

	    head = bodies.point {
	       position = {0, 0, 1},
	       mass = physics.spheremass (1, 0.1),
	       eye = frames.observer {}
	    },
	 },
      },

      unlink = function (self)
		  local meta = getmetatable (self)

		  meta.rest = {0, 0, 0}
	       end,

      link = function (self)
		local meta = getmetatable (self)

		meta.rest = {meta.state[1],
			     meta.state[2],
			     meta.state[3]}

		self.torso.position = self.parent.position or {0, 0, 0}
		self.torso.neck.head.position = 
		                math.add (self.torso.position,
					  {meta.rest[1] *
					   math.cos (meta.rest[2]) *
					   math.sin (meta.rest[3]),
					   meta.rest[1] *
					   math.sin (meta.rest[2]) *
					   math.sin (meta.rest[3]),
					   meta.rest[1] *
					   math.cos (meta.rest[3])})

		self.torso.neck.head.orientation =
		   transforms.relue (0,
				     math.deg(meta.rest[3]),
				     math.deg(meta.rest[2]))

		-- Configure the rig.
		
		self.anchor = self.parent.position or {0, 0, 0}
		self.axes = {transforms.fromnode (self.torso.neck.head,
						  {0, 1, 0}),
			     nil, 
			     self.parent.orientation and
			        math.column (self.parent.orientation, 3) or
			        {0, 0, 1}}

		self.torso.neck.axis =
		   transforms.fromnode (self.torso.neck.head,
					{0, 0, 1})
		
		self.torso.neck.preload = 0
		self.preload = {0, 0, 0}

		physics.reattach (self)
		physics.reattach (self.torso.neck)
	     end,
   }

   meta = getmetatable(self)
   meta.damping = 1000
   meta.stiffness = 3000
   meta.state = {1, 0, 0}
   meta.rest = {0, 0, 0}

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function(table, key)
		     if key == "radius" then
			return meta.state[1]
		     elseif key == "azimuth" then
			return meta.state[2]
		     elseif key == "elevation" then
			return meta.state[3]
		     elseif key == "stiffness" then
			return meta.stiffness
		     elseif key == "damping" then
			return meta.damping
		     elseif key == "speed" then
			-- return math.length (self.system.head.velocity)
		     else
			return oldindex(table, key)
		     end
		  end

   meta.__newindex = function(table, key, value)
			if key == "radius" then
			   meta.state[1] = math.clamp (value, 0, 1 / 0)

			   adjustorbit (table, meta.state[1], nil, nil)
			elseif key == "azimuth" then
			   meta.state[2] = math.clamp (value,
						       -math.pi + meta.rest[2],
						       math.pi + meta.rest[2])

			   adjustorbit (table, nil, meta.state[2], nil)
			elseif key == "elevation" then
			   meta.state[3] = math.clamp (value,
						       -math.pi + meta.rest[3],
						       math.pi + meta.rest[3])

			   adjustorbit (table, nil, nil, meta.state[3])
			elseif key == "stiffness" then
			   meta.stiffness = value

			   self.torso.neck.stiffness = value
			   oldnewindex(table, key, value)
			elseif key == "damping" then
			   meta.damping = value

			   self.torso.neck.damping = value
			   oldnewindex(table, key, value)
			else
			   oldnewindex(table, key, value)
			end
		     end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function director.firstperson (values)
   local self, meta, oldindex, oldnewindex

   self = bodies.system {
      setup = false,
      rest = {},
      state = {0, 0, 0},

      -- The bodies.

      head = bodies.point {
	 mass = physics.spheremass (1, 0.1),
	 eye = frames.observer {}
      },

      -- Joints.

      neck = springs.euler {
	 axes = {{0, 1, 0}, {1, 0, 0}, {0, 0, 1}},

	 stiffness = 3000,
	 damping = 1000,
      },
   }

   meta = getmetatable(self)
   meta.damping = 1000
   meta.stiffness = 3000

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function(table, key)
		     if key == "azimuth" then
			return table.state[1]
		     elseif key == "elevation" then
			return table.state[2]
		     elseif key == "stiffness" then
			return meta.stiffness
		     elseif key == "damping" then
			return meta.damping
		     else
			return oldindex(table, key)
		     end
		  end

   meta.__newindex = function(table, key, value)
			if key == "azimuth" then
			   adjustperson (table, value, nil)
			elseif key == "elevation" then
			   adjustperson (table, nil, value)
			elseif key == "stiffness" then
			   meta.stiffness = value

			   self.neck.stiffness = value
			elseif key == "damping" then
			   meta.damping = value

			   self.neck.damping = value
			else
			   oldnewindex(table, key, value)
			end
		     end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function director.script (values)
   local self, meta, oldindex, oldnewindex

   self = frames.timer {
      period = 0,

      tick = function (self, tick, delta, elapsed)
		if not meta.event then
		   local next = 1 / 0

		   for i, event in pairs (meta) do
		      if tonumber (i) and i > elapsed and i < next then
			 next = i
			 meta.event = event
		      end
		   end

		   self.period = next - elapsed
		else
		   callhooks (meta.event, self)

		   meta.event = nil
		   self.period = 0
		end
	     end
   }

   meta = getmetatable(self)

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function(table, key)
		     if tonumber(key) then
			return meta[key]
		     else
			return oldindex(table, key)
		     end
		  end

   meta.__newindex = function(table, key, value)
			if tonumber(key) then
			   meta[tonumber(key)] = value

			   meta.event = nil
			   self.period = 0
			else
			   oldnewindex(table, key, value)
			end
		     end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end
