/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>

#include "windflow.h"
#include "meteorology.h"

@implementation Windflow

-(Windflow *) init
{
    self = [super init];

    self->resolution = 1;
    self->scale = 1;
    self->size[0] = 1;
    self->size[1] = 1;
    self->size[2] = 1;
    
    return self;
}

-(void) get
{
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "resolution")) {
	lua_pushnumber (_L, self->resolution);
    } else if (!xstrcmp(k, "scale")) {
	lua_pushnumber (_L, self->scale);
    } else if (!xstrcmp(k, "size")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3 ; i += 1) {
            lua_pushnumber (_L, self->size[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "resolution")) {
	self->resolution = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "scale")) {
	self->scale = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "size")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->size[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else {
	[super set];
    }
}

-(void) stepBy: (double) h at: (double) t
{
    self->time = t + h;
}

-(void) traversePass: (int)pass
{
    double *R, *r;
    double i, j, k;
    
    if (pass == 2) {
	R = self->rotation;
	r = self->translation;
	
	glUseProgramObjectARB(0);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);
	glLineWidth (1.5);
	glDepthMask (GL_FALSE);

	glColor4f(0.1, 1.0, 0.1, 1.0);

	glBegin(GL_LINES);
	
	for (i = -size[0] / 2;
	     i < size[0] / 2;
	     i += 1.0 / self->resolution) {
	    for (j = -size[1] / 2;
		 j < size[1] / 2;
		 j += 1.0 / self->resolution) {
		for (k = -size[2] / 2;
		     k < size[2] / 2;
		     k += 1.0 / self->resolution) {
		    double x, y, z, tau[3];

		    x = r[0] + i * R[0] + j * R[1] + k * R[2];
		    y = r[1] + i * R[3] + j * R[4] + k * R[5];
		    z = r[2] + i * R[6] + j * R[7] + k * R[8];
		    
		    get_turbulence_at (x, y, z, self->time, tau);

		    glVertex3f (x, y, z);
		    glVertex3f (x + self->scale * tau[0],
				y + self->scale * tau[1],
				z + self->scale * tau[2]);
		}
	    }
	}
	
	glEnd();

	glDepthMask (GL_TRUE);
	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_DEPTH_TEST);
    }
    
    [super traversePass: pass];
}

@end

