/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>

#include "transform.h"

double zero[3] = {0, 0, 0}, *origin = zero;

static void callhooks (lua_State *L, void *key, int reference)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    luaX_call(L, 1, 0);
	} else if (lua_istable (L, -1)) {
	    int i, n;
	    
	    n = lua_objlen (L, -1);

	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (L, -2, i + 1);

		lua_pushlightuserdata (L, key);
		lua_gettable (L, -3);
		luaX_call (L, 1, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

static void recurse (id root)
{
    id child;
    
    if ([root isKindOf: [Transform class]]) {
	[root transform];
    } else {
	for (child = [root children] ; child ; child = [child sister]) {
	    recurse (child);
	}
    }
}

@implementation Transform

-(Transform *) init
{
    char *list[] = {"orientation", "position", "transform"};
    int i, j;
    
    [super init];
    [self add: sizeof(list) / sizeof(char *) Properties:list];

    self->transform = LUA_REFNIL;
    
    for (i = 0 ; i < 3 ; i += 1) {
	for (j = 0 ; j < 3 ; j += 1) {
	    self->orientation[i * 3 + j] = i == j ? 1 : 0;
	    self->rotation[i * 3 + j] = i == j ? 1 : 0;
	}

	self->position[i] = 0;
	self->translation[i] = 0;
    }

    return self;
}

-(void) free
{
     luaL_unref (_L, LUA_REGISTRYINDEX, self->transform);

     [super free];
}

-(id) parentTransform
{
    id parent;

    for (parent = [self parent];
	 parent && ![parent isKindOf: [Transform class]];
	 parent = [parent parent]);

    return parent;
}

-(void) transformCustom
{
    id child, sister;
    
    /* Calculate the homogenous transform matrix. */
    
    self->matrix[0] = self->rotation[0];
    self->matrix[1] = self->rotation[3];
    self->matrix[2] = self->rotation[6];
    self->matrix[3] = 0;
    
    self->matrix[4] = self->rotation[1];
    self->matrix[5] = self->rotation[4];
    self->matrix[6] = self->rotation[7];
    self->matrix[7] = 0;
    
    self->matrix[8] = self->rotation[2];
    self->matrix[9] = self->rotation[5];
    self->matrix[10] = self->rotation[8];
    self->matrix[11] = 0;

    self->matrix[12] = self->translation[0];
    self->matrix[13] = self->translation[1];
    self->matrix[14] = self->translation[2];
    self->matrix[15] = 1;	    

    callhooks (_L, self, self->transform);

    /* Here we save the sister node beforehand because the child
       node might be unlinked by its hook function which will
       result in a prematurely ended traversal. */
    
    for (child = [self children] ; child ; child = sister) {
	sister = [child sister];
	recurse (child);
    }
}

-(void) transformToTranslation: (double *) r
                   andRotation: (double *) R
{
    int i;

    /* Just return the position and orientation. */
	
    for (i = 0 ; i < 3 ; i += 1) {
	self->translation[i] = r[i];
    }

    for (i = 0 ; i < 9 ; i += 1) {
	self->rotation[i] = R[i];
    }

    [self transformCustom];
}

-(void) transformAsRoot
{
    int i;

    /* Just return the position and orientation. */
	
    for (i = 0 ; i < 3 ; i += 1) {
	self->translation[i] = self->position[i] - origin[i];
    }

    for (i = 0 ; i < 9 ; i += 1) {
	self->rotation[i] = self->orientation[i];
    }

    [self transformCustom];
}

-(void) transformRelativeTo: (double *) p
{
    double *oldorigin;

    oldorigin = origin;
    origin = p;

    [self transformAsRoot];

    origin = oldorigin;
}

-(void) transform
{
    id parent;
    double *r, *R, *p, *W;
    int i, j;

    parent = [self parentTransform];
    
    if (parent) {
	p = [self position];
	W = [self orientation];
    
	r = [parent translation];
	R = [parent rotation];

	for (i = 0 ; i < 3 ; i += 1) {
	    for (j = 0 ; j < 3 ; j += 1) {
		self->rotation[i * 3 + j] = R[i * 3] * W[j] +
		                            R[i * 3 + 1] * W[j + 3] +
		                            R[i * 3 + 2] * W[j + 6];
	    }

	    self->translation[i] = R[i * 3] * p[0] +
		                   R[i * 3 + 1] * p[1] +
		                   R[i * 3 + 2] * p[2] +
		                   r[i]; 
	}
    
	[self transformCustom];
    } else {
	[self transformAsRoot];
    }
}

-(double *) position
{
    return self->position;
}

-(double *) orientation
{
    return self->orientation;
}

-(double *) translation
{
    return self->translation;
}

-(double *) rotation
{
    return self->rotation;
}

-(double *) matrix
{
    return self->matrix;
}

-(void) get
{
    const char *k;
    int i;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "position")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->position[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "orientation")) {
        lua_newtable (_L);
        
        for(i = 0; i < 9; i += 1) {
            lua_pushnumber (_L, self->orientation[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "transform")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->transform);
    } else {
	[super get];
    }
}
 
-(void) set
{    
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "position")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->position[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "orientation")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 9 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->orientation[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "transform")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->transform);
        self->transform = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	[super set];
    }
}
 
@end
