/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>

#include "texture.h" 
#include "clamped.h" 
#include "periodic.h" 
#include "mirrored.h" 

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int generic_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object free];

    return 0;
}

static int texture_len(lua_State *L)
{
    id T;

    T = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [T width]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [T height]);
    lua_rawseti(L, -2, 2);

    lua_pushnumber(L, [T components]);
    lua_rawseti(L, -2, 3);

    return 1;
}

static int texture_tostring(lua_State *L)
{
    id T;

    T = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [T name]);
    lua_pushstring(L, " texture map");
    lua_concat(L, 2);
    
    return 1;
}

int constructtexture(lua_State *L)
{
    Class class;
    id T;
    unsigned char *pixels;
    int i, size[3], levels[2];
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "levels");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            levels[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    } else {
	levels[0] = 0;
	levels[1] = 1000;
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 3 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "pixels");
    lua_gettable(L, 1);

    if (lua_istable (L, -1)) {
	if (!lua_getmetatable (L, -1)) {
	    lua_newtable(L);
	}
    
	lua_getfield (L, -1, "unsigned char");

	if (lua_isstring (L, -1)) {
	    pixels = (unsigned char *)malloc(lua_strlen (L, -1));
	    memcpy(pixels, lua_tostring (L, -1), lua_strlen (L, -1));

	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 2);

	    pixels = (unsigned char *)malloc(size[0] * size[1] * size[2] *
					     sizeof(unsigned char));

	    for(i = 0 ; i < size[0] * size[1] * size[2] ; i += 1) {
		lua_pushinteger (L, i + 1);
		lua_gettable (L, -2);

		pixels[i] = (unsigned char)(UCHAR_MAX * lua_tonumber(L, -1));
	    
		lua_pop(L, 1);
	    }
	}
    } else {
	pixels = NULL;
    }
    
    lua_pop(L, 1);

    /* Create and initialize the texture userdata. */

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    if (pixels) {
	T = [[class alloc] initWithPixels: pixels
	                   ofSize: size
	                   andLevelsBetween: levels[0] and: levels[1]];
    } else {
	T = [[class alloc] init];
    }
    
    *(id *)lua_newuserdata(L, sizeof(id)) = T;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, texture_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, texture_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, generic_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    free (pixels);
    
    return 1;
}

int luaopen_textures (lua_State *L)
{
    int i;

    Class textures[] = {
	[Clamped class], [Periodic class], [Mirrored class]
    };
    
    /* Create the texture constructors. */
    
    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(textures) / sizeof(textures[0]) ; i += 1) {
	lua_pushlightuserdata (L, textures[i]);
	lua_pushcclosure (L, constructtexture, 1);
	lua_setfield(L, -2, decapitalize(strdupa([textures[i] name])));
    }

    lua_setglobal (L, lua_tostring (L, 1));

    return 0;
}
