/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>

#include "profiler.h"

@implementation Profiler

-(Profiler *) init
{
    char *list[] = {"finishtime", "preparetime", "steptime", "traversetime"};
    
    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->steptime = 0;    
    self->preparetime = 0;    
    self->traversetime[0] = 0;    
    self->traversetime[1] = 0;    
    self->finishtime = 0;    
    
    return self;
}

-(void) begin
{
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->then);   
    [super begin];
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->now);

    self->steptime = 0;
    self->begintime = 
	self->now.tv_sec - self->then.tv_sec +
	(self->now.tv_nsec - self->then.tv_nsec) / 1e9;
}

-(void) stepBy: (double) h at: (double) t
{
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->then);   
    [super stepBy: h at: t];
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->now);
    
    self->steptime += 
	self->now.tv_sec - self->then.tv_sec +
	(self->now.tv_nsec - self->then.tv_nsec) / 1e9;
}

-(void) prepare
{
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->then);   
    [super prepare];
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->now);

    self->preparetime = 
	self->now.tv_sec - self->then.tv_sec +
	(self->now.tv_nsec - self->then.tv_nsec) / 1e9;
}

-(void) traversePass: (int) pass
{
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->then);   
    [super traversePass: pass];
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->now);

    self->traversetime[pass] = 
	self->now.tv_sec - self->then.tv_sec +
	(self->now.tv_nsec - self->then.tv_nsec) / 1e9;
}

-(void) finish
{
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->then);   
    [super finish];
    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &self->now);

    self->finishtime = 
	self->now.tv_sec - self->then.tv_sec +
	(self->now.tv_nsec - self->then.tv_nsec) / 1e9;
}

-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "begintime")) {
        lua_pushnumber (_L, self->begintime);
    } else if (!xstrcmp(k, "steptime")) {
        lua_pushnumber (_L, self->steptime);
    } else if (!xstrcmp(k, "preparetime")) {
        lua_pushnumber (_L, self->preparetime);
    } else if (!xstrcmp(k, "traversetime")) {
	int i;
	
	lua_newtable (_L);

	for (i = 0 ; i < 2 ; i += 1) {
	    lua_pushnumber (_L, self->traversetime[i]);
	    lua_rawseti (_L, -2, i);
	}
    } else if (!xstrcmp(k, "finishtime")) {
        lua_pushnumber (_L, self->finishtime);
    } else  {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "begintime")) {
    } else if (!xstrcmp(k, "steptime")) {
    } else if (!xstrcmp(k, "preparetime")) {
    } else if (!xstrcmp(k, "traversetime")) {
    } else if (!xstrcmp(k, "finishtime")) {
    } else {
	[super set];	
    }
}

@end
