/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "body.h"
#include "meteorology.h"
#include "piston.h"

static double lookup (double x, double *values, int length)
{
    double *a, *b;
    int k;
    
    if (length > 0) {
	for(k = 0, a = values, b = a + 2;
	    k < length - 4 && b[0] <= x ;
	    k += 2, a = b, b += 2);
	
	return a[1] + (b[1] - a[1]) / (b[0] - a[0]) * (x - a[0]);
    } else {
	return 0;
    }
}

@implementation Piston

-(Piston *) init
{
    char *list[] = {
	"brakepower", "diameter", "idle", "inertia", "output", "power",
	"ratio", "speed", "throttle", "thrust"
    };

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->speed = 800 * M_PI / 30;
    self->idle = 500 * M_PI / 30;
    self->diameter = 1;
    self->ratio = 1;
    self->throttle = 1;
    self->bypass = 0;
    self->inertia = 1;

    self->coefficients[0].length = 0;
    self->coefficients[0].values = NULL;

    self->coefficients[1].length = 0;
    self->coefficients[1].values = NULL;

    self->brakepower.length = 0;
    self->brakepower.values = NULL;
    
    return self;
}

-(void) stepBy: (double) h at: (double) t
{
    double *r, *R;
    const dReal *r_a, *v_a, *R_a;
    double J, C_T, C_P, P_p, P_e;
    double omega, D, V, I, Nu, rho, delta;

    /* Calculate the propeller's airspeed. */
    
    R_a = dBodyGetRotation([[self parent] body]);
    r_a = dBodyGetPosition([[self parent] body]);
    v_a = dBodyGetLinearVel([[self parent] body]);
    V = R_a[0] * v_a[0] + R_a[4] * v_a[1] + R_a[8] * v_a[2];

    I = self->inertia;
    D = self->diameter;
    omega = self->speed;
    Nu = self->ratio * self->speed / (2 * M_PI);

    self->bypass -= h * (omega - self->idle);
    self->bypass = self->bypass < 0 ? 0 : self->bypass;
    self->bypass = self->bypass > 1 ? 1 : self->bypass;
    
    rho = get_density_at (r_a[2]);\
    delta = (self->throttle + self->bypass) *
	    get_pressure_at (r_a[2]) / get_pressure_at (0);

    /* Lookup the power available at the propeller shaft. */

    P_e = delta * lookup (omega,
			  self->brakepower.values,
			  self->brakepower.length);

    /* Calculate power consumed and thrust
       generated by the propeller. */
    
    J = V / (D * Nu);

    C_T = lookup(J,
		 self->coefficients[0].values,
		 self->coefficients[0].length);

    C_P = lookup(J,
		 self->coefficients[1].values,
		 self->coefficients[1].length);

    P_p = C_P * Nu * Nu * Nu * D * D * D * D * D * rho;

    self->speed = omega + (P_e - P_p) / omega / I * h;
    self->output = C_T * Nu * Nu * D * D * D * D * rho;

    /* Finally apply the calculated thrust
       along the engine's z axis. */
    
    r = [self position];
    R = [self orientation];
    
    dBodyAddRelForceAtRelPos ([[self parent] body],
			      self->output * R[2],
			      self->output * R[5],
			      self->output * R[8],
			      r[0], r[1], r[2]);
    
    [super stepBy: h at: t];
}

-(void) get
{
    const char *k;
    int i;
    
    k = lua_tostring(_L, -1);

    if (!xstrcmp(k, "output")) {
	lua_pushnumber(_L, self->output);
    } else if (!xstrcmp(k, "idle")) {
	lua_pushnumber(_L, self->idle);
    } else if (!xstrcmp(k, "speed")) {
	lua_pushnumber(_L, self->speed);
    } else if (!xstrcmp(k, "throttle")) {
	lua_pushnumber(_L, self->throttle);
    } else if (!xstrcmp(k, "ratio")) {
	lua_pushnumber(_L, self->ratio);
    } else if (!xstrcmp(k, "inertia")) {
	lua_pushnumber(_L, self->inertia);
    } else if (!xstrcmp(k, "diameter")) {
	lua_pushnumber(_L, self->diameter);
    } else if (!xstrcmp(k, "brakepower")) {
	lua_newtable (_L);
	
	for (i = 0 ; i < self->brakepower.length ; i += 1) {
	    lua_pushnumber (_L, self->brakepower.values[i]);
	    lua_rawseti (_L, -2, i + 1);
	}
    } else if (!xstrcmp(k, "thrust")) {
	lua_newtable (_L);
	
	for (i = 0 ; i < self->coefficients[0].length ; i += 1) {
	    lua_pushnumber (_L, self->coefficients[0].values[i]);
	    lua_rawseti (_L, -2, i + 1);
	}
    } else if (!xstrcmp(k, "power")) {
	lua_newtable (_L);
	
	for (i = 0 ; i < self->coefficients[1].length ; i += 1) {
	    lua_pushnumber (_L, self->coefficients[1].values[i]);
	    lua_rawseti (_L, -2, i + 1);
	}
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i, n;
    
    k = lua_tostring(_L, -2);

    if (!xstrcmp(k, "throttle")) {
	self->throttle = lua_tonumber(_L, 3);

	if (self->throttle < 0) {
	    self->throttle = 0;
	}

	if (self->throttle > 1) {
	    self->throttle = 1;
	}
    } else if (!xstrcmp(k, "idle")) {
	self->idle = lua_tonumber(_L, 3);
    } else if (!xstrcmp(k, "ratio")) {
	self->ratio = lua_tonumber(_L, 3);
    } else if (!xstrcmp(k, "inertia")) {
	self->inertia = lua_tonumber(_L, 3);
    } else if (!xstrcmp(k, "diameter")) {
	self->diameter = lua_tonumber(_L, 3);
    } else if (!xstrcmp(k, "brakepower")) {
	if (lua_istable (_L, 3)) {
	    n = luaX_objlen(_L, 3);

	    self->brakepower.length = n;
	    self->brakepower.values = realloc(self->brakepower.values,
					      n * sizeof(double));

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti(_L, 3, i + 1);
		self->brakepower.values[i] = lua_tonumber(_L, -1);

		lua_pop(_L, 1);
	    }
	} else {
	    self->brakepower.length = 0;
	}
    } else if (!xstrcmp(k, "thrust")) {
	if (lua_istable (_L, 3)) {
	    n = luaX_objlen(_L, 3);

	    self->coefficients[0].length = n;
	    self->coefficients[0].values = realloc(self->coefficients[0].values,
						   n * sizeof(double));

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti(_L, 3, i + 1);
		self->coefficients[0].values[i] = lua_tonumber(_L, -1);

		lua_pop(_L, 1);
	    }
	} else {
	    self->coefficients[0].length = 0;
	}
    } else if (!xstrcmp(k, "power")) {
	if (lua_istable (_L, 3)) {
	    n = luaX_objlen(_L, 3);

	    self->coefficients[1].length = n;
	    self->coefficients[1].values = realloc(self->coefficients[1].values,
						   n * sizeof(double));

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti(_L, 3, i + 1);
		self->coefficients[1].values[i] = lua_tonumber(_L, -1);

		lua_pop(_L, 1);
	    }
	} else {
	    self->coefficients[1].length = 0;
	}
    } else {
	[super set];
    }
}

@end
