/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <ctype.h>
#include <lua.h>
#include <lauxlib.h>

#include "windflow.h"
#include "vortex.h"

static int size[4] = {0, 0, 0, 0};
static float *samples;
static double scale[2] = {1, 1};

static double *values[3];
static int lengths[3];

static double lookup (double x, double *values, int length)
{
    double *a, *b;
    int k;
    
    if (length > 0) {
	for(k = 0, a = values, b = a + 2;
	    k < 2 * length - 4 && b[0] <= x ;
	    k += 2, a = b, b += 2);
	
	return a[1] + (b[1] - a[1]) / (b[0] - a[0]) * (x - a[0]);
    } else {
	return 0;
    }
}

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int node_len(lua_State *L)
{
    int n;
    
    lua_getmetatable (L, 1);
    n = luaX_objlen (L, -1);
    lua_pushnumber (L, n);
   
    return 1;
}

static int node_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [object name]);
   
    return 1;
}

static int node_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object get];

    return 1;
}

static int node_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object set];

    return 0;
}

static int node_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object free];

    return 0;
}

static int constructnode (lua_State *L)
{
    Class class;
    id object;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    /* Create the userdata... */

    object = [[class alloc] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, (lua_CFunction)node_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)node_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    /* Add a weak reference. */
    
    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    return 1;
}

static int turbulence_index (lua_State *L)
{
    const char *k;
    
    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "size")) {
	lua_newtable (L);

	/* Implement this at some point. */
    } else if (!xstrcmp(k, "scale")) {
	lua_newtable (L);
	lua_pushnumber (L, scale[0]);
	lua_rawseti (L, -2, 1);
	lua_pushnumber (L, scale[1]);
	lua_rawseti (L, -2, 2);
    } else if (!xstrcmp(k, "samples")) {
	lua_newtable (L);

	/* Implement this at some point. */
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int turbulence_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "size")) {
	if(lua_istable(L, 3)) {
	    for(i = 0 ; i < 4 ; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		size[i] = lua_tonumber(L, -1);
                
		lua_pop(L, 1);
	    }
	} else {
	    size[0] = 0;
	    size[1] = 0;
	    size[2] = 0;
	}
    } else if (!xstrcmp(k, "scale")) {
	if(lua_istable(L, 3)) {
	    for(i = 0 ; i < 2 ; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		scale[i] = lua_tonumber(L, -1);
                
		lua_pop(L, 1);
	    }
	}
    } else if (!xstrcmp(k, "samples")) {
	if (lua_istable(L, 3)) {
	    int n;
	    
	    n = luaX_objlen(L, 3);
	    samples = (float *)realloc(samples, n * sizeof(float));

	    for(i = 0 ; i < n ; i += 1) {
		lua_pushinteger (_L, i + 1);
		lua_gettable (_L, -2);

		samples[i] = (float)lua_tonumber(L, -1);

		lua_pop(L, 1);
	    }
	}
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

double get_temperature_at (double h)
{
    return lookup(h, values[0], lengths[0]);
}

double get_pressure_at (double h)
{
    return lookup(h, values[1], lengths[1]);
}

double get_density_at (double h)
{
    return lookup(h, values[2], lengths[2]);
}

void get_turbulence_at (double x, double y, double z, double t, double *v)
{
    double x_0, x_1, y_0, y_1, z_0, z_1, t_0, t_1;
    double c, r[3], r_0, r_1, r_2, r_3;
    int i, j, delta_0, delta_1, delta_2, delta_3;

    if (size[0] > 0 && size[1] > 0 && size[2] > 0 && size[3] > 0) {
	r[0] = x; r[1] = y; r[2] = z;
	c = [Vortex evaluateAt: r];
	/* printf ("%f\n", c); */
	
	r_0 = fmod (fabs(x) / c / scale[0], size[0]);
	r_1 = fmod (fabs(y) / c / scale[0], size[1]);
	r_2 = fmod (fabs(z) / c / scale[0], size[2]);
	r_3 = fmod (t / c / scale[1], size[3]);

	x_1 = modf(r_0, &x_0);
	y_1 = modf(r_1, &y_0);
	z_1 = modf(r_2, &z_0);
	t_1 = modf(r_3, &t_0);

	/* Calculate initial vertex location. */
	
	i = t_0 * size[0] * size[1] * size[2] +
	    z_0 * size[0] * size[1] +
	    y_0 * size[0] +
	    x_0;

	/* As well as the offsets to its neighbor vertices
	   (taking the periodicity into account). */
	
	delta_0 = (int)x_0 < size[0] - 1 ? 1 : -(size[0] - 1);
	delta_1 = (int)y_0 < size[1] - 1 ? size[0] : -(size[1] - 1) * size[0];
	delta_2 = (int)z_0 < size[2] - 1 ?
	          size[0] * size[1] :
	          -(size[2] - 1) * size[1] * size[0];
	delta_3 = (int)t_0 < size[3] - 1 ?
	          size[0] * size[1] * size[2] :
	          -(size[3] - 1) * size[2] * size[1] * size[0];

	/* Interpolate the turbulence lattice at (r_0, r_1, r_2, r_3). */
		
	for (j = 0 ; j < 3 ; j += 1) {
	    double v_0000, v_0001, v_0010, v_0100;
	    double v_0011, v_0101, v_0110, v_0111;
	    double v_1000, v_1001, v_1010, v_1100;
	    double v_1011, v_1101, v_1110, v_1111;

	    v_0000 = samples[(i) * 3 + j];
	    v_1000 = samples[(i + delta_0) * 3 + j];
	    v_0100 = samples[(i + delta_1) * 3 + j];
	    v_0010 = samples[(i + delta_2) * 3 + j];
	    v_1100 = samples[(i + delta_1 + delta_0) * 3 + j];
	    v_1010 = samples[(i + delta_2 + delta_0) * 3 + j];
	    v_0110 = samples[(i + delta_2 + delta_1) * 3 + j];
	    v_1110 = samples[(i + delta_2 + delta_1 + delta_0) * 3 + j];

	    v_0001 = samples[(i + delta_3) * 3 + j];
	    v_1001 = samples[(i + delta_3 + delta_0) * 3 + j];
	    v_0101 = samples[(i + delta_3 + delta_1) * 3 + j];
	    v_0011 = samples[(i + delta_3 + delta_2) * 3 + j];
	    v_1101 = samples[(i + delta_3 + delta_1 + delta_0) * 3 + j];
	    v_1011 = samples[(i + delta_3 + delta_2 + delta_0) * 3 + j];
	    v_0111 = samples[(i + delta_3 + delta_2 + delta_1) * 3 + j];
	    v_1111 = samples[(i + delta_3 + delta_2 + delta_1 + delta_0) * 3 + j];

	    v[j] = v_0000 * (1 - x_1) * (1 - y_1) * (1 - z_1) * (1 - t_1) +
		   v_1000 * x_1 * (1 - y_1) * (1 - z_1) * (1 - t_1) +
		   v_0100 * (1 - x_1) * y_1 * (1 - z_1) * (1 - t_1) +
		   v_0010 * (1 - x_1) * (1 - y_1) * z_1 * (1 - t_1) +
		   v_1010 * x_1 * (1 - y_1) * z_1 * (1 - t_1) +
		   v_0110 * (1 - x_1) * y_1 * z_1 * (1 - t_1) +
		   v_1100 * x_1 * y_1 * (1 - z_1) * (1 - t_1) +
		   v_1110 * x_1 * y_1 * z_1 * (1 - t_1) +
		   v_0001 * (1 - x_1) * (1 - y_1) * (1 - z_1) * t_1 +
		   v_1001 * x_1 * (1 - y_1) * (1 - z_1) * t_1 +
		   v_0101 * (1 - x_1) * y_1 * (1 - z_1) * t_1 +
		   v_0011 * (1 - x_1) * (1 - y_1) * z_1 * t_1 +
		   v_1011 * x_1 * (1 - y_1) * z_1 * t_1 +
		   v_0111 * (1 - x_1) * y_1 * z_1 * t_1 +
		   v_1101 * x_1 * y_1 * (1 - z_1) * t_1 +
		   v_1111 * x_1 * y_1 * z_1 * t_1;

	    v[j] *= c;
	}
    } else {
	v[0] = 0;
	v[1] = 0;
	v[2] = 0;
    }
}
	 
static int atmosphere_index (lua_State *L)
{
    const char *k;
    int i;
    
    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "temperature")) {
	lua_newtable (L);
	
	for (i = 0 ; i < lengths[0] ; i += 1) {
	    lua_pushnumber (L, values[0][2 * i]);
	    lua_pushnumber (L, values[0][2 * i + 1]);
	    lua_rawset (L, -3);
	}
    } else if (!xstrcmp(k, "pressure")) {
	lua_newtable (L);
	
	for (i = 0 ; i < lengths[1] ; i += 1) {
	    lua_pushnumber (L, values[1][2 * i]);
	    lua_pushnumber (L, values[1][2 * i + 1]);
	    lua_rawset (L, -3);
	}
    } else if (!xstrcmp(k, "density")) {
	lua_newtable (L);
	
	for (i = 0 ; i < lengths[2] ; i += 1) {
	    lua_pushnumber (L, values[2][2 * i]);
	    lua_pushnumber (L, values[2][2 * i + 1]);
	    lua_rawset (L, -3);
	}
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int compare (const void *a, const void *b)
{
    if (((double *)a)[1] == ((double *)b)[1]) {
	return 0;
    } else if (((double *)a)[1] < ((double *)b)[1]) {
	return -1;
    } else {
	return 1;
    }
}

static int atmosphere_newindex (lua_State *L)
{
    const char *k;
    int n;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "temperature")) {
	if (lua_istable (L, 3)) {
	    /* Get count of samples. */

	    lua_pushnil (L);
	    for (n = 0;
		 lua_next(L, 3) != 0;
		 n += lua_type(L, -1) == LUA_TNUMBER) {
		lua_pop(L, 1);
	    }

	    lengths[0] = n;
	    values[0] = (double *)realloc(values[0], 2 * n * sizeof(double));

	    /* Now get the samples. */
	    
	    lua_pushnil (L);
	    for (n = 0;
		 lua_next(L, 3) != 0;
		 n += lua_type(L, -1) == LUA_TNUMBER) {
		values[0][2 * n] = lua_tonumber(L, -2);
		values[0][2 * n + 1] = lua_tonumber(L, -1);
		lua_pop(L, 1);
	    }

	    /* And sort. */
	    
	    qsort (values[0], lengths[0], 2 * sizeof(double), compare);
	} else {
	    lengths[0] = 0;
	}
    } else if (!xstrcmp(k, "pressure")) {
	if (lua_istable (L, 3)) {
	    /* Get count of samples. */

	    lua_pushnil (L);
	    for (n = 0;
		 lua_next(L, 3) != 0;
		 n += lua_type(L, -1) == LUA_TNUMBER) {
		lua_pop(L, 1);
	    }

	    lengths[1] = n;
	    values[1] = (double *)realloc(values[1], 2 * n * sizeof(double));

	    /* Now get the samples. */
	    
	    lua_pushnil (L);
	    for (n = 0;
		 lua_next(L, 3) != 0;
		 n += lua_type(L, -1) == LUA_TNUMBER) {
		values[1][2 * n] = lua_tonumber(L, -2);
		values[1][2 * n + 1] = lua_tonumber(L, -1);
		lua_pop(L, 1);
	    }

	    /* And sort. */
	    
	    qsort (values[1], lengths[1], 2 * sizeof(double), compare);
	} else {
	    lengths[1] = 0;
	}
    } else if (!xstrcmp(k, "density")) {
	if (lua_istable (L, 3)) {
	    /* Get count of samples. */

	    lua_pushnil (L);
	    for (n = 0;
		 lua_next(L, 3) != 0;
		 n += lua_type(L, -1) == LUA_TNUMBER) {
		lua_pop(L, 1);
	    }

	    lengths[2] = n;
	    values[2] = (double *)realloc(values[2], 2 * n * sizeof(double));

	    /* Now get the samples. */
	    
	    lua_pushnil (L);
	    for (n = 0;
		 lua_next(L, 3) != 0;
		 n += lua_type(L, -1) == LUA_TNUMBER) {
		values[2][2 * n] = lua_tonumber(L, -2);
		values[2][2 * n + 1] = lua_tonumber(L, -1);
		lua_pop(L, 1);
	    }

	    /* And sort. */
	    
	    qsort (values[2], lengths[2], 2 * sizeof(double), compare);
	} else {
	    lengths[2] = 0;
	}
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}
   
int luaopen_meteorology (lua_State *L)
{
    int i;
    
    Class nodes[] = {
	[Windflow class], [Vortex class]
    };	

    /* Create the turbulence table. */
    
    lua_newtable (L);

    {
	char *list[] = {"samples", "scale", "size"};

	lua_newtable (L);
	lua_pushinteger (_L, sizeof (list) / sizeof (char *));
	lua_setfield (_L, -2, "__bloat");
	lua_pushlightuserdata (_L, list);
	lua_setfield (_L, -2, "__properties");
	lua_pushcfunction (L, turbulence_index);
	lua_setfield (L, -2, "__index");
	lua_pushcfunction (L, turbulence_newindex);
	lua_setfield (L, -2, "__newindex");
	lua_setmetatable (L, -2);
    }
 
    lua_setglobal (L, "turbulence");

    /* Create the atmosphere table. */
    
    lua_newtable (L);

    {
	char *list[] = {
	    "density", "pressure", "temperature"
	};
	
	lua_newtable (L);
	lua_pushinteger (_L, sizeof (list) / sizeof (char *));
	lua_setfield (_L, -2, "__bloat");
	lua_pushlightuserdata (_L, list);
	lua_setfield (_L, -2, "__properties");
	lua_pushcfunction (L, atmosphere_index);
	lua_setfield (L, -2, "__index");
	lua_pushcfunction (L, atmosphere_newindex);
	lua_setfield (L, -2, "__newindex");
	lua_setmetatable (L, -2);
    }
    
    lua_setglobal (L, "atmosphere");

    /* And export all nodes. */
    
    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(nodes) / sizeof(nodes[0]) ; i += 1) {
	lua_pushlightuserdata (L, nodes[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([nodes[i] name])));
    }

    lua_setglobal (L, lua_tostring (L, 1));
    
    return 0;
}
