/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <math.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <assert.h>

#include "elevation.h"
#include "atmosphere.h"
#include "land.h"

static const GLchar *vertexSource = 
"varying vec3 vertex;   	     	 			  \n"
" 								  \n"
"void main() 			   		     		  \n"
"{		 		   		     		  \n"
"    vertex = vec3(gl_ModelViewMatrix * gl_Vertex);		  \n"
"              							  \n"
"    gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex;	  \n"
"    gl_TexCoord[0].s = dot(gl_ObjectPlaneS[0], gl_Vertex);	  \n"
"    gl_TexCoord[0].t = dot(gl_ObjectPlaneT[0], gl_Vertex);	  \n"
"}		 		   		     		  \n";

static const GLchar *headerTemplate =
"const int N = %d;\n";

static const GLchar *fragmentSource =
"uniform sampler2D sampler, detailSampler[N];				\n"
"uniform float turbidity, factor, beta_p, power;			\n"
"uniform vec3 sunDirection, sunColor, beta_r;	  		        \n"
"				   		     			\n"
"varying vec3 vertex;   	     	     				\n"
"				   		     			\n"
"void main()                            	     			\n"
"{	                                	     			\n"
"    vec3 texel, tau, pigments[N], detail, hsv;	                       	\n"
"    float cosine, phase_r, phase_p, rho, H, M, m, C, distances[N];     \n"
"    int i;             						\n"
"	   		     						\n"
"    cosine = dot(normalize(vertex), sunDirection);  			\n"
"    texel = vec3(texture2D(sampler, gl_TexCoord[0].st));      		\n"
"        								\n"
"    M = max(max (texel.r, texel.g), texel.b); 				\n"
"    m = min(min (texel.r, texel.g), texel.b); 				\n"
"    C = M - m; 							\n"
"                                                                       \n"
"    if (C > 0.0) { 					  		\n"
"        if (texel.r == M) { 				  		\n"
"            H = mod((texel.g - texel.b) / C, 6.0) / 6.0; 		\n"
"        } else if (texel.g == M) { 			  		\n"
"            H = ((texel.b - texel.r) / C + 2.0) / 6.0;   		\n"
"        } else {					  		\n"
"            H = ((texel.r - texel.g) / C + 4.0) / 6.0;   		\n"
"        } 						  		\n"
"    } else {								\n"
"        H = 0.0;							\n"
"    }                                                          	\n"
"                                      					\n"
"    hsv = vec3(H, C / M, M);      					\n"
"			   		     				\n"
"    for (i = 0, C = 0.0 ; i < N ; i += 1) {   				\n"
"        vec3 v;							\n"
"        v = hsv - gl_TextureMatrix[i + 1][0].xyz;   			\n"
"        v.x = min(v.x, 1.0 - v.x);           				\n"
"        v *= gl_TextureMatrix[i + 1][1].xyz;        			\n"
"        distances[i] = 1.0 / pow(dot(v, v), power); 			\n"
"        C += distances[i]; 						\n"
"    }	     								\n"
"                                                                       \n"
"    for (i = 0, detail = vec3(0.0) ; i < N ; i += 1) {  		\n"
"             detail += distances[i] / C * 				\n"
"                       texture2D(detailSampler[i], 			\n"
"                                 gl_TextureMatrix[i + 1][2].st *	\n"
"                                 gl_TexCoord[0].st).rgb;		\n"
"    }                                                                  \n"
"                                                                       \n"
"    phase_r = 0.059683 * (1.0 + cosine * cosine);		        \n"
"    tau = exp(beta_r * vertex.z);  				        \n"
"    rho = exp((0.6544 * turbidity - 0.6510) * beta_p * vertex.z);      \n"
"			   		     				\n"
"    texel = (texel.r + texel.g + texel.b) / 3.0 * detail;	  	\n"
"			   		     				\n"
"    gl_FragColor = vec4(sunColor * mix(50.0 * vec3(phase_r),		\n"
"                                       factor * texel,			\n"
"                                       tau), rho);			\n"
"			   		     				\n"
"    //gl_FragColor = vec4(vec3(detail), 1.0);                          \n"
"    //gl_FragColor = vec4(vec3(C), 1.0);   				\n"
"}                                      	     			\n";

@implementation Land

-(Land *) init
{
    char *list[] = {"albedo", "separation", "palette"};

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->program = 0;
    self->palette = 0;   
    self->albedo = 1;
    self->separation = 1;
    self->detail = NULL;
    self->pigments = NULL;
				       
    return self;
}

-(void) free
{
    glDeleteTextures (self->palette, self->detail);

    free (self->detail);
    free (self->pigments);
        
    [super free];
}

-(void) get
{
    const char *k;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "albedo")) {
	lua_pushnumber (_L, self->albedo);
    } else if (!xstrcmp(k, "separation")) {
	lua_pushnumber (_L, self->separation);
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "albedo")) {
	self->albedo = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "separation")) {
	self->separation = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "palette")) {
	int i, j, n;

	/* Free the current resources. */
	
	if (self->detail) {
	    glDeleteTextures (self->palette, self->detail);
	    free (self->detail);
	}

	if (self->pigments) {
	    free (self->pigments);
	}
	
	if (lua_istable (_L, 3)) {
	    /* Allocate resources. */
	    
	    n = luaX_objlen (_L, 3);

	    self->detail = (GLuint *)calloc (n, sizeof (GLuint));
	    self->pigments = (double (*)[8])calloc (n, sizeof (double[8]));

	    glGenTextures(n, self->detail);

	    /* And load all pigments. */
	    
	    for (j = 0 ; j < n ; j += 1) {
		GLenum error;
		double pigment[8] = {0, 0, 0, 0, 0, 0, 0, 0};
		unsigned char *pixels = NULL;
		int length = 0;

		lua_rawgeti(_L, 3, j + 1);

		if (lua_istable (_L, -1)) {
		    /* The detail map pixels. */
		
		    lua_rawgeti (_L, -1, 1);

		    if (lua_istable (_L, -1)) {		
			length = luaX_objlen (_L, -1);

			if (!lua_getmetatable (_L, -1)) {
			    lua_newtable(_L);
			}

			lua_getfield (_L, -1, "unsigned char");

			if (lua_isstring (_L, -1)) {
			    pixels = (unsigned char *)malloc(lua_strlen (_L, -1));
			    memcpy(pixels, lua_tostring (_L, -1), lua_strlen (_L, -1));

			    lua_pop (_L, 2);
			} else {
			    lua_pop (_L, 2);

			    pixels = (unsigned char *)malloc(length * sizeof (unsigned char));

			    for(i = 0 ; i < length ; i += 1) {
				lua_pushinteger (_L, i + 1);
				lua_gettable (_L, -2);
				pixels[i] = (unsigned char)(UCHAR_MAX * lua_tonumber(_L, -1));
				lua_pop(_L, 1);
			    }
			}
		    }

		    lua_pop(_L, 1);
		
		    /* The resolution. */
		
		    lua_rawgeti (_L, -1, 2);

		    if (lua_istable (_L, -1)) {
			for (i = 0 ; i < 2 ; i += 1) {
			    lua_rawgeti (_L, -1, i + 1);
			    pigment[i] = lua_tonumber (_L, -1);
			    lua_pop(_L, 1);
			}
		    }

		    lua_pop(_L, 1);
		
		    /* The HSV color target. */
		
		    lua_rawgeti (_L, -1, 3);

		    if (lua_istable (_L, -1)) {
			for (i = 0 ; i < 3 ; i += 1) {
			    lua_rawgeti (_L, -1, i + 1);

			    if (lua_isnumber(_L, -1)) {
				pigment[i + 2] = lua_tonumber (_L, -1);
				pigment[i + 5] = 1;
			    } else {
				pigment[i + 2] = 0;
				pigment[i + 5] = 0;
			    }			    
			
			    lua_pop(_L, 1);
			}
		    }

		    lua_pop(_L, 1);

		    /* Go on and load the pgiment and
		       create the texture object. */

		    memcpy (self->pigments[j], pigment, sizeof (double[8]));
    		    length = (int)sqrt(length / 3);

		    glGetError();
		    glBindTexture(GL_TEXTURE_2D, self->detail[j]);

		    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
		    glPixelStorei(GL_UNPACK_ROW_LENGTH, length);

		    if (_WINDOW) {
			gluBuild2DMipmaps (GL_TEXTURE_2D,
					   GL_RGB,
					   length, length,
					   GL_RGB,
					   GL_UNSIGNED_BYTE,
					   pixels);
		    }

		    glTexParameteri(GL_TEXTURE_2D,
				    GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
				    GL_LINEAR_MIPMAP_LINEAR);

		    glTexParameteri(GL_TEXTURE_2D,
				    GL_TEXTURE_WRAP_S, GL_REPEAT);
		    glTexParameteri(GL_TEXTURE_2D,
				    GL_TEXTURE_WRAP_T, GL_REPEAT);

		    glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
		    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
		
		    error = glGetError();
		    if(error != GL_NO_ERROR) {
			printf ("Could not create detail texture (%s)\n",
				(char *)gluErrorString(error));
		    }
		}

		lua_pop(_L, 1);
	    }

	    [self rebuildWithPalette: n];
	}	
    } else {
	[super set];
    }
}

-(void) rebuildWithPalette: (int)c
{    
    GLchar header[strlen(headerTemplate) + 5];
    GLuint vertex, fragment;
    int i, p, m, n;

    printf ("\nRebuilding program for %s node with "
	    "a palette of %d pigments.\n", [self name], c);
    
    if (self->program != 0) {
	glDeleteProgram (self->program);
    }

    self->program = glCreateProgramObjectARB();
    self->palette = c;

    snprintf (header, sizeof(header), headerTemplate, self->palette);

    /* Compile and link the shader. */
	
    vertex = glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB);
    glShaderSourceARB(vertex, 1,
		      (const GLchar **)&vertexSource, NULL);
	
    glCompileShaderARB(vertex);
	
    fragment = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);
    glShaderSourceARB(fragment, 2,
		      (const GLchar *[2]){header, fragmentSource},
		      NULL);
	
    glCompileShaderARB(fragment);

    glAttachObjectARB(self->program, vertex);
    glAttachObjectARB(self->program, fragment);
    glLinkProgramARB (self->program);

    /* Get uniform locations. */

    self->sampler = glGetUniformLocationARB (self->program, "sampler");
    self->factor = glGetUniformLocationARB (self->program, "factor");
    self->power = glGetUniformLocationARB (self->program, "power");

    for (i = 0 ; i < self->palette ; i += 1) {
	char name[64];

	snprintf (name, 64, "detailSampler[%d]", i);
	self->detailSampler[i] = glGetUniformLocationARB (self->program,
							  name);
    }
    
    self->turbidity = glGetUniformLocationARB (self->program, "turbidity");
    self->rayleigh = glGetUniformLocationARB (self->program, "beta_r");
    self->mie = glGetUniformLocationARB (self->program, "beta_p");
    self->sunDirection = glGetUniformLocationARB (self->program,
						  "sunDirection");
    self->sunColor = glGetUniformLocationARB (self->program,
					      "sunColor");

    /* Now verify it. */
	
    if (_WINDOW) {
	glGetObjectParameterivARB (self->program,
				   GL_OBJECT_LINK_STATUS_ARB,
				   &p);

	if (p != GL_TRUE) {
	    printf ("\nThe program for %s nodes did not "
		    "build properly.\n", [self name]);
	}

	glGetObjectParameterivARB(vertex,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(vertex, n, &m, buffer);
	    printf ("Info log for the `%s' vertex source follows:\n\n%s\n",
		    [self name], buffer);
	}

	glGetObjectParameterivARB(fragment,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(fragment, n, &m, buffer);
	    printf ("Info log for the `%s' fragment source follows:\n\n%s\n",
		    [self name], buffer);
	}

	glGetObjectParameterivARB(self->program,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(self->program, n, &m, buffer);
	    printf ("Info log for the `%s' program follows:\n\n%s\n",
		    [self name], buffer);
	}
    }
}

-(void) traversePass: (int)pass
{
    id parent, child;
    int i;
    
    if (pass == 1) {
/*     glPolygonMode (GL_FRONT_AND_BACK, GL_LINE); */
	glUseProgramObjectARB(self->program);
	glUniform1iARB (self->sampler, 0);

	for (i = 0 ; i < self->palette ; i += 1) {
	    GLdouble M[16] = {self->pigments[i][2], 
			      self->pigments[i][3],
			      self->pigments[i][4],
			      0,
			      self->pigments[i][5],
			      self->pigments[i][6],
			      self->pigments[i][7],
			      0,
			      0.625 / self->pigments[i][0],
			      0.625 / self->pigments[i][1],
			      0, 0,
			      0, 0, 0, 0};

	    glActiveTexture(GL_TEXTURE1 + i);
	    glBindTexture(GL_TEXTURE_2D, self->detail[i]);
	    glMatrixMode (GL_TEXTURE);
	    glLoadMatrixd (M);
	    
	    glUniform1iARB (self->detailSampler[i], i + 1);
	}
	
	glUniform1fARB (self->factor, self->albedo);
	glUniform1fARB (self->power, self->separation);

	for (parent = [self parent] ; parent ; parent = [parent parent]) {
	    for (child = [parent children] ; child ; child = [child sister]) {
		if ([child isMemberOf: [Atmosphere class]]) {
		    glUniform3fvARB (self->sunDirection, 1, [child direction]);
		    glUniform3fvARB (self->sunColor, 1, [child sunlight]);
		    glUniform3fvARB (self->rayleigh, 1, [child rayleigh]);
		    glUniform1fARB (self->mie, [child mie]);
		    glUniform1fARB (self->turbidity, [child turbidity]);
		}
	    }
	}

	[super traversePass: pass];
    } else {
	[super traversePass: pass];
    }
}

@end
