/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <math.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <assert.h>

#include "atmosphere.h"
#include "earth.h"

@implementation Earth

-(Earth *) init
{
    char *list[] = {"albedo"};

    const int length = 256;
    GLfloat pixels[length];
    GLenum error;
    int i;
    
    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->albedo = 1;

    /* Create the fog texture object. */

    for (i = 0 ; i < length ; i += 1) {
	pixels[i] = exp (log(0.01) / length * i);
    }
    
    glGetError();
    glActiveTexture (GL_TEXTURE0);
    glGenTextures(1, &self->fog);
    glBindTexture(GL_TEXTURE_1D, self->fog);

    glTexImage1D (GL_TEXTURE_1D,
		  0, GL_ALPHA,
		  length, 0,
		  GL_ALPHA,
		  GL_FLOAT,
		  pixels);

    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		
    error = glGetError();
    if(error != GL_NO_ERROR) {
	printf ("Could not create the fog texture (%s)\n",
		(char *)gluErrorString(error));
    }
				       
    return self;
}

-(void) free
{
    glDeleteTextures (1, &self->fog);

    [super free];
}

-(void) get
{
    const char *k;

    k = lua_tostring(_L, 2);

    if (lua_isnumber (_L, 2)) {
    } else if (!xstrcmp(k, "albedo")) {
	lua_pushnumber (_L, self->albedo);
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "albedo")) {
	self->albedo = lua_tonumber (_L, -1);
    } else {
	[super set];
    }
}

-(void) traversePass: (int)pass
{
    id parent, child;

    if (pass == 1) {
	glUseProgramObjectARB(0);
	
	glActiveTexture(GL_TEXTURE0);
	glEnable (GL_TEXTURE_2D);
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);	
	glEnable (GL_TEXTURE_GEN_S);
	glEnable (GL_TEXTURE_GEN_T);
	glTexGeni (GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	glTexGeni (GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();
	glLoadIdentity ();

	glActiveTexture(GL_TEXTURE2);
	glBindTexture (GL_TEXTURE_1D, self->fog);
	glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glEnable (GL_TEXTURE_1D);

	for (parent = [self parent] ; parent ; parent = [parent parent]) {
	    for (child = [parent children] ; child ; child = [child sister]) {
		if ([child isMemberOf: [Atmosphere class]]) {
		    float plane[4] = {0, 0, 0, 0};

		    plane[2] = 1.5e-6 * [child turbidity] * [child turbidity] -
			       1.8e-5 * [child turbidity];
		    
		    glColor3f ([child sunlight][0] * self->albedo,
			       [child sunlight][1] * self->albedo,
			       [child sunlight][2] * self->albedo);
		    
		    glTexGeni (GL_S, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
		    glTexGenfv(GL_S, GL_EYE_PLANE, plane);
		    glEnable (GL_TEXTURE_GEN_S);
		}
	    }
	}
	
	glPopMatrix();

	[super traversePass: pass];
	
	glActiveTexture(GL_TEXTURE0);
	glDisable (GL_TEXTURE_GEN_S);
	glDisable (GL_TEXTURE_GEN_T);
	glDisable (GL_TEXTURE_2D);
	
	glActiveTexture(GL_TEXTURE2);
	glDisable (GL_TEXTURE_GEN_S);
	glDisable (GL_TEXTURE_1D);
    } else {
	[super traversePass: pass];
    }
}

@end
