/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include "culler.h"

static int flag;

@implementation Culler

-(Culler *) init
{
    /* Get the configuration. */
    
    lua_getglobal (_L, "options");

    lua_getfield (_L, -1, "drawbounds");
    self->debug = lua_toboolean (_L, -1);
    lua_pop (_L, 2);

    self = [super init];

    return self;
}

-(void) traversePass: (int)pass
{
    double T[16];
    int i;

    if (!flag) {
	/* Calculate T = P * M. */
    
	glGetDoublev(GL_MODELVIEW_MATRIX, T);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glMultMatrixd(T);
	glGetDoublev(GL_PROJECTION_MATRIX, T);
	glPopMatrix();

	/* Left. */

	self->planes[0][0] = T[3] + T[0];
	self->planes[0][1] = T[7] + T[4];
	self->planes[0][2] = T[11] + T[8];
	self->planes[0][3] = T[15] + T[12];

	/* Right. */

	self->planes[1][0] = T[3] - T[0];
	self->planes[1][1] = T[7] - T[4];
	self->planes[1][2] = T[11] - T[8];
	self->planes[1][3] = T[15] - T[12];

	/* Bottom. */

	self->planes[2][0] = T[3] + T[1];
	self->planes[2][1] = T[7] + T[5];
	self->planes[2][2] = T[11] + T[9];
	self->planes[2][3] = T[15] + T[13];

	/* Top. */

	self->planes[3][0] = T[3] - T[1];
	self->planes[3][1] = T[7] - T[5];
	self->planes[3][2] = T[11] - T[9];
	self->planes[3][3] = T[15] - T[13];

	/* Near. */

	self->planes[4][0] = T[3] + T[2];
	self->planes[4][1] = T[7] + T[6];
	self->planes[4][2] = T[11] + T[10];
	self->planes[4][3] = T[15] + T[14];

	/* Far. */

	self->planes[5][0] = T[3] - T[2];
	self->planes[5][1] = T[7] - T[6];
	self->planes[5][2] = T[11] - T[10];
	self->planes[5][3] = T[15] - T[14];

	/* Normalize the planes. */

	for(i = 0 ; i < 6 ; i += 1) {
	    float m;

	    m = sqrt(self->planes[i][0] * self->planes[i][0] +
		     self->planes[i][1] * self->planes[i][1] +
		     self->planes[i][2] * self->planes[i][2]);

	    self->planes[i][0] /= m;
	    self->planes[i][1] /= m;
	    self->planes[i][2] /= m;
	    self->planes[i][3] /= m;
	}
    }
    
    [super traversePass: pass];
}

-(void) finish
{
    flag = 0;
    
    [super finish];
}

@end
