/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <gtk/gtk.h>
#include <webkit/webkit.h>

static gboolean close_view(WebKitWebView *view, gpointer data)
{
    GtkWidget *parent;

    parent = gtk_widget_get_toplevel(GTK_WIDGET(view));
    gtk_widget_destroy (parent);
    
    return TRUE;
}

static void notify_title (WebKitWebView* view,
			  GParamSpec* pspec,
			  gpointer data)
{
    GtkWidget *parent;
    const gchar * title;

    parent = gtk_widget_get_toplevel(GTK_WIDGET(view));
    
    title = webkit_web_view_get_title(view);
    gtk_window_set_title (GTK_WINDOW(parent), title ? title : "Techne");
}

static void notify_progress (WebKitWebView* view,
			     GParamSpec* pspec,
			     gpointer data)
{
    GtkWidget *parent;
    GString *string;
    gchar *title;
    gint progress;

    parent = gtk_widget_get_toplevel(GTK_WIDGET(view));

    title = (gchar *)webkit_web_view_get_title(view);
    progress = webkit_web_view_get_progress (view) * 100;
    string = g_string_new (title ? title : "Techne");

    if (progress < 100) {
        g_string_append_printf (string, " (%d%%)", progress);
    }
    
    title = g_string_free (string, FALSE);
    gtk_window_set_title (GTK_WINDOW(parent), title);
    g_free (title);
}

static void configure_window (WebKitWebWindowFeatures* features,
			      GParamSpec* pspec,
			      WebKitWebView *view)
{
    GtkWidget *parent;
    int width, height, x, y, fullscreen;
    
    g_object_get (G_OBJECT (features),
    		  "width", &width,
    		  "height", &height,
    		  "x", &x,
    		  "y", &y,
    		  "fullscreen", &fullscreen,
    		  NULL);

    parent = gtk_widget_get_toplevel(GTK_WIDGET(view));

    gtk_window_resize (GTK_WINDOW (parent), width, height);
    gtk_window_move (GTK_WINDOW (parent), x, y);

    if (fullscreen) {
	gtk_window_fullscreen (GTK_WINDOW (parent));
    } else {
	gtk_window_unfullscreen (GTK_WINDOW (parent));
    }

    /* printf ("%d, %d, %d, %d\n", width, height, x, y); */
}

static void connect_features (WebKitWebView* view,
			      GParamSpec* pspec,
			      gpointer data)
{
    WebKitWebWindowFeatures *features;

    features = webkit_web_view_get_window_features (view);

    g_signal_connect (G_OBJECT(features), "notify::width",
    		      G_CALLBACK(configure_window), view);
    g_signal_connect (G_OBJECT(features), "notify::height",
    		      G_CALLBACK(configure_window), view);
    g_signal_connect (G_OBJECT(features), "notify::x",
    		      G_CALLBACK(configure_window), view);
    g_signal_connect (G_OBJECT(features), "notify::y",
    		      G_CALLBACK(configure_window), view);
    g_signal_connect (G_OBJECT(features), "notify::fullscreen",
    		      G_CALLBACK(configure_window), view);
}

static gboolean show_window (WebKitWebView *view, gpointer data)
{
    GtkWidget *parent;
    
    parent = gtk_widget_get_toplevel(GTK_WIDGET(view));

    gtk_widget_grab_focus (GTK_WIDGET (view));
    gtk_widget_show_all (parent);

    
    return FALSE;
}

static gboolean start_download (WebKitWebView *view,
				WebKitDownload *download,
				gpointer data)
{
    GtkWidget *dialog, *parent;
    gchar *filename, *uri;
    
    filename = (gchar *)webkit_download_get_suggested_filename(download);
    parent = gtk_widget_get_toplevel(GTK_WIDGET(view));
    dialog = gtk_file_chooser_dialog_new ("Save File",
					  GTK_WINDOW(parent),
					  GTK_FILE_CHOOSER_ACTION_SAVE,
					  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					  GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					  NULL);
    
    gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog), filename);
    gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog),
						    TRUE);
    
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
	filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	uri = g_strconcat ("file://", filename, NULL);

	webkit_download_set_destination_uri (download, uri);
	webkit_download_start (download);

	g_free (filename);
	g_free (uri);
    }
    
    gtk_widget_destroy (dialog);
    
    return TRUE;
}

gboolean decide_policy (WebKitWebView *webView,
			WebKitWebFrame *frame,
			WebKitNetworkRequest *request,
			gchar *mimetype,
			WebKitWebPolicyDecision *policy_decision,
			gpointer user_data)
{
    if (webkit_web_view_can_show_mime_type(webView, mimetype)) {
	webkit_web_policy_decision_use (policy_decision);
    } else {
	webkit_web_policy_decision_download (policy_decision);
    }
    
    return TRUE;
}

static WebKitWebView* create_view (WebKitWebView *parent,
				   WebKitWebFrame *frame,
				   gpointer data)
{
    WebKitWebView* view;
    GtkWidget *window;
    GtkWidget *scrolled;

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW(window), "Techne");
    gtk_widget_set_name (window, "GtkLauncher");
    gtk_window_set_default_size (GTK_WINDOW (window), 800, 600);
    /* gtk_window_set_deletable(GTK_WINDOW(window), FALSE); */

    scrolled = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);

    view = WEBKIT_WEB_VIEW (webkit_web_view_new ());

    connect_features (view, NULL, NULL);

    g_signal_connect (view, "notify::title",
    		      G_CALLBACK (notify_title), NULL);
    g_signal_connect (view, "notify::progress",
    		      G_CALLBACK (notify_progress), NULL);
    g_signal_connect (view, "notify::window-features",
    		      G_CALLBACK (connect_features), NULL);
    g_signal_connect (view, "create-web-view",
		      G_CALLBACK (create_view), NULL);
    g_signal_connect (view, "web-view-ready",
    		      G_CALLBACK (show_window), NULL);
    g_signal_connect (view, "close-web-view",
    		      G_CALLBACK (close_view), NULL);
    g_signal_connect (view, "download-requested",
    		      G_CALLBACK (start_download), NULL);
    g_signal_connect (view, "mime-type-policy-decision-requested",
    		      G_CALLBACK (decide_policy), NULL);

    gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (view));
    gtk_container_add (GTK_CONTAINER (window), scrolled);

    return view;
}

int main (int argc, char* argv[])
{
    WebKitWebView *view;
    GtkWidget *window;
    char *uri = NULL, *geometry = NULL;
    int i;
    
    gtk_init (&argc, &argv);

    for (i = 1 ; i < argc ; i += 1) {
	if (!strcmp (argv[i], "--geometry") && i < argc - 1) {
	    geometry = argv[i + 1];
	    i += 1;
	} else if (!uri && (argv[i][0] != '-')) {
	    uri = argv[i];
	} else {
	    fprintf (stderr, "usage: %s [--geometry specification] [url]\n",
		     argv[0]);

	    return 1;
	}
    }

    view = create_view (NULL, NULL, NULL);
    window = gtk_widget_get_toplevel (GTK_WIDGET (view));

    g_signal_connect (window, "delete-event",
		      G_CALLBACK (gtk_main_quit), NULL);
    
    if (geometry &&
	!gtk_window_parse_geometry (GTK_WINDOW (window), geometry)) {
        fprintf (stderr,
		 "%s: failed to parse geometry specification.\n",
		 argv[0]);
    }
    
    webkit_web_view_load_uri (view, uri ? uri : "http://localhost:29176");

    show_window (view, NULL);
    
    gtk_main ();

    return 0;
}
