-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "widgets"
require "moremath"
require "shapes"

function widgets.speedometer (values)
   local meta, self, oldnewindex, newindex

   self = widgets.column {
      gauge = widgets.clock {
	 padding = {0.01, 0.01},
	 color = {1, 1, 1},
	 opacity = 0.5,
	 
	 radius = {0.08, 0.055},
	 range = {0, 220},
	 spacing = {5, 4},
	 spread = {-4 * math.pi / 5, 3 * math.pi / 5},

	 markings = frames.transform {
	    [1] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">0</span>]]
	    },

	    [2] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">20</span>]]
	    },

	    [3] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">40</span>]]
	    },

	    [4] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">60</span>]]
	    },

	    [5] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">80</span>]]
	    },

	    [7] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">120</span>]]
	    },

	    [9] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">160</span>]]
	    },

	    [11] = widgets.layout {
	       text = [[<span font="Sans 7" color="White">200</span>]]
	    },
	 },
	 
	 hands = frames.transform {
	    [1] = shapes.line {
	       [1] = {0, -0.1},
	       [2] = {0, 0.6},
	       [3] = {0.03, 0},
	       [4] = {0, -0.1},
	       [5] = {-0.03, 0},
	       [6] = {0, 0.6},
	    },

	    [2] = shapes.line {
	       [1] = {0, -0.1},
	       [2] = {0, 0.75},
	       [3] = {0.05, 0},
	       [4] = {0, -0.1},
	       [5] = {-0.05, 0},
	       [6] = {0, 0.75},
	    },
	 },
      },
      
      reading = widgets.frame {
	 color = {1, 1, 1},
	 opacity = 0.5,

	 layout = widgets.layout {
	    padding = {0.01, 0.01},

	    timer = frames.timer {
	       period = 1,
	       frames = 0,
	       dead = 0,

	       tick = function (self, tick, delta, elapsed)
			 local rate

			 if graphics.frames == 0 then
			    self.dead = elapsed
			 end

			 rate = {
			    (graphics.frames - self.frames) / delta,
			    graphics.frames / (elapsed - self.dead)
			 }

			 self.ancestry[3].gauge.reading = rate
			 self.parent.text = string.format ([[
<span font="Sans 10" color="White">%.1f(%.1f) Hz</span>]], rate[1], rate[2])
			 
			 self.frames = graphics.frames
		      end
	    },
	 },
      },
   }

   meta = getmetatable (self)

   meta.color = {1, 1, 1}
   meta.opacity = 0.5
   meta.background = 0.5

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     local meta = getmetatable(self)

		     if key == "color" then
			return meta.color
		     elseif key == "opacity" then
			return meta.opacity
		     elseif key == "background" then
			return meta.background
		     else
			return oldindex (self, key)
		     end
		  end

   meta.__newindex = function (self, key, value)
      local meta = getmetatable(self)

      if key == "color" then
	 meta.color = value

	 self.gauge.hands[1].color = value
	 self.gauge.hands[2].color = value
	 self.gauge.color = value
	 self.reading.color = value
      elseif key == "opacity" then
	 meta.opacity = value

	 self.gauge.opacity = value
	 self.reading.opacity = value
      elseif key == "background" then
	 meta.background = value

	 self.gauge.background = value
	 self.reading.background = value
      else
	 oldnewindex(self, key, value)
      end
   end

   meta.__tostring = function(self)
			 return "Speedometer"
		      end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end	    

function widgets.highlight (values)
   local meta, self, oldnewindex, newindex

   self = frames.transform {
      outline = shapes.halo {
	 color = {1, 1, 1},
	 opacity = 0,
	 width = 1,
      },

      annotation = widgets.annotation {
	 position = {0, 0, 0},
	 
	 padding = {0.01, 0.01},
	 color = {1, 1, 1}, 
	 opacity = 0,
	 thickness = 1,
	 radius = 0,
	 angle = 45,
	 
	 layout = widgets.layout {
	    opacity = 0,
	 },
      },

      fader = frames.timer {
	 period = 0.01,
	 opacity = 0,
	 speed = -1,

	 tick = function (self, ticks, delta, elapsed)
	    self.opacity = math.clamp(
	       self.opacity + self.speed * delta,
	       0, 1)

	    self.parent.outline.opacity = self.opacity
	    self.parent.annotation.opacity = meta.text and self.opacity or 0
	    self.parent.annotation.layout.opacity = meta.text and self.opacity or 0
	 end,
      },
   }

   meta = getmetatable (self)

   meta.color = {1, 1, 1}
   meta.opacity = 1
   meta.width = 1
   meta.thickness = 1
   meta.radius = 0
   meta.angle = 45

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     local meta = getmetatable(self)

		     if key == "color" then
			return meta.color
		     elseif key == "opacity" then
			return meta.opacity
		     elseif key == "width" then
			return meta.width
		     elseif key == "text" then
			return meta.text
		     elseif key == "speed" then
			return meta.speed
		     elseif key == "thickness" then
			return meta.thickness
		     elseif key == "radius" then
			return meta.radius
		     elseif key == "angle" then
			return meta.angle
		     elseif key == "mesh" then
			return self.outline.mesh
		     else
			return oldindex (self, key)
		     end
		  end

   meta.__newindex = function (self, key, value)
      local meta = getmetatable(self)

      if key == "color" then
	 meta.color = value

	 self.outline.color = value
	 self.annotation.color = value
      elseif key == "text" then
	 meta.text = value

	 self.annotation.layout.text = value
      elseif key == "width" then
	 meta.width = value

	 self.outline.width = value
      elseif key == "thickness" then
	 meta.thickness = value

	 self.annotation.thickness = value
      elseif key == "radius" then
	 meta.radius = value
	 
	 self.annotation.radius = value
      elseif key == "angle" then
	 meta.angle = value

	 self.annotation.angle = value
      elseif key == "opacity" then
	 meta.opacity = value

	 self.fader.opacity = value
      elseif key == "speed" then
	 meta.speed = value

	 self.fader.speed = value
      elseif key == "mesh" then
	 self.outline.mesh = value
      else
	 oldnewindex(self, key, value)
      end
   end

   meta.__tostring = function(self)
			 return "Highlight"
		      end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end	    


function widgets.console (values)
   local meta, self, oldnewindex, newindex

   self = widgets.display {
      align = {-1, -1},
   }
   
   meta = getmetatable (self)
   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function (self, key)
		     if key == "text" then
			return self.string and self.string.text
		     else
			return oldindex (self, key)
		     end
		  end

   meta.__newindex = function (self, key, value)
			if key == "text" then
			   self.string = widgets.layout {
			      align = {-1, 0},

			      text = tostring(value),

			      fader = frames.timer {
				 period = 0.02,
				 timeout = 5,
				 
				 tick = function (self, n, delta, elapsed)
					   self.parent.opacity = self.timeout -
					      elapsed
					   
					   if elapsed >= self.timeout then
					      self.ancestry[2] = nil
					   end
					end
			      }
			   }
			else
			   oldnewindex(self, key, value)
			end
		     end

   meta.__tostring = function(self)
			 return "Console"
		      end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end
