/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include "timer.h"

static void calltimedhooks (lua_State *L, void *key, int reference,
			    int tick, double delta, double elapsed)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	    
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    lua_pushnumber (L, tick);
	    lua_pushnumber (L, delta);
	    lua_pushnumber (L, elapsed);
	    luaX_call(L, 4, 0);
	} else if (lua_istable (L, -1)) {
	    int i, n;
	    
	    n = lua_objlen (L, -1);

	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (L, -2, i + 1);

		lua_pushlightuserdata (L, key);
		lua_gettable (L, -3);

		lua_pushnumber (L, tick);
		lua_pushnumber (L, delta);
		lua_pushnumber (L, elapsed);
		luaX_call (L, 4, 0);
	    }
	    
	    lua_pop (L, 1);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Timer

-(Timer *) init
{
    char *list[] = {"period", "tick", "state"};
    
    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->tick = LUA_REFNIL;    
    self->period = 1;
    
    return self;
}

-(void) toggle
{
    [super toggle];

    if ([self linked]) {
	clock_gettime (CLOCK_REALTIME, &self->checkpoint);
	self->elapsed = 0;
	self->delta = 0;
	self->count = 0;
    }
}

-(void) free
{
     luaL_unref (_L, LUA_REGISTRYINDEX, self->tick);

     [super free];
}

-(void) tick
{
    struct timespec time;

    clock_gettime (CLOCK_REALTIME, &time);   

    self->delta = time.tv_sec - self->checkpoint.tv_sec +
	         (time.tv_nsec - self->checkpoint.tv_nsec) / 1e9;

    if (self->delta > self->period) {
	self->elapsed += self->delta;
	self->checkpoint = time;
	self->count += 1;

	calltimedhooks (_L, self,
			self->tick, self->count,
			self->delta, self->elapsed);
    }
}

-(void) begin
{
    [self tick];
    [super begin];
}

-(void) stepBy: (double) h at: (double) t
{
    [self tick];
    [super stepBy: h at: t];
}

-(void) prepare
{
    [self tick];
    [super prepare];
}

-(void) traversePass: (int) pass
{
    [self tick];
    [super traversePass: pass];
}

-(void) finish
{
    [self tick];
    [super finish];
}

-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "period")) {
        lua_pushnumber (_L, self->period);
    } else if (!xstrcmp(k, "tick")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->tick);
    } else if (!xstrcmp(k, "state")) {
	lua_newtable (_L);

	lua_pushnumber (_L, self->count);
	lua_rawseti (_L, -2, 1);
	lua_pushnumber (_L, self->elapsed);
	lua_rawseti (_L, -2, 2);
    } else  {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "period")) {
        self->period = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "tick")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->tick);
        self->tick = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "state")) {
    } else {
	[super set];	
    }
}

@end
