/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glext.h>
#include "surface.h"
#include "light.h"
#include "ambient.h"
#include "fog.h"

static const GLchar *header =
"#version 110							    \n"
"#pragma optionNV(strict on)                                        \n"
"								    \n"
"const int maxLightCount = 8;					    \n"
"const int lightCount = 1;					    \n"
"								    \n"
"uniform vec3 sourcePositions[maxLightCount];			    \n"
"uniform vec3 sourceConstants[maxLightCount];			    \n"
"uniform vec3 sourceAttenuations[maxLightCount];	    	    \n"
"uniform mat3 sourceOrientations[maxLightCount];		    \n"
"uniform mat4 shadowMatrices[maxLightCount];			    \n"
"uniform bool sourceFlags[maxLightCount];			    \n"
"						    		    \n"
"uniform sampler2DShadow shadowMaps[maxLightCount];	            \n"
"uniform sampler2D sourceMaps[maxLightCount];    	            \n"
"						    		    \n"
"uniform vec3 ambienceConstant;					    \n"
"uniform sampler2D ambienceMap;			    	            \n"
"uniform mat3 ambienceFrame;					    \n"
"						    		    \n"
"uniform float fogOffset;				   	    \n"
"uniform vec2 fogDensity;				    	    \n"
"uniform vec3 fogColor;		    	    	    	    	    \n"
"								    \n"
"uniform bool hasAmbience, hasFog;		    	       	    \n"
"								    \n"
"vec3 sourcePosition (int i)	    				    \n"
"{								    \n"
"    return sourcePositions[i];				    	    \n"
"}								    \n"    
"								    \n"
"mat3 sourceOrientation (int i)	    				    \n"
"{								    \n"
"    return sourceOrientations[i];				    \n"
"}								    \n"    
"								    \n"
"float shadowFactor (int i, vec3 point)	    	    		    \n"
"{								    \n"
"    vec4 c;						    	    \n"
"			   		     			    \n"
"    if (sourceFlags[i]) {	     			    	    \n"
"        c = shadowMatrices[i] * vec4(point, 1.0);	    	    \n"
"			   		     			    \n"
"        return float(shadow2DProj (shadowMaps[i], c));       	    \n"
"    } else {		   		     			    \n"
"        return 1.0;       	    				    \n"
"    }								    \n"    
"}								    \n"    
"								    \n"
"float attenuationFactor (int i, vec3 point)	    		    \n"
"{								    \n"
"    vec3 l;					    	    	    \n"
"    float dsquared;					    	    \n"
"			   		     			    \n"
"    l = sourcePositions[i] - point;	    		    	    \n"
"    dsquared = dot(l, l);	    		    		    \n"
"			   		     			    \n"
"    return 1.0 / dot(sourceAttenuations[i],		    	    \n"
"                     vec3(1.0, sqrt(dsquared), dsquared));    	    \n"
"}								    \n"    
"								    \n"
"vec3 luminousIntensity (int i, vec3 vector)	    		    \n"
"{								    \n"
"    vec2 c;						    	    \n"
"			   		     			    \n"
"    c = 0.5 + 0.5 * vec2(dot(vector, sourceOrientations[i][0]),    \n"
"                         dot(vector, sourceOrientations[i][1]));   \n"
"			   		     			    \n"
"    if(dot(vector, sourceOrientations[i][2]) < 0.0) {		    \n"
"        return sourceConstants[i] + 		    		    \n"
"               vec3(texture2D(sourceMaps[i], c));		    \n"
"    } else {							    \n"
"        return  sourceConstants[i];		    		    \n"
"    }								    \n"
"}								    \n"    
"								    \n"
"vec3 ambientIntensity(vec3 normal)			   	    \n"
"{						   	 	    \n"    
"    vec2 c;						    	    \n"
"			   		     			    \n"
"    if (hasAmbience) {	     			    		    \n"
"        c = 0.5 + 0.5 * vec2(dot(normal, ambienceFrame[0]),        \n"
"                             dot(normal, ambienceFrame[2]));  	    \n"
"			   		     			    \n"
"        return ambienceConstant + 		   	    	    \n"
"               vec3(texture2D(ambienceMap, c));      	    	    \n"
"    } else {		   		     			    \n"
"        return vec3(0.0, 0.0, 0.0);   				    \n"
"    }								    \n"    
"}						  	 	    \n"
"						   	 	    \n"
"vec3 befog(vec3 intensity, vec3 point)				    \n"
"{						   	 	    \n"    
"    float z, f;					    	    \n"
"						   	 	    \n"
"    if (hasFog) {	     			    		    \n"
"        z = min(point.z + fogOffset, 0.0);	    	    	    \n"
"        f = exp(fogDensity[0] * z -			    	    \n"
"                fogDensity[1] * z * z);    	    	    	    \n"
"						   	 	    \n"
"        return f * intensity + (1.0 - f) * fogColor.rgb; 	    \n"
"    } else {		   		     			    \n"
"        return intensity;       	    			    \n"
"    }								    \n"    
"}						  	 	    \n"
"						   	 	    \n"
"#line 1							    \n";

@implementation Surface

-(Surface *)init
{
    int i;

    self->surface.lightCount =
	glGetUniformLocationARB (self->program, "lightCount");

    self->surface.hasAmbience =
	glGetUniformLocationARB (self->program, "hasAmbience");

    self->surface.hasFog =
	glGetUniformLocationARB (self->program, "hasFog");

    self->surface.fogDensity =
	glGetUniformLocationARB (self->program, "fogDensity");

    self->surface.fogOffset =
	glGetUniformLocationARB (self->program, "fogOffset");

    self->surface.fogColor =
	glGetUniformLocationARB (self->program, "fogColor");

    self->surface.ambienceFrame =
	glGetUniformLocationARB (self->program, "ambienceFrame");

    self->surface.ambienceMap =
	glGetUniformLocationARB (self->program, "ambienceMap");

    self->surface.ambienceConstant =
	glGetUniformLocationARB (self->program, "ambienceConstant");
    
    for (i = 0 ; i < 8 ; i += 1) {
	char name[64];

	snprintf(name, 64, "sourcePositions[%d]", i);
	self->surface.sourcePositions[i] =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "sourceOrientations[%d]", i);
	self->surface.sourceOrientations[i] =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "shadowMatrices[%d]", i);
	self->surface.shadowMatrices[i] =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "sourceConstants[%d]", i);
	self->surface.sourceConstants[i] =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "sourceAttenuations[%d]", i);
	self->surface.sourceAttenuations[i] =
	    glGetUniformLocationARB (self->program, name);
	
	snprintf(name, 64, "sourceFlags[%d]", i);
	self->surface.sourceFlags[i] =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "shadowMaps[%d]", i);
	self->surface.shadowMaps[i] =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "sourceMaps[%d]", i);
	self->surface.sourceMaps[i] =
	    glGetUniformLocationARB (self->program, name);
    }
    
    [super init];

    return self;
}

-(void) build
{
    self->program = glCreateProgramObjectARB();
}

-(void)attachVertexSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);
    
    self->vertex = glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB);
    glShaderSourceARB(self->vertex, 1, (const GLchar **)&newsource, NULL);
    glCompileShaderARB(self->vertex);
}

-(void)attachFragmentSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);
    
    self->fragment = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);
    glShaderSourceARB(self->fragment, 1, (const GLchar **)&newsource, NULL);
    glCompileShaderARB(self->fragment);
}

-(void)link
{
    glAttachObjectARB(self->program, self->vertex);
    glAttachObjectARB(self->program, self->fragment);
    glLinkProgramARB (self->program);
}

-(void)verify
{
    int p, m, n;

    if (_WINDOW) {
	glGetObjectParameterivARB (self->program,
				   GL_OBJECT_LINK_STATUS_ARB,
				   &p);

	if (p != GL_TRUE) {
	    printf ("\nThe program for %s nodes did not "
		    "build properly.\n", [self name]);
	}

	glGetObjectParameterivARB(self->vertex,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(self->vertex, n, &m, buffer);
	    printf ("Info log for the `%s' vertex source follows:\n\n%s\n",
		    [self name], buffer);
	}

	glGetObjectParameterivARB(self->fragment,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(self->fragment, n, &m, buffer);
	    printf ("Info log for the `%s' fragment source follows:\n\n%s\n",
		    [self name], buffer);
	}

	glGetObjectParameterivARB(self->program,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(self->program, n, &m, buffer);
	    printf ("Info log for the `%s' program follows:\n\n%s\n",
		    [self name], buffer);
	}
    }
}

-(void)traversePass: (int) pass
{
    id parent, child;
    int n = 0, hasAmbience = 0, hasFog = 0;
    
    if (_WINDOW && pass > 0) {
	for (parent = [self parent] ; parent ; parent = [parent parent]) {
	    for (child = [parent children] ; child ; child = [child sister]) {
		if ([child isMemberOf: [Light class]]) {
		    [child loadPositionInto: self->surface.sourcePositions[n]
		         andOrientationInto: self->surface.sourceOrientations[n]];

		    [child loadAttenuationInto: self->surface.sourceAttenuations[n]];

		    [child loadIntesityConstantInto: self->surface.sourceConstants[n]
		                         andMapInto: self->surface.sourceMaps[n]];

		    [child loadShadowMatrixInto: self->surface.shadowMatrices[n]                                     andMapInto: self->surface.shadowMaps[n]
		                    andFlagInto: self->surface.sourceFlags[n]];

		    n += 1;
		} else if ([child isMemberOf: [Ambient class]] && !hasAmbience) {
		    [child loadOrientationInto: self->surface.ambienceFrame];
		    [child loadIntesityConstantInto: self->surface.ambienceConstant
		                         andMapInto:self->surface.ambienceMap];

		    hasAmbience += 1;
		} else if ([child isMemberOf: [Fog class]] && !hasFog) {
		    [child loadOffsetInto: self->surface.fogOffset];
		    [child loadDensityInto: self->surface.fogDensity];
		    [child loadColorInto: self->surface.fogColor];
		    
		    hasFog += 1;
		}
	    }
	}

	glUniform1iARB (self->surface.lightCount, n);
	glUniform1iARB (self->surface.hasAmbience, hasAmbience);
	glUniform1iARB (self->surface.hasFog, hasFog);
    }
    
    [super traversePass: pass];
    
    glUseProgramObjectARB(0);
}

@end
