/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "racetrack.h"
#include "wheel.h"

static int callback (dGeomID TriMesh, dGeomID RefObject, int TriangleIndex)
{
    id object;

    /* Don't collide the racetrack backup geom with the wheel backup geom. */
    
    if (dGeomGetClass (RefObject) == dCylinderClass) {
	object = dGeomGetData(RefObject);
    
	if ([object isKindOf: [Wheel class]]) {
	    return 0;
	} else {
	    return 1;
	}
    } else {
	return 1;
    }
}

static int test (double *segments, int index, int size, double tolerance,
		 const dReal *q, dReal *n, dReal *d,
		 double *longitude, double *latitude)
{
    double theta_0, p_0[3], delta_wl, delta_wr, w_l0, w_r0, u, v;
    double S, kappa_0, delta_kappa, g_0, delta_g, e_0, delta_e;
    int i, j, s, t;

    for (s = 0, t = 0 ;
    	 t < size ;
    	 t += 1, s >= 0 ? (s = -(s + 1)) : (s *= -1)) {
    	/* if (t > 0) */
    	/*     printf ("%d, %d\n", index, t); */
	
    	i = (index + size + s) % size;
	j = i > 0 ? i - 1 : i;
	/* printf ("*** %d, %f\n", i, segments[10 * i]); */

	/* Initalize the lot. */
   
	w_l0 = segments[10 * j + 1];
	w_r0 = segments[10 * j + 2];
	kappa_0 = segments[10 * j + 3];
	g_0 = segments[10 * j + 4];
	e_0 = segments[10 * j + 5];

	p_0[0] = segments[10 * i + 6];
	p_0[1] = segments[10 * i + 7];
	p_0[2] = segments[10 * i + 8];
	theta_0 = segments[10 * i + 9];	

	S = segments[10 * i];
	delta_wl = segments[10 * i + 1] - w_l0;
	delta_wr = segments[10 * i + 2] - w_r0;
	delta_kappa = segments[10 * i + 3] - kappa_0;
	delta_g = segments[10 * i + 4] - g_0;
	delta_e = segments[10 * i + 5] - e_0;
	    
	if (fabs(kappa_0) < 1e-6 && fabs(delta_kappa) < 1e-6) {
	    double c, s;

	    /* This is a tangent. */
	    
	    c = cos(theta_0);
	    s = sin(theta_0);
	    
	    u = (q[0] - p_0[0]) * c + (q[1] - p_0[1]) * s;
	    v = -(q[0] - p_0[0] - u * c) * s + (q[1] - p_0[1] - u * s) * c;

	    if (u > 0 && u < S &&
		v > -(w_l0 + u / S * delta_wl) &&
		v < w_r0 + u / S * delta_wr) {
		double e, g, phi, psi;

		e = e_0 + delta_e * u / S;
		g = g_0 + delta_g * u / S;
		
		phi = atan(e);
		psi = -atan(g);

		n[0] = sin(theta_0) * sin(phi) +
		    cos(phi) * sin (psi) * cos (theta_0);
		n[1] = -cos(theta_0) * sin(phi) +
		    cos(phi) * sin (psi) * sin (theta_0);
		n[2] = cos(phi) * cos(psi);

		*d = p_0[2] +
		    u * g_0 + 0.5 * delta_g * u * u / S +
		    v * e - q[2];

		if (longitude) {
		    double u_0;
		    int ii;

		    for (ii = 0, u_0 = 0 ; ii < i; u_0 += segments[10 * ii], ii += 1);
		    *longitude = u_0 + u;
		}

		if (latitude) {
		    *latitude = v;
		}
		/* printf ("%d, %d, %f, %f\n", i, j, g_0, delta_g); */
		/* printf ("%d: %f, %f, %f, %f, %f, %f, %f\n", i, u, v, e, phi, n[1], n[2], *d); */

		return i;
	    }
	} else if (fabs(kappa_0) > 1e-6 && fabs(delta_kappa) < 1e-6) {
	    double m, s, r, theta, c[2];

	    /* This is a circular segment. */
	    
	    s = kappa_0 < 0 ? -1 : 1;
	    r = fabs(1 / kappa_0);
		
	    c[0] = p_0[0] - s * r * sin(theta_0);
	    c[1] = p_0[1] + s * r * cos(theta_0);

	    m = sqrt ((q[0] - c[0]) * (q[0] - c[0]) +
		      (q[1] - c[1]) * (q[1] - c[1]));

	    theta = atan2(-s * (c[0] - q[0]), s * (c[1] - q[1]));

	    u = s * (theta - theta_0) * r;

	    if (u < 0) {
		u += 2 * M_PI * r;
	    } else if (u > 2 * M_PI * r) {
		u -= 2 * M_PI * r;
	    }
	    
	    v = s * (r - m);
	    
	    if (u > 0 && u < S &&
		v > -(w_l0 + u / S * delta_wl) &&
		v < w_r0 + u / S * delta_wr) {
		double e, g, phi, psi;

		e = e_0 + delta_e * u / S;
		g = g_0 + delta_g * u / S;
		
		phi = atan(e);
		psi = -atan(g);
		
		n[0] = sin(theta) * sin(phi) +
		    cos(phi) * sin (psi) * cos (theta);
		n[1] = -cos(theta) * sin(phi) +
		    cos(phi) * sin (psi) * sin (theta);
		n[2] = cos(phi) * cos(psi);

		*d = p_0[2] +
		    u * g_0 + 0.5 * delta_g * u * u / S +
		    v * e - q[2];

		if (longitude) {
		    double u_0;
		    int ii;

		    for (ii = 0, u_0 = 0 ; ii < i; u_0 += segments[10 * ii], ii += 1);
		    *longitude = u_0 + u;
		}

		if (latitude) {
		    *latitude = v;
		}

		return i;
	    }
	} else {
	    double s, r, c[2], m, theta;
	    int k;

	    /* This is acutally a tranistional curve, i.e.
	       a spiral but we'll just pretend its a bunch
	       of circular sub-segments of varying radius
	       pasted together. */
	    
	    k = (int)fmax(ceil(fabs(delta_kappa) / tolerance), 1);

	    for (j = 0 ; j < k ; j += 1) {
		kappa_0 += delta_kappa / (k + 1);
	    
		s = kappa_0 < 0 ? -1 : 1;
		r = fabs(1 / kappa_0);
		
		c[0] = p_0[0] - s * r * sin(theta_0);
		c[1] = p_0[1] + s * r * cos(theta_0);

		m = sqrt ((q[0] - c[0]) * (q[0] - c[0]) +
			  (q[1] - c[1]) * (q[1] - c[1]));
		
		theta = atan2(-s * (c[0] - q[0]), s * (c[1] - q[1]));
	    
		u = s * (theta - theta_0) * r;
		
		if (u < 0) {
		    u += 2 * M_PI * r;
		} else if (u > 2 * M_PI * r) {
		    u -= 2 * M_PI * r;
		}
		
		v = s * (r - m);

		if (u > 0 && u < S / k &&
		    v > -(w_l0 + u / S * delta_wl) &&
		    v < w_r0 + u / S * delta_wr) {
		    double e, g, phi, psi;
		    
		    e = e_0 + delta_e * u / S;
		    g = g_0 + delta_g * u / S;

		    /* { */
		    /* 	static void *foo; */

		    /* 	if (!foo) foo = q; */

		    /* 	if (foo == q) { */
		    /* 	    printf ("%d, %f, %f, %f, %f\n", i, p_0[0], p_0[1], */
		    /* 		    c[0] + s * r * sin(theta_0+ s * S / k / r), */
		    /* 		    c[1] - s * r * cos(theta_0+ s * S / k / r)); */
		    /* 	} */
		    /* } */
		
		    phi = atan(e);
		    psi = -atan(g);
		
		    n[0] = sin(theta) * sin(phi) +
			cos(phi) * sin (psi) * cos (theta);
		    n[1] = -cos(theta) * sin(phi) +
			cos(phi) * sin (psi) * sin (theta);
		    n[2] = cos(phi) * cos(psi);

		    *d = p_0[2] +
			u * g_0 + 0.5 * delta_g * u * u / S +
			v * e - q[2];

		    if (longitude) {
			double u_0;
			int ii;

			for (ii = 0, u_0 = 0 ; ii < i; u_0 += segments[10 * ii], ii += 1);
			*longitude = u_0 + S * j / k + u;
		    }

		    if (latitude) {
			*latitude = v;
		    }

		    return i;
		}

		/* Update for the next segment. */

		e_0 += delta_e / k;
		theta_0 += s * S / k / r;
		w_l0 += delta_wl / k;
		w_r0 += delta_wr / k;
		
		p_0[0] = c[0] + s * r * sin(theta_0);
		p_0[1] = c[1] - s * r * cos(theta_0);
		p_0[2] += S / k * g_0 + 0.5 * delta_g * S / k / k;	    

		g_0 += delta_g / k;
	    }
	}
    }
    
    return -1;
}

static int collideWithWheel (dGeomID track,
			     dGeomID wheel,
			     int flags,
			     dContactGeom *contact,
			     int skip)
{
    struct wheeldata *wheeldata;
    struct trackdata *trackdata;

    const dReal *R_t, *r_t, *R_w, *r_w;
    dVector3 n = {0, 0, 1}, r, rprime;
    dMatrix3 R;
    
    trackdata = dGeomGetClassData (track);
    wheeldata = dGeomGetClassData (wheel);
	
    r_w = dGeomGetPosition (wheel);
    R_w = dGeomGetRotation (wheel);
	
    r_t = dGeomGetPosition (track);
    R_t = dGeomGetRotation (track);

    dOP (rprime, -, r_w, r_t);
    dMULTIPLY1_331 (r, R_t, rprime);
    dMULTIPLY1_333 (R, R_t, R_w);
   
    wheeldata->contact.g1 = wheel;
    wheeldata->contact.g2 = track;

    wheeldata->axial[0] = R[1];
    wheeldata->axial[1] = R[5];
    wheeldata->axial[2] = R[9];

    /* Get an estimate of the surface normal by
       testing the wheel center location. */
    
    test (trackdata->segments,
	  wheeldata->index,
	  trackdata->size,
	  trackdata->tolerance,
	  r, n, &wheeldata->contact.depth,
	  NULL, NULL);
	
    dCROSS(wheeldata->longitudinal, =, wheeldata->axial, n);
    dCROSS(wheeldata->radial, =, wheeldata->axial, wheeldata->longitudinal);
    dCROSS(wheeldata->lateral, =, n, wheeldata->longitudinal);

    dSafeNormalize3 (wheeldata->radial);

    /* Calculate the contact point. */

    wheeldata->contact.pos[0] =
	r[0] + wheeldata->radial[0] * wheeldata->radii[0] -
	n[0] * wheeldata->radii[1];
	
    wheeldata->contact.pos[1] =
	r[1] + wheeldata->radial[1] * wheeldata->radii[0] -
	n[1] * wheeldata->radii[1];
	
    wheeldata->contact.pos[2] =
	r[2] + wheeldata->radial[2] * wheeldata->radii[0] -
	n[2] * wheeldata->radii[1];

    wheeldata->index = test (trackdata->segments,
			     wheeldata->index,
			     trackdata->size,
			     trackdata->tolerance,
			     wheeldata->contact.pos,
			     wheeldata->contact.normal,
			     &wheeldata->contact.depth,
			     NULL, NULL);

    wheeldata->airborne = (wheeldata->index < 0) ||
	(wheeldata->contact.depth < 0);

    dMULTIPLY0_331 (rprime, R_t, wheeldata->contact.pos);
    dOP (wheeldata->contact.pos, +, rprime, r_t);

    dOPE (rprime, =, wheeldata->axial);
    dMULTIPLY0_331 (wheeldata->axial, R_t, rprime);

    dOPE (rprime, =, wheeldata->lateral);
    dMULTIPLY0_331 (wheeldata->lateral, R_t, rprime);

    dOPE (rprime, =, wheeldata->longitudinal);
    dMULTIPLY0_331 (wheeldata->longitudinal, R_t, rprime);

    dOPE (rprime, =, wheeldata->radial);
    dMULTIPLY0_331 (wheeldata->radial, R_t, rprime);

    dOPE (rprime, =, wheeldata->contact.normal);
    dMULTIPLY0_331 (wheeldata->contact.normal, R_t, rprime);
    
    /* printf ("%f, %f, %f\n", */
    /* 	    wheeldata->contact.pos[0], */
    /* 	    wheeldata->contact.pos[1], */
    /* 	    wheeldata->contact.pos[2]); */
   
    return 0;
}

static dColliderFn * getCollider (int num)
{
    if (num == dWheelClass) {
	return collideWithWheel;
    } else {
	return 0;
    }
}

@implementation Racetrack

-(Racetrack *)init
{
    char *list[] = {
	"tesselation", "scale"
    };
    
    struct trackdata *data;
    
    if (!dTrackClass) {
	struct dGeomClass class = {
	    sizeof (struct trackdata),
	    getCollider,
	    dInfiniteAABB,
	    0, 0
	};
      
	dTrackClass = dCreateGeomClass (&class);
    }
    
    self->meshdata = dGeomTriMeshDataCreate ();
    self->mesh = dCreateTriMesh (NULL, self->meshdata, callback, NULL, NULL);
    dGeomSetData (self->mesh, self);

    self->geom = dCreateGeom (dTrackClass);
    dGeomSetData (self->geom, self);

    self->vertices = NULL;
    self->normals = NULL;
    self->uv = NULL;
    self->indices = NULL;
    self->length = 0;
    self->dirty = 0;

    self->scale[0] = 1;
    self->scale[1] = 1;
    
    self->tesselation[0] = 2 * M_PI / 32;
    self->tesselation[1] = 0.01 / 10;
    self->tesselation[2] = 0.1 / 10;

    data = dGeomGetClassData (self->geom);

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    data->segments = NULL;
    data->size = 0;
    data->tolerance = 0.01 / 10;

    return self;
}

-(void) fasten
{
    [super fasten];

    dGeomSetBody (self->mesh, NULL);
}

-(void) release
{
    [super release];

    dGeomSetBody (self->mesh, self->body);
}

-(void) insertInto: (dSpaceID) new
{
    if (self->space) {
    	dSpaceRemove (self->space, self->mesh);
    }
    
    if (new) {
    	dSpaceAdd (new, self->mesh);
    }

    [super insertInto: new];
}

-(void) update
{
    struct trackdata *data;

    double theta_0, p_0[3], delta_wl, delta_wr, w_l0, w_r0, u_0;
    double S, kappa_0, delta_kappa, g_0, delta_g, e_0, delta_e;
    double phi, psi;
    int i, j;

    data = dGeomGetClassData (self->geom);

    u_0 = 0;

    theta_0 = 0;
    p_0[0] = 0;
    p_0[1] = 0;
    p_0[2] = 0;

    w_l0 = data->segments[1];
    w_r0 = data->segments[2];
    kappa_0 = data->segments[3];
    g_0 = data->segments[4];
    e_0 = data->segments[5];
    
    phi = atan(e_0);
    psi = -atan(g_0);
    
    self->vertices = realloc (self->vertices, 2 * 3 * sizeof (double));
    self->normals = realloc (self->normals, 2 * 3 * sizeof (double));
    self->uv = realloc (self->uv, 2 * 2 * sizeof (double));
    self->length = 0;

    /* Kickstart the strip. */

    self->vertices[0] = 0;
    self->vertices[1] = w_r0;
    self->vertices[2] = e_0 * w_r0;

    self->vertices[3] = 0;
    self->vertices[4] = -w_l0;
    self->vertices[5] = -e_0 * w_l0;

    self->normals[0] = cos(phi) * sin (psi);
    self->normals[1] = -sin(phi);
    self->normals[2] = cos(phi) * cos(psi);
    
    self->normals[3] = self->normals[0];
    self->normals[4] = self->normals[1];
    self->normals[5] = self->normals[2];

    self->uv[0] = 0;
    self->uv[1] = w_r0 / self->scale[1];
    
    self->uv[2] = 0;
    self->uv[3] = -w_l0 / self->scale[1];
    
    self->length += 2;

    /* printf ("%f, %f, %f\n", 0, w_r0, e_0 * w_r0); */
    /* printf ("%f, %f, %f\n", 0, -w_l0, -e_0 * w_l0); */

    for (i = 0 ; i < data->size ; i += 1) {
	data->segments[10 * i + 6] = p_0[0];
	data->segments[10 * i + 7] = p_0[1];
	data->segments[10 * i + 8] = p_0[2];
	data->segments[10 * i + 9] = theta_0;

	S = data->segments[10 * i];
	delta_wl = data->segments[10 * i + 1] - w_l0;
	delta_wr = data->segments[10 * i + 2] - w_r0;
	delta_kappa = data->segments[10 * i + 3] - kappa_0;
	delta_g = data->segments[10 * i + 4] - g_0;
	delta_e = data->segments[10 * i + 5] - e_0;

	if (fabs(kappa_0) < 1e-6 && fabs(delta_kappa) < 1e-6) {
	    double dv[2], u;
	    int m;

	    m = fmax(ceil (sqrt(fabs (delta_g) * S / 8 / self->tesselation[2])), 1);

	    /* printf ("++ %d, %f, %f\n", i, e_0, delta_e); */
		
	    /* A tangent. */

	    dv[0] = -sin(theta_0);
	    dv[1] = cos(theta_0);

	    for (j = 1 ; j <= m ; j += 1) {
		double h, p[2];

		u = S * j / m;
		h = u * g_0 + 0.5 * delta_g * u * u / S;
		
		p[0] = p_0[0] + S * j / m * cos(theta_0);
		p[1] = p_0[1] + S * j / m * sin(theta_0);

		e_0 += delta_e / m;
		w_l0 += delta_wl / m;
		w_r0 += delta_wr / m;

		phi = atan(e_0);
		psi = -atan(g_0 + delta_g * u / S);
		
		self->vertices = realloc (self->vertices,
					  (self->length + 2) *
					  3 * sizeof (double));
		
		self->normals = realloc (self->normals,
					 (self->length + 2) *
					 3 * sizeof (double));
		
		self->uv = realloc (self->uv,
				    (self->length + 2) *
				    2 * sizeof (double));
		
		self->vertices[3 * self->length] = p[0] + w_r0 * dv[0];
		self->vertices[3 * self->length + 1] = p[1] + w_r0 * dv[1];
		self->vertices[3 * self->length + 2] = p_0[2] + h + e_0 * w_r0;

		self->vertices[3 * self->length + 3] = p[0] - w_l0 * dv[0];
		self->vertices[3 * self->length + 4] = p[1] - w_l0 * dv[1];
		self->vertices[3 * self->length + 5] = p_0[2] + h - e_0 * w_l0;

		self->normals[3 * self->length + 0] =
		    sin(theta_0) * sin(phi) +
		    cos(phi) * sin (psi) * cos (theta_0);
		self->normals[3 * self->length + 1] =
		    -cos(theta_0) * sin(phi) +
		    cos(phi) * sin (psi) * sin (theta_0);
		self->normals[3 * self->length + 2] =
		    cos(phi) * cos(psi);

		self->normals[3 * self->length + 3] =
		    self->normals[3 * self->length + 0];
		self->normals[3 * self->length + 4] =
		    self->normals[3 * self->length + 1];
		self->normals[3 * self->length + 5] =
		    self->normals[3 * self->length + 2];

		self->uv[2 * self->length] = (u_0 + u) / self->scale[0];
		self->uv[2 * self->length + 1] = w_r0 / self->scale[1];
    
		self->uv[2 * self->length + 2] = (u_0 + u) / self->scale[0];
		self->uv[2 * self->length + 3] = -w_l0 / self->scale[1];

		self->length += 2;
	    }
		    
	    p_0[0] += S * cos(theta_0);
	    p_0[1] += S * sin(theta_0);
	    p_0[2] += S * g_0 + 0.5 * delta_g * S;
	} else if (fabs(kappa_0) > 1e-6 && fabs(delta_kappa) < 1e-6) {
	    double u, r, c[2];
	    int m;

	    r = 1 / kappa_0;

	    m = fmax(ceil(S / self->tesselation[0]),
		     ceil (sqrt(fabs (delta_g) * S / 8 / self->tesselation[2])));
	    m = m > 0 ? m : 1;
		
	    c[0] = p_0[0] - r * sin(theta_0);
	    c[1] = p_0[1] + r * cos(theta_0);

	    for (j = 0, u = S / m ; j < m ; j += 1, u += S / m) {
		double h, theta;
		    
		theta = theta_0 + S * (j + 1) / m / r;
		    
		e_0 += delta_e / m;
		w_l0 += delta_wl / m;
		w_r0 += delta_wr / m;
		    
		h = u * g_0 + 0.5 * delta_g * u * u / S;

		phi = atan(e_0);
		psi = -atan(g_0 + delta_g * u / S);
		
		self->vertices = realloc (self->vertices,
					  (self->length + 2) *
					  3 * sizeof (double));
		
		self->normals = realloc (self->normals,
					 (self->length + 2) *
					 3 * sizeof (double));
		
		self->uv = realloc (self->uv,
					  (self->length + 2) *
					  2 * sizeof (double));
		
		self->vertices[3 * self->length] =
		    c[0] + (r - w_r0) * sin(theta);
		self->vertices[3 * self->length + 1] =
		    c[1] - (r - w_r0) * cos(theta);
		self->vertices[3 * self->length + 2] =
		    p_0[2] + h + w_r0 * e_0;

		self->vertices[3 * self->length + 3] =
		    c[0] + (r + w_l0) * sin(theta);
		self->vertices[3 * self->length + 4] =
		    c[1] - (r + w_l0) * cos(theta);
		self->vertices[3 * self->length + 5] =
		    p_0[2] + h - w_l0 * e_0;

		self->normals[3 * self->length + 0] =
		    sin(theta) * sin(phi) +
		    cos(phi) * sin (psi) * cos (theta);
		self->normals[3 * self->length + 1] =
		    -cos(theta) * sin(phi) +
		    cos(phi) * sin (psi) * sin (theta);
		self->normals[3 * self->length + 2] =
		    cos(phi) * cos(psi);

		self->normals[3 * self->length + 3] =
		    self->normals[3 * self->length + 0];
		self->normals[3 * self->length + 4] =
		    self->normals[3 * self->length + 1];
		self->normals[3 * self->length + 5] =
		    self->normals[3 * self->length + 2];

		self->uv[2 * self->length] = (u_0 + u) / self->scale[0];
		self->uv[2 * self->length + 1] = w_r0 / self->scale[1];
    
		self->uv[2 * self->length + 2] = (u_0 + u) / self->scale[0];
		self->uv[2 * self->length + 3] = -w_l0 / self->scale[1];

		self->length += 2;

		/* printf ("%f, %f, %f\n", c[0] + (r - w_r0) * sin(theta), */
		/* 	    c[1] - (r - w_r0) * cos(theta), */
		/* 	    p_0[2] + h + w_r0 * e_0); */
		/* printf ("%f, %f, %f\n", c[0] + (r + w_l0) * sin(theta), */
		/* 	    c[1] - (r + w_l0) * cos(theta), */
		/* 	    p_0[2] + h - w_l0 * e_0); */
	    }
		
	    theta_0 += S / r;
	    p_0[0] = c[0] + r * sin(theta_0);
	    p_0[1] = c[1] - r * cos(theta_0);
	    p_0[2] += S * g_0 + 0.5 * delta_g * S;
	} else {
	    double r, c[2], u;
	    int m, n, k;

	    m = (int)fmax(ceil(fabs(delta_kappa) / self->tesselation[1]), 1);

	    n = fmax(ceil(S / m / self->tesselation[0]),
		     ceil (sqrt(fabs (delta_g * S) / m / m / 8 / self->tesselation[2])));
	    n = n > 0 ? n : 1;
	    /* printf ("* %d, %d, %d\n", i, m, n); */
		
	    for (j = 0, u = 0; j < m ; j += 1) {
		double h;

		kappa_0 += delta_kappa / (m + 1);
		r = 1 / kappa_0;
		
		c[0] = p_0[0] - r * sin(theta_0);
		c[1] = p_0[1] + r * cos(theta_0);

		for (k = 0 ; k < n ; k += 1) {
		    double theta;
		    
		    e_0 += delta_e / m / n;
		    w_l0 += delta_wl / m / n;
		    w_r0 += delta_wr / m / n;
		    u += S / m / n;
		    
		    theta = theta_0 + S * (k + 1) / n / r / m;
		    h = u * g_0 + 0.5 * delta_g * u * u / S;
	
		    phi = atan(e_0);
		    psi = -atan(g_0 + delta_g * u / S);
	
		    self->vertices = realloc (self->vertices,
					      (self->length + 2) *
					      3 * sizeof (double));

		    self->normals = realloc (self->normals,
					     (self->length + 2) *
					     3 * sizeof (double));

		    self->uv = realloc (self->uv,
					(self->length + 2) *
					2 * sizeof (double));
		
		    self->vertices[3 * self->length] =
			c[0] + (r - w_r0) * sin(theta);
		    self->vertices[3 * self->length + 1] =
			c[1] - (r - w_r0) * cos(theta);
		    self->vertices[3 * self->length + 2] =
			p_0[2] + h + w_r0 * e_0;

		    self->vertices[3 * self->length + 3] =
			c[0] + (r + w_l0) * sin(theta);
		    self->vertices[3 * self->length + 4] =
			c[1] - (r + w_l0) * cos(theta);
		    self->vertices[3 * self->length + 5] =
			p_0[2] + h - w_l0 * e_0;

		    self->normals[3 * self->length + 0] =
			sin(theta) * sin(phi) +
			cos(phi) * sin (psi) * cos (theta);
		    self->normals[3 * self->length + 1] =
			-cos(theta) * sin(phi) +
			cos(phi) * sin (psi) * sin (theta);
		    self->normals[3 * self->length + 2] =
			cos(phi) * cos(psi);

		    self->normals[3 * self->length + 3] =
			self->normals[3 * self->length + 0];
		    self->normals[3 * self->length + 4] =
			self->normals[3 * self->length + 1];
		    self->normals[3 * self->length + 5] =
			self->normals[3 * self->length + 2];

		    self->uv[2 * self->length] = (u_0 + u) / self->scale[0];
		    self->uv[2 * self->length + 1] = w_r0 / self->scale[1];
    
		    self->uv[2 * self->length + 2] = (u_0 + u) / self->scale[0];
		    self->uv[2 * self->length + 3] = -w_l0 / self->scale[1];
		    
		    self->length += 2;
	
		    /* printf ("%f, %f, %f\n", c[0] + (r - w_r0) * sin(theta), */
		    /* 		c[1] - (r - w_r0) * cos(theta), */
		    /* 		p_0[2] + h + w_r0 * e_0); */
		    /* printf ("%f, %f, %f\n", c[0] + (r + w_l0) * sin(theta), */
		    /* 		c[1] - (r + w_l0) * cos(theta), */
		    /* 		p_0[2] + h - w_l0 * e_0); */
		}

		theta_0 += S / r / m;
		
		p_0[0] = c[0] + r * sin(theta_0);
		p_0[1] = c[1] - r * cos(theta_0);
	    }

	    p_0[2] += S * g_0 + 0.5 * delta_g * S;
	}

	u_0 += S;
	w_l0 = data->segments[10 * i + 1];
	w_r0 = data->segments[10 * i + 2];
	kappa_0 = data->segments[10 * i + 3];
	g_0 = data->segments[10 * i + 4];
	e_0 = data->segments[10 * i + 5];
    }

    /* printf ("!!! %f, %f, %f,  %f\n", p_0[0], p_0[1], p_0[2], theta_0 / M_PI); */

    /* Write the indices. */

    self->indices = realloc (self->indices, 6 * (self->length / 2 - 1) * sizeof (dTriIndex));

    for (i = 0 ; i < self->length / 2 - 1 ; i += 1) {
	self->indices[6 * i] = 2 * i;
	self->indices[6 * i + 1] = 2 * i + 1;
	self->indices[6 * i + 2] = 2 * i + 2;
	self->indices[6 * i + 3] = 2 * i + 2;
	self->indices[6 * i + 4] = 2 * i + 1;
	self->indices[6 * i + 5] = 2 * i + 3;
    }

    /* Build the mesh data. */
    
    dGeomTriMeshSetData (self->mesh, NULL);
    dGeomTriMeshDataDestroy (self->meshdata);
    
    self->meshdata = dGeomTriMeshDataCreate ();

    dGeomTriMeshDataBuildDouble (self->meshdata,
			   self->vertices, 3 * sizeof(double), self->length,
			   self->indices, 6 * (self->length / 2 - 1),
			   3 * sizeof (dTriIndex));

    dGeomTriMeshSetData (self->mesh, self->meshdata);
    
    self->dirty = 0;
}

-(void) begin
{
    [super begin];

    if (self->dirty) {
	[self update];
    }
}

/* -(void) transform */
/* { */
/*     dGeomTriMeshDataSet(self->meshdata, */
/* 			TRIMESH_LAST_TRANSFORMATION, */
/* 			(void *) matrix); */
/*     [super transform]; */
/* } */

-(void) get
{
    struct trackdata *data;

    const char *k;
    int i, j;
    
    data = dGeomGetClassData (self->geom);
    k = lua_tostring (_L, 2);

    if (lua_istable (_L, 2)) {
	double n[3], d, u, v;
	double p[3] = {0, 0, 0};
	
	lua_rawgeti (_L, 2, 1);
	p[0] = lua_tonumber (_L, -1);

	lua_rawgeti (_L, 2, 2);
	p[1] = lua_tonumber (_L, -1);

	lua_pop (_L, 2);

	if (self->dirty) {
	    [self update];
	}

	if (test (data->segments, 0, data->size, data->tolerance, p, n, &d, &u, &v) >= 0) {
	    lua_newtable (_L);
	    lua_pushnumber (_L, u);
	    lua_rawseti (_L, -2, 1);
	    
	    lua_pushnumber (_L, v);
	    lua_rawseti (_L, -2, 2);
	    
	    lua_pushnumber (_L, d);
	    lua_rawseti (_L, -2, 3);	    
	} else {
	    lua_pushboolean (_L, 0);
	}
    } else if (lua_isnumber (_L, 2)) {
	j = lua_tonumber (_L, 2) - 1;
	
	lua_newtable (_L);

        for(i = 0; i < 6; i += 1) {
            lua_pushnumber (_L, data->segments[10 * j + i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "vertices")) {
	if (self->dirty) {
	    [self update];
	}

	lua_newtable (_L);

	for (i = 0 ; i < 3 * self->length ; i += 1) {
	    lua_pushnumber (_L, self->vertices[i]);
	    lua_rawseti (_L, -2, i + 1);
	}
    } else if (!xstrcmp(k, "scale")) {
	lua_newtable (_L);
	
	lua_pushnumber (_L, self->scale[0]);
	lua_rawseti (_L, 3, 1);
	
	lua_pushnumber (_L, self->scale[1]);
	lua_rawseti (_L, 3, 2);
    } else if (!xstrcmp(k, "tesselation")) {
	lua_newtable (_L);
	
	lua_pushnumber (_L, self->tesselation[0]);
	lua_rawseti (_L, 3, 1);
	
	lua_pushnumber (_L, self->tesselation[1]);
	lua_rawseti (_L, 3, 2);
	
	lua_pushnumber (_L, self->tesselation[2]);
	lua_rawseti (_L, 3, 3);
    } else {
	[super get];
    }
}

-(void) set
{
    struct trackdata *data;

    const char *k;
    int i, j;

    data = dGeomGetClassData (self->geom);
    k = lua_tostring (_L, 2);

    if (lua_isnumber (_L, 2)) {
        if(lua_istable (_L, 3)) {
	    j = lua_tonumber (_L, 2);

	    if (j > data->size) {
		data->size = j;
		data->segments = (double *)realloc (data->segments,
						    10 * j * sizeof(double));
	    }

            for(i = 0 ; i < 6 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                data->segments[10 * (j - 1) + i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    self->dirty = 1;
        }
    } else if (!xstrcmp(k, "scale")) {
	lua_rawgeti (_L, 3, 1);
	self->scale[0] = lua_tonumber (_L, -1);
	
	lua_rawgeti (_L, 3, 2);
	self->scale[1] = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "tesselation")) {
	lua_rawgeti (_L, 3, 1);
	self->tesselation[0] = lua_tonumber (_L, -1);
	
	lua_rawgeti (_L, 3, 2);
	self->tesselation[1] = lua_tonumber (_L, -1);
	data->tolerance = lua_tonumber (_L, -1);
	
	lua_rawgeti (_L, 3, 3);
	self->tesselation[2] = lua_tonumber (_L, -1);

	lua_pop (_L, 3);
    } else {
	[super set];
    }
}

-(void) free
{
    struct trackdata *data;

    data = dGeomGetClassData (self->geom);

    if (data->segments) {
	free (data->segments);
    }
    
    if (self->vertices) {
	free (self->vertices);
	free (self->normals);
	free (self->uv);
	free (self->indices);
    }

    dGeomTriMeshDataDestroy (self->meshdata);
    dGeomDestroy (self->mesh);

    [super free];
}

-(void) traversePass: (int)pass
{
   if (pass == 1) {
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([self matrix]);

	glEnable (GL_DEPTH_TEST);
	glEnable (GL_CULL_FACE);
   
	/* glPolygonMode (GL_FRONT_AND_BACK, GL_LINE); */
 
	glClientActiveTexture (GL_TEXTURE0);

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_NORMAL_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	    
	glVertexPointer(3, GL_DOUBLE, 0, self->vertices);
	glNormalPointer(GL_DOUBLE, 0, self->normals);
	glTexCoordPointer(2, GL_DOUBLE, 0, self->uv);

	glDrawArrays(GL_TRIANGLE_STRIP, 0, length);

	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);

	glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
	
	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
    
	glPopMatrix();
    }
    
    [super traversePass: pass];
}

@end
