/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <math.h>
#include <unistd.h>

#include "plConfig.h"
#include "plplot.h"
#include "plevent.h"

static void prepare (lua_State *L, PLFLT **xs, PLFLT **ys, int *n)
{
    int i;

    /* Setup style and color. */
    
    lua_getfield (L, lua_upvalueindex(1), "color");
    if (lua_istable (L, -1)) {
	int r, g, b;
	
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);
	lua_rawgeti (L, -3, 3);

	r = lua_tonumber(L, -3) * 255;
	g = lua_tonumber(L, -2) * 255;
	b = lua_tonumber(L, -1) * 255;
	
	lua_pop (L, 3);
	
	plscol0 (lua_tointeger (L, 1), r, g, b);
	plcol0(lua_tointeger (L, 1));
    }	
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "size");
    if (lua_isnumber (L, -1)) {
	plschr (0, lua_tonumber (L, -1));
	plssym (0, lua_tonumber (L, -1));
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "style");
    if (lua_isnumber (L, -1)) {
	int n;

	n = lua_tointeger (L, -1);

	if (n > 0 && n < 9) {
	    pllsty (lua_tointeger (L, -1));
	}
	
	if (n >= 0 && n < 9) {
	    plpsty (lua_tointeger (L, -1));
	}
    } else {
	pllsty (1);
	plpsty (0);
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "width");
    if (lua_isnumber (L, -1)) {
	plwid (lua_tonumber (L, -1));
    } else {
	plwid (1);
    }
    lua_pop(L, 1);

    if (n && xs && ys) {
	*n = luaX_objlen (L, lua_upvalueindex(1));
	*xs = (PLFLT *)malloc (*n * sizeof(PLFLT));
	*ys = (PLFLT *)malloc (*n * sizeof(PLFLT));
    
	for (i = 0 ; i < *n ; i += 1) {
	    lua_rawgeti (L, lua_upvalueindex(1), i + 1);

	    if (lua_istable (L, -1)) {
		lua_rawgeti (L, -1, 1);
		lua_rawgeti (L, -2, 2);

		(*xs)[i] = lua_tonumber (L, -2);
		(*ys)[i] = lua_tonumber (L, -1);

		lua_pop (L, 2);
	    }

	    lua_pop (L, 1);
	}
    }
}

static int constructcartesian (lua_State *L)
{
    char *buffer;
    size_t size;
    FILE *file;
    PLFLT xtick, ytick, fontscale = 3;
    PLINT nxsub, nysub;
    int i, n, ticks = 1;

    luaL_checktype (L, 1, LUA_TTABLE);

    /* plplot only seems to support writing to files
       so we'll have to open a memory buffer as a file. */

    file = open_memstream (&buffer, &size);
    plsfile (file);

    /* Set the output image size. */
    
    lua_getfield (L, 1, "size");
    if (lua_istable (L, -1)) {
    	char *geometry;

    	lua_rawgeti (L, -1, 1);
    	lua_rawgeti (L, -2, 2);
	
    	asprintf (&geometry, "%sx%s",
    		  lua_tostring (L, -2),
    		  lua_tostring (L, -1));

    	lua_pop (L, 2);

    	plsetopt("-geometry", geometry);
    } else {
    	plsetopt("-geometry", "400x400");
    }
    lua_pop(L, 1);
   
    plsdev ("svg");
    plscmap0n (256);
    
    /* Set the background color. */
    
    lua_getfield (L, 1, "background");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);
	lua_rawgeti (L, -3, 3);

	plscol0 (0,
		 lua_tonumber(L, -3) * 255,
		 lua_tonumber(L, -2) * 255,
		 lua_tonumber(L, -1) * 255);
	
	lua_pop (L, 3);
    }	
    lua_pop(L, 1);

    plinit();

    pladv(1);
    plschr (2.5, 1);
    plvsta();

    /* Set the foreground color. */
    
    lua_getfield (L, 1, "foreground");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);
	lua_rawgeti (L, -3, 3);

	plscol0 (255,
		 lua_tonumber(L, -3) * 255,
		 lua_tonumber(L, -2) * 255,
		 lua_tonumber(L, -1) * 255);
	
	lua_pop (L, 3);
    }
    lua_pop(L, 1);

    lua_getfield (L, 1, "fontscale");
    if (lua_isnumber (L, -1)) {
	fontscale = lua_tonumber (L, -1);
    }
    lua_pop(L, 1);

    /* Set the window coordinate mapping. */
    
    lua_getfield (L, 1, "window");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);
	lua_rawgeti (L, -3, 3);
	lua_rawgeti (L, -4, 4);

	plwind (lua_tonumber(L, -4),
		lua_tonumber(L, -3),
		lua_tonumber(L, -2),
		lua_tonumber(L, -1));
	
	lua_pop (L, 4);
    } else {
	plwind (-1, 1, -1, 1);
    }
    lua_pop(L, 1);

    plschr (0.85 * fontscale, 1);
    plwid(1);
    plcol0(255);

    lua_getfield (L, 1, "grid");
    if (lua_toboolean (L, -1)) {
	PLINT mark[1] = {250}, space[1] = {1000};

	plstyl(1, mark, space);
	plbox ("g", 0, 0, "g", 0, 0);
    }
    lua_pop(L, 1);

    lua_getfield (L, 1, "ticks");
    ticks = lua_toboolean (L, -1);
    
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	xtick = lua_tonumber (L, -1);
	lua_pop (L, 1);

	lua_rawgeti (L, -1, 2);
	nxsub = lua_tointeger (L, -1);
	lua_pop (L, 1);

	lua_rawgeti (L, -1, 3);
	ytick = lua_tonumber (L, -1);
	lua_pop (L, 1);

	lua_rawgeti (L, -1, 4);
	nysub = lua_tointeger (L, -1);
	lua_pop (L, 1);
    } else {
	xtick = 0;
	nxsub = 0;
	ytick = 0;
	nysub = 0;
    }
    lua_pop(L, 1);
    
    pllsty(1);

    lua_getfield (L, 1, "axes");
    if (lua_toboolean (L, -1)) {
	if (ticks) {
	    plbox ("ants", xtick, nxsub, "ants", ytick, nysub);
	} else {
	    plbox ("na", 0, 0, "na", 0, 0);
	}	    
    }
    lua_pop(L, 1); 

    lua_getfield (L, 1, "box");
    if (lua_toboolean (L, -1)) {
	if (ticks) {
	    plbox ("bcnts", xtick, nxsub, "bcnts", ytick, nysub);
	} else {
	    plbox ("bcn", 0, 0, "bcn", 0, 0);
	}	    
    }
    lua_pop(L, 1);

    lua_getfield (L, 1, "labels");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);
	lua_rawgeti (L, -3, 3);

	plschr (fontscale, 1);

	pllab(lua_tostring (L, -3),
	      lua_tostring (L, -2),
	      lua_tostring (L, -1));

	lua_pop(L, 1);
    }
    lua_pop(L, 1);

    /* Plot the graphs. */

    n = luaX_objlen (L, 1);

    for (i = 0 ; i < n ; i += 1) {
	lua_rawgeti (L, 1, i + 1);
	if (lua_isfunction (L, -1)) {
	    lua_pushinteger (L, i + 1);
	    lua_pcall (L, 1, 0, 0);
	} else {
	    lua_pop(L, 1);
	}
    }
    
    plend();

    lua_pushlstring (L, buffer, size);
    free(buffer);
    
    return 1;
}

static int constructplot (lua_State *L)
{
    lua_CFunction function;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, lua_upvalueindex(1), LUA_TFUNCTION);
    
    function = lua_tocfunction (L, lua_upvalueindex(1));
    lua_pushcclosure (L, function, 1);

    return 1;
}

static int line (lua_State *L)
{
    PLFLT *xs, *ys;
    PLINT n;

    prepare (L, &xs, &ys, &n);
    
    /* Plot the samples connected with lines. */    

    plline (n, xs, ys);

    free(xs);
    free(ys);

    return 0;
}

static int polygon (lua_State *L)
{
    PLFLT *xs, *ys;
    PLINT n;

    prepare (L, &xs, &ys, &n);
    
    /* Plot the samples connected with lines. */    

    plfill (n, xs, ys);

    free(xs);
    free(ys);

    return 0;
}

static int lines (lua_State *L)
{
    PLFLT *xs, *ys;
    int i, n;

    prepare (L, &xs, &ys, &n);
    
    /* Plot the samples connected with lines. */    

    for (i = 1 ; i < n ; i += 2) {
	pljoin (xs[i - 1], ys[i - 1], xs[i], ys[i]);
    }

    free(xs);
    free(ys);

    return 0;
}

static int symbols (lua_State *L)
{
    PLFLT *xs, *ys;
    PLINT symbol;
    int n;
    
    lua_getfield (L, lua_upvalueindex(1), "symbol");
    if (lua_isnumber (L, -1)) {
	symbol = lua_tointeger (L, -1);
    } else {
	symbol = 0;
    }
    lua_pop(L, 1);

    prepare (L, &xs, &ys, &n);
    
    /* Plot the symbols. */    

    plpoin (n, xs, ys, symbol);

    free(xs);
    free(ys);

    return 0;
}

static int arc (lua_State *L)
{
    PLFLT x = 0, y = 0, r = 1, rho = 1, theta = 0, phi = 360;
    
    /* Setup style and color. */
    
    lua_getfield (L, lua_upvalueindex(1), "position");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);

	x = lua_tonumber(L, -2);
	y = lua_tonumber(L, -1);

	lua_pop(L, 2);
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "radii");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);

	r = lua_tonumber(L, -2);
	rho = lua_tonumber(L, -1);

	lua_pop(L, 2);
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "radius");
    if (lua_isnumber (L, -1)) {
	r = lua_tonumber(L, -1);
	rho = lua_tonumber(L, -1);
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "range");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);

	theta = lua_tonumber(L, -2);
	phi = lua_tonumber(L, -1);

	lua_pop(L, 2);
    }
    lua_pop(L, 1);

    prepare(L, NULL, NULL, NULL);

    /* Draw an arc. */
    plarc (x, y, r, rho, theta, phi, 0);
    
    return 0;
}

static int label (lua_State *L)
{
    PLFLT x = 0, y = 0, dx = 1, dy = 0, justification = 0;
    const char *text;
    
    /* Setup style and color. */
    
    lua_getfield (L, lua_upvalueindex(1), "justification");
    if (lua_isnumber (L, -1)) {
	justification = lua_tonumber (L, -1);
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "position");
    if (lua_istable (L, -1)) {
	lua_rawgeti (L, -1, 1);
	lua_rawgeti (L, -2, 2);

	x = lua_tonumber(L, -2);
	y = lua_tonumber(L, -1);

	lua_pop(L, 2);
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "orientation");
    if (lua_isnumber (L, -1)) {
	dx = cos(lua_tonumber (L, -1) * M_PI / 180);
	dy = sin(lua_tonumber (L, -1) * M_PI / 180);
    }
    lua_pop(L, 1);
    
    lua_getfield (L, lua_upvalueindex(1), "text");
    if (lua_isstring (L, -1)) {
	text = lua_tostring (L, -1);
    } else {
	text = NULL;
    }
    lua_pop(L, 1);

    prepare(L, NULL, NULL, NULL);

    /* Draw the label. */

    if (text) {
	plptex (x, y, dx, dy, justification, text);
    }

    return 0;
}

int luaopen_plotting (lua_State *L)
{
    int i;
    
    const char *keys[] = {"line", "lines", "polygon",
			  "symbols", "label", "arc"};
    const lua_CFunction values[] = {line, lines, polygon,
				    symbols, label, arc};

    const luaL_Reg plotting[] = {
	{"cartesian", constructcartesian},
	{NULL, NULL}
    };

    luaL_register (L, "plotting", plotting);
    
    for (i = 0 ; i < sizeof(keys) / sizeof(keys[0]) ; i += 1) {
	lua_pushstring (L, keys[i]);
	lua_pushcfunction (L, values[i]);
	lua_pushcclosure (L, constructplot, 1);
	lua_settable (L, -3);
    }

    return 0;
}
