/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <ode/ode.h>

#include "body.h"
#include "joint.h"
#include "system.h"
#include "polyhedron.h"

static int addforce (lua_State *L)
{
    Body *object;
    double F[3], p[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    F[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}

	if (lua_istable (L, 3)) {
	    for(i = 0; i < 3; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		p[i] = lua_tonumber (L, -1);

		lua_pop(L, 1);
	    }
	
	    dBodyAddForceAtPos ([object body],
				F[0], F[1], F[2],
				p[0], p[1], p[2]);
	} else {
	    dBodyAddForce ([object body], F[0], F[1], F[2]);
	}

	dBodyEnable([object body]);
    }

    return 0;
}

static int addrelativeforce (lua_State *L)
{
    Body *object;
    double F[3], p[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    F[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}

	if (lua_istable (L, 3)) {
	    for(i = 0; i < 3; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		p[i] = lua_tonumber (L, -1);

		lua_pop(L, 1);
	    }
	
	    dBodyAddRelForceAtRelPos ([object body],
				      F[0], F[1], F[2],
				      p[0], p[1], p[2]);
	} else {
	    dBodyAddRelForce ([object body], F[0], F[1], F[2]);
	}

	dBodyEnable([object body]);
    }

    return 0;
}

static int addtorque (lua_State *L)
{
    Body *object;
    double T[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    T[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}
	
	dBodyAddTorque ([object body], T[0], T[1], T[2]);
	dBodyEnable([object body]);
    }

    return 0;
}

static int addrelativetorque (lua_State *L)
{
    Body *object;
    double T[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    T[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}
	
	dBodyAddRelTorque ([object body], T[0], T[1], T[2]);
	dBodyEnable([object body]);
    }

    return 0;
}

static void pushmass (lua_State *L, dMass *mass)
{
    int i, j;
    
    lua_newtable(L);

    lua_pushnumber(L, mass->mass);
    lua_rawseti(L, -2, 1);

    lua_newtable(L);

    for(i = 0 ; i < 3 ; i += 1) {
	lua_pushnumber(L, mass->c[i]);
	lua_rawseti(L, -2, i + 1);
    }

    lua_rawseti(L, -2, 2);

    lua_newtable(L);

    for(j = 0 ; j < 3 ; j += 1) {
	for(i = 0 ; i < 3 ; i += 1) {
	    lua_pushnumber(L, mass->I[j * 4 + i]);
	    lua_rawseti(L, -2, j * 3 + i + 1);
	}
    }

    lua_rawseti(L, -2, 3);
}

static void tomass (lua_State *L, int t, dMass *mass)
{    
    int i, j;

    dMassSetZero(mass);
	
    if(lua_istable (_L, t)) {
	lua_rawgeti (_L, t, 1);
	mass->mass = lua_tonumber (_L, -1);
	lua_pop (_L, 1);
	
	lua_rawgeti (_L, t, 2);

	for(i = 0 ; i < 3 ; i += 1) {
	    lua_rawgeti (_L, -1, i + 1);
		
	    mass->c[i] = lua_tonumber (_L, -1);
		
	    lua_pop (_L, 1);
	}
	
	lua_pop (_L, 1);

	lua_rawgeti (_L, t, 3);
	
	for(i = 0 ; i < 3 ; i += 1) {
	    for(j = 0 ; j < 3 ; j += 1) {
		lua_rawgeti (_L, -1, i * 3 + j + 1);

		mass->I[i * 4 + j] = lua_tonumber (_L, -1);

		lua_pop (_L, 1);
	    }
	}

	lua_pop (_L, 1);
    }
}

static int adjustmass (lua_State *L)
{
    dMass mass;
    dReal m;

    m = (dReal)luaL_checknumber (L, 2);

    tomass (L, 1, &mass);
    dMassAdjust (&mass, m);
    pushmass (L, &mass);

    return 1;
}

static int translatemass (lua_State *L)
{
    dMass mass;
    dReal r[3];
    int i;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TTABLE);

    for(i = 0; i < 3 ; i += 1) {
	lua_rawgeti(L, 2, i + 1);
	r[i] = lua_tonumber (L, -1);
	lua_pop(L, 1);
    }

    tomass (L, 1, &mass);
    dMassTranslate (&mass, r[0], r[1], r[2]);
    pushmass (L, &mass);

    return 1;
}

static int rotatemass (lua_State *L)
{
    dMass mass;
    dMatrix3 R;
    int i, j;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TTABLE);

    for(j = 0; j < 3 ; j += 1) {
	for(i = 0; i < 3 ; i += 1) {
	    lua_rawgeti(L, 2, 3 * j + i + 1);
	    R[4 * j + i] = lua_tonumber (L, -1);
	    lua_pop(L, 1);
	}
    }

    tomass (L, 1, &mass);
    dMassRotate (&mass, R);
    pushmass (L, &mass);

    return 1;
}

static int spheremass (lua_State *L)
{
    dMass mass;
    dReal rho, r;

    rho = (dReal)luaL_checknumber (L, 1);
    r = (dReal)luaL_checknumber (L, 2);

    dMassSetSphere (&mass, rho, r);
    pushmass (L, &mass);

    return 1;
}

static int boxmass (lua_State *L)
{
    dMass mass;
    dReal rho, a, b, c;

    rho = (dReal)luaL_checknumber (L, 1);
    a = (dReal)luaL_checknumber (L, 2);
    b = (dReal)luaL_checknumber (L, 3);
    c = (dReal)luaL_checknumber (L, 4);

    dMassSetBox (&mass, rho, a, b, c);
    pushmass (L, &mass);

    return 1;
}

static int capsulemass (lua_State *L)
{
    dMass mass;
    dReal rho, l, r;

    rho = (dReal)luaL_checknumber (L, 1);
    r = (dReal)luaL_checknumber (L, 2);
    l = (dReal)luaL_checknumber (L, 3);

    dMassSetCapsule (&mass, rho, 3, r, l);
    
    pushmass (L, &mass);

    return 1;
}

static int polyhedronmass (lua_State *L)
{
    id polyhedron;
    dGeomID geom;
    dMass mass;
    dReal rho;

    polyhedron = *(id *)lua_touserdata (L, 1);
    rho = (dReal)luaL_checknumber (L, 2);
    
    geom = dCreateTriMesh (NULL, [polyhedron data], NULL, NULL, NULL);
    dMassSetTrimesh (&mass, rho, geom);
    dGeomDestroy (geom);

    printf ("center of mass is at (%f, %f, %f)\n", mass.c[0], mass.c[1], mass.c[2]);
    
    mass.c[0] = 0;
    mass.c[1] = 0;
    mass.c[2] = 0;

    pushmass (L, &mass);

    return 1;
}

static int spring (lua_State *L)
{
    dReal k_d, k_s, h;

    k_s = (dReal)luaL_checknumber (L, 1);
    k_d = (dReal)luaL_checknumber (L, 2);

    lua_getglobal (L, "dynamics");
    lua_getfield (L, -1, "stepsize");
    
    h = (dReal)lua_tonumber (L, -1);

    lua_newtable (L);
    lua_pushnumber (L, 1.0 / (h * k_s + k_d));
    lua_rawseti (L, -2, 1);
    lua_pushnumber (L, h * k_s / (h * k_s + k_d));
    lua_rawseti (L, -2, 2);

    return 1;
}

static int contact (lua_State *L)
{
    dContact contact;
    dJointID j;
    Body *a, *b;
    dReal mu, depth, epsilon, sigma, tau;
    dVector3 pos, normal;
    int i, simple;

    simple = lua_toboolean (L, lua_upvalueindex (1));

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TUSERDATA);
    luaL_checktype (L, 3, LUA_TTABLE);
    luaL_checktype (L, 4, LUA_TTABLE);

    depth = (dReal)luaL_checknumber (L, 5);
    mu = (dReal)luaL_optnumber (L, 6, 0);
    epsilon = (dReal)luaL_optnumber (L, 7, 0);
    sigma = (dReal)luaL_optnumber (L, 8, 0);
    tau = (dReal)luaL_optnumber (L, 9, 1);

    for(i = 0; i < 3; i += 1) {
	lua_rawgeti(L, 3, i + 1);
	pos[i] = lua_tonumber (L, -1);

	lua_rawgeti(L, 4, i + 1);
	normal[i] = lua_tonumber (L, -1);
	
	lua_pop(L, 2);
    }

    a = *(Body **)lua_touserdata (L, 1);
    b = *(Body **)lua_touserdata (L, 2);

    /* Set surface parameters. */
    
    contact.surface.mode = 0;
		    
    if (mu > 0) {
	if (!simple) {
	    contact.surface.mode |= dContactApprox1;
	}
	
	contact.surface.mode |= dContactFDir1;
	contact.surface.mu = mu;
    }

    if (epsilon > 0) {
	contact.surface.mode |= dContactBounce;
	contact.surface.bounce = epsilon;
	contact.surface.bounce_vel = 0.01;
    }

    if (sigma > 0) {
	contact.surface.mode |= dContactSoftCFM;
	contact.surface.soft_cfm = sigma;
    }

    if (tau < 1) {
	contact.surface.mode |= dContactSoftERP;
	contact.surface.soft_erp = tau;
    }

    for (i = 0 ; i < 3; i += 1) {
	contact.geom.pos[i] = pos[i];
	contact.geom.normal[i] = normal[i];
    }

    dSafeNormalize3 (contact.geom.normal);

    {
	dBodyID p, q;
	dVector3 u, v, delta;
	dReal *r, *n, ndotdelta;

	r = pos;
	n = normal;
	p = [a body];
	q = [b body];
		
	if (p) {
	    dBodyGetPointVel (p, r[0], r[1], r[2], u);
	} else {
	    dSetZero (u, 3);
	}
		
	if (q) {
	    dBodyGetPointVel (q, r[0], r[1], r[2], v);
	} else {
	    dSetZero (v, 3);
	}	

	dOP (delta, -, u, v);		
	ndotdelta = dDOT(n, delta);

	/* Only consider this contact if the
	   relative velocity points inward. */

	if (simple || ndotdelta <= 0) {
	    contact.fdir1[0] = delta[0] - ndotdelta * n[0];
	    contact.fdir1[1] = delta[1] - ndotdelta * n[1];
	    contact.fdir1[2] = delta[2] - ndotdelta * n[2];

	    dSafeNormalize3(contact.fdir1);
    
	    contact.geom.depth = depth;
	    contact.geom.g1 = [a geom];
	    contact.geom.g2 = [b geom];

	    j = dJointCreateContact (_WORLD, _GROUP, &contact);

	    dJointAttach (j, [a body], [b body]);
	}
    }

    return 0;
}

static int joined (lua_State *L)
{
    Body *a, *b;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TUSERDATA);

    a = *(Body **)lua_touserdata (L, 1);
    b = *(Body **)lua_touserdata (L, 2);

    if ([a body] && [b body]) {
	lua_pushboolean (L, dAreConnected([a body], [b body]));
    } else {
	lua_pushboolean (L, 0);
    }
    
    return 1;
}

static int pointfrombody(lua_State *L)
{
    id object;
    dBodyID body;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    body = [object body];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetRelPointPos (body, b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int vectorfrombody(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyVectorToWorld ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int pointtobody(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetPosRelPoint ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int vectortobody(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyVectorFromWorld ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int pointvelocity(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetPointVel ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int anaesthetize (lua_State *L)
{
    id object;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    
    object = *(id*)lua_touserdata (L, 1);
    if ([object body]) {
	dBodyDisable ([object body]);
    }
    
    return 0;
}

static int wake (lua_State *L)
{
    id object;

    luaL_checktype (L, 1, LUA_TUSERDATA);

    object = *(id*)lua_touserdata (L, 1);
    if ([object body]) {
	dBodyEnable ([object body]);
    }
    
    return 0;
}

static int reattach (lua_State *L)
{
    id object;

    luaL_checktype (L, 1, LUA_TUSERDATA);

    object = *(id *)lua_touserdata (L, 1);
    [object update];
    
    /* [object toggle]; */
    /* [object toggle]; */
    
    return 0;
}

int luaopen_physics (lua_State *L)
{
    const luaL_Reg physics[] = {
	{"sleep", anaesthetize},
	{"wake", wake},
	{"addforce", addforce},
	{"addtorque", addtorque},
	{"addrelativeforce", addrelativeforce},
	{"addrelativetorque", addrelativetorque},
	{"adjustmass", adjustmass},
	{"translatemass", translatemass},
	{"rotatemass", rotatemass},
	{"spheremass", spheremass},
	{"boxmass", boxmass},
	{"capsulemass", capsulemass},
	{"polyhedronmass", polyhedronmass},
	{"spring", spring},
	{"joined", joined},
	{"pointfrombody", pointfrombody},
	{"pointtobody", pointtobody},
	{"vectorfrombody", vectorfrombody},
	{"vectortobody", vectortobody},
	{"pointvelocity", pointvelocity},
	{"reattach", reattach},
	{NULL, NULL}
    };

    luaL_register (L, "physics", physics);

    lua_pushstring (L, "addcontact");
    lua_pushboolean (L, 0);
    lua_pushcclosure (L, contact, 1);
    lua_settable (L, -3);

    lua_pushstring (L, "addsimplecontact");
    lua_pushboolean (L, 1);
    lua_pushcclosure (L, contact, 1);
    lua_settable (L, -3);

    return 0;
}
