/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <stdarg.h>
#include <ctype.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <microhttpd.h>

#include "network.h"
#include "peer.h"

struct context {
    char *data;
    size_t size;
};    

static int connections, pages, mime, port, block;
static struct MHD_Daemon *http;

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static const char *reader (lua_State *L, void *data, size_t *size)
{
    struct context *c = data;

    if (c->size > 0) {
	lua_pushstring (L, "return function (query, post)");
	lua_pushlstring (L, c->data, c->size);
	lua_pushstring (L, "end");
	lua_concat (L, 3);

	c->data = strdup (lua_tostring (L, -1));
	c->size = 0;

	*size = lua_strlen (L, -1);

	lua_pop (L, 1);
	
	return c->data;
    } else {
	free (c->data);
	return NULL;
    }
}

static int iterate_post  (void *cls,
			  enum MHD_ValueKind kind,
			  const char *key,
			  const char *filename,
			  const char *content_type,
			  const char *transfer_encoding,
			  const char *data,
			  uint64_t off,
			  size_t size)
{
    lua_pushstring (_L, key);
    lua_pushstring (_L, key);
    lua_gettable (_L, -3);
    
    if (off == 0) {
	if (!lua_isnil (_L, -1)) {
	    /* If a value is already there, check whether this is
	       a group table and if not create one and put it in. */
	    
	    if (lua_istable (_L, -1) && lua_getmetatable (_L, -1)) {
		lua_pop (_L, 1);
	    } else {
		lua_newtable (_L);
		lua_newtable (_L);
		lua_setmetatable (_L, -2);
		
		lua_insert (_L, -2);
		lua_rawseti (_L, -2, 1);
	    }

	    lua_pushnil (_L);
	}
    } else {
	if (lua_istable (_L, -1) && lua_getmetatable (_L, -1)) {
	    int n;

	    lua_pop (_L, 1);
	    n = lua_objlen (_L, -1);
	    
	    /* We need to continue a grouped value, so
	       remove it from the group table. */

	    lua_rawgeti (_L, -1, n);
	    
	    lua_pushnil (_L);
	    lua_rawseti (_L, -3, n);
	}
    }
    
    /* If it's a file upload the value is a table
       {filename, mime type, data}, otherwise it's
       just the data. */
    
    if (filename) {
	/* If no vaule's there already create one
	   from scratch, otherwise fill it in. */
	
	if (!lua_istable (_L, -1)) {
	    lua_pop (_L, 1);
	    lua_newtable (_L);

	    lua_pushstring (_L, filename);
	    lua_rawseti (_L, -2, 1);

	    lua_pushstring (_L, content_type);
	    lua_rawseti (_L, -2, 2);

	    lua_pushlstring (_L, data, size);
	    lua_rawseti (_L, -2, 3);

	    printf ("  %s = %s/%s\n",
		    key, filename, content_type);
	} else {
	    lua_rawgeti (_L, -1, 3);
	    lua_pushlstring (_L, data, size);
	    lua_concat (_L, 2);
	    lua_rawseti (_L, -2, 3);
	}

	/* Insert the value into the group table if needed. */
	
	if (lua_istable (_L, -2)) {
	    lua_rawseti (_L, -2, lua_objlen (_L, -2) + 1);
	}
    } else {
	/* Process the new value. */
	
	if (lua_isstring (_L, -1)) {
	    lua_pushlstring (_L, data, size);
	    lua_concat (_L, 2);
	} else if (lua_isnil (_L, -1)) {
	    lua_pop (_L, 1);
	    lua_pushlstring (_L, data, size);
	}

	if (off == 0) {
	    if (lua_strlen (_L, -1) > 40) {
		printf ("  %s = %.40s...\n", key, lua_tostring (_L, -1));
	    } else {
		printf ("  %s = %s\n", key, lua_tostring (_L, -1));
	    }
	}

	if (lua_istable (_L, -2)) {
	    lua_rawseti (_L, -2, lua_objlen (_L, -2) + 1);
	}
    }
			
    lua_settable (_L, -3);    
	
    /* printf ("%s, %s, %s, %s, %lu, %zd, %p\n", key, filename, content_type, transfer_encoding, off, size, data); */

    return MHD_YES;
}

static int iterate_get (void *cls, enum MHD_ValueKind kind,
			const char *key, const char *value)
{
    printf ("  %s = %s\n", key, value);
    
    lua_pushstring (_L, key);
    lua_pushstring (_L, key);
    lua_gettable (_L, -3);

    if (lua_isnil (_L, -1)) {
	lua_pop(_L, 1);
	lua_pushstring (_L, value);
    } else if (lua_istable (_L, -1)) {
	lua_pushstring (_L, value);
	lua_rawseti (_L, -2, lua_objlen (_L, -2) + 1);
    } else {
	assert (lua_isstring (_L, -1));
	
	lua_newtable (_L);
	lua_insert (_L, -2);
	lua_rawseti (_L, -2, 1);
	lua_pushstring (_L, value);
	lua_rawseti (_L, -2, 2);
    }
			
    lua_settable (_L, -3);

    return MHD_YES;
}

static int respond (void *cls,
		    struct MHD_Connection *connection,
		    const char *url, const char *method, const char *version,
		    const char *upload_data, size_t *upload_data_size,
		    void **con_cls)
{
    if (*con_cls == NULL) {
	printf ("Received a %s request for URL %s.\n", method, url);
    
    	*con_cls = calloc (1, sizeof (struct context));
	connections += 1;

    	return MHD_YES;
    }

    /* Aggregate the upload data into a single buffer. */
    
    if (*upload_data_size > 0) {
	struct context *c = (struct context *)*con_cls;
	
    	c->data = realloc (c->data, c->size + *upload_data_size);
    	memcpy (c->data + c->size, upload_data, *upload_data_size);
	
    	c->size += *upload_data_size;
    	*upload_data_size = 0;

    	return MHD_YES;
    }
    
    if (!strcmp (method, "HEAD") || !strcmp (method, "GET") || !strcmp (method, "POST")) {
	struct context *c = (struct context *)*con_cls;
	int h_0;

	h_0 = lua_gettop (_L);

	/* Parse the query and post data if any and
	   gather all fields inside a table. */
	
	lua_newtable (_L);

	/* Parse the query pairs. */
	    
	MHD_get_connection_values (connection,
				   MHD_GET_ARGUMENT_KIND,
				   iterate_get,
				   NULL);
	    
	/* And potentially pre-parsed post data pairs. */
	    
	MHD_get_connection_values (connection,
				   MHD_POSTDATA_KIND,
				   iterate_get,
				   NULL);

	/* And finally the post data buffer. */
	    
	if (c->size > 0) {
	    struct MHD_PostProcessor *post;
	    post = MHD_create_post_processor(connection,
					     512,
					     iterate_post, NULL);

	    /* If the data is parseable put it into the
	       query table otherwise feed the whole chunk
	       as-is via the post parameter. */
		
	    if (post &&
		MHD_post_process(post, c->data, c->size) == MHD_NO) {
		lua_pushlstring (_L, c->data, c->size);
	    }

	    if (post) {
		MHD_destroy_post_processor (post);
	    }
	}

	/* Push nil if no POST data was there. */
	    
	if (lua_type (_L, -1) != LUA_TSTRING) {
	    lua_pushnil (_L);
	}

	/* Retrieve the MIME type and content for the page. */
			    
	lua_rawgeti (_L, LUA_REGISTRYINDEX, mime);

	if (lua_istable (_L, -1)) {
	    lua_pushstring (_L, url);
	    lua_gettable (_L, -2);
	    lua_replace (_L, -2);
	}
			    
	lua_rawgeti (_L, LUA_REGISTRYINDEX, pages);

	if (lua_istable (_L, -1)) {
	    lua_pushstring (_L, url);
	    lua_gettable (_L, -2);
	    lua_replace (_L, -2);

	    if (lua_isfunction (_L, -1)) {
		lua_pushstring (_L, method);
		lua_pushstring (_L, url);
		lua_pushstring (_L, version);
    
		lua_pcall(_L, 3, 1, 0);

		if (lua_type (_L, -1) != LUA_TSTRING) {
		    lua_settop (_L, h_0);
		    return MHD_YES;
		}
	    }
	}

	/* Return the content of the requested page
	   after filtering it through the Lua interpreter. */
			    
	if (lua_isstring (_L, -1)) {
	    struct MHD_Response *response;
	    int result;
	    char *s;

	    /* Find and substitute <?lua ... ?> chunks. */

	    for (s = strstr(lua_tostring(_L, -1), "<?lua");
		 s;
		 s = strstr(lua_tostring(_L, -1), "<?lua")) {
		struct context chunk;
		const char *e, *t;
		int l;
		
		/* printf ("** %d\n", lua_gettop(_L)); */

		for (e = lua_tostring(_L, -1), l = 1;
		     e < s;
		     l += (*e == '\n'),e += 1);

		lua_pushfstring (_L, "=[%s: %d]", url, l);

		for (e = strstr(s, "?>"), t = strstr(s + 2, "<?");
		     e && t && e > t;
		     t = strstr(e + 2, "<?"), e = strstr(e + 2, "?>"));

		if (e != NULL) {
		    chunk.data = s + 5;
		    chunk.size = e - s - 5;

		    if (lua_load (_L, reader, (void *)&chunk, lua_tostring (_L, -1)) ||
			lua_pcall(_L, 0, 1, 0)) {
			puts (lua_tostring (_L, -1));
		    } else {
			/* Now call the function with the specified
			   values. */

			lua_pushvalue(_L, -6);
			lua_pushvalue(_L, -6);
			
			if (lua_pcall(_L, 2, 1, 0)) {
			    puts (lua_tostring (_L, -1));
			}
		    }

		    /* Push an empty string if nothing was returned. */
    
		    if (!lua_isstring (_L, -1)) {
			lua_pop (_L, 1);
			lua_pushstring (_L, "");
		    }
		} else {
		    e = s + strlen (s) - 2;

		    lua_pushfstring (_L,
				     "[%s: %d]: unterminated Lua code tag",
				     url, l);
		    
		    puts (lua_tostring (_L, -1));
		}

		/* Pop the chunk descriptor off the stack. */
		
		lua_replace (_L, -2);
		
		/* Substitute the result into the content string. */
		
		lua_pushlstring (_L, e + 2, s + strlen (s) - e - 2);
		lua_pushlstring (_L,
				 lua_tostring (_L, -3),
				 s - lua_tostring (_L, -3));
		
		lua_replace (_L, -4);
		lua_concat (_L, 3);
		
		/* printf ("** %d\n", lua_gettop(_L)); */
	    }
	    
	    /* Create and queue the response. */

	    response = MHD_create_response_from_data (luaX_objlen (_L, -1),
						      (void *)lua_tostring (_L, -1),
						      MHD_NO, MHD_YES);

	    MHD_add_response_header (response, "Server", "Techne/" VERSION);
	    MHD_add_response_header (response, "Cache-Control", "no-store");
	    /* MHD_add_response_header (response, "Connection", "close"); */
	    MHD_add_response_header (response, "Content-Type",
				     lua_tostring (_L, -2) ?
				     lua_tostring (_L, -2) : "text/html");
	    
	    result = MHD_queue_response (connection, MHD_HTTP_OK, response);
	    MHD_destroy_response (response);

	    return result;
	} else {
	    struct MHD_Response *response;
	    int result;

	    const char *content=
		"<html><body>\r\n"
		"<h2>Not Found</h2>\r\n"
		"The requested page could not be found.\r\n"
		"</body></html>\r\n";

	    response = MHD_create_response_from_data (strlen(content),
						      (void *)content,
						      MHD_NO, MHD_NO);

	    MHD_add_response_header (response, "Server", "Techne/" VERSION);
	    /* MHD_add_response_header (response, "Connection", "close"); */
	    MHD_add_response_header (response, "Content-Type", "text/html");
	    
	    result = MHD_queue_response (connection,
					 MHD_HTTP_NOT_FOUND,
					 response);
	    MHD_destroy_response (response);

	    return result;
	}

	lua_settop (_L, h_0);
    } else {
	struct MHD_Response *response;
	int result;

	const char *content =
	    "<html><body>\r\n"
	    "<h2>Not Implemented</h2>\r\n"
	    "The request is not implemeted by this server.\r\n"
	    "</body></html>\r\n";

	response = MHD_create_response_from_data (strlen(content),
						  (void *)content,
						  MHD_NO, MHD_NO);

	MHD_add_response_header (response, "Server", "Techne/" VERSION);
	/* MHD_add_response_header (response, "Connection", "close"); */
	MHD_add_response_header (response, "Content-Type", "text/html");
	    
	result = MHD_queue_response (connection,
				     MHD_HTTP_NOT_IMPLEMENTED,
				     response);
	MHD_destroy_response (response);

	return result;
    }
}

static void finish (void *cls, struct MHD_Connection *connection,
		    void **con_cls, enum MHD_RequestTerminationCode toe)
{
    struct context *c = (struct context *)*con_cls;

    /* Free any allocated context data. */
    
    if (c) {
	if (c->data) {
	    free (c->data);
	}

	free (c);
    }

    connections -= 1;
}

static void run()
{
    if (http) {
	fd_set read, write, except;
	int n = 0;

	assert (connections >= 0);

	/* Wait for traffic to arrive if in blocking mode. */

	if (block && connections == 0) {
	    FD_ZERO(&read);
	    FD_ZERO(&write);
	    FD_ZERO(&except);
      
	    MHD_get_fdset(http, &read, &write, &except, &n);
	    select (n + 1, &read, &write, &except, NULL);
	}
    
	/* Check for http requests. */

	MHD_run (http);
    }
}

static int generic_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [object name]);
   
    return 1;
}

static int generic_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object get];

    return 1;
}

static int generic_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object set];

    return 0;
}

static int generic_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object free];

    return 0;
}

static int dummy_index(lua_State *L)
{
    lua_pushnil (L);
    
    return 1;
}

static int dummy_newindex(lua_State *L)
{
    return 0;
}

static int peer_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [object name]);
    lua_pushstring(L, " foundry");
    lua_concat(L, 2);
    
    return 1;
}

static int peer_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object freeSocket];

    return 0;
}

static int peer_call (lua_State *L)
{
    id parent, child;

    /* Create the userdata... */
	
    parent = *(id *)lua_touserdata(L, 1);
    child = [[parent copy] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = child;
    
    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, generic_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, child);
    lua_pushvalue (L, -3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int constructnode (lua_State *L)
{
    Class class;
    id object;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    /* Create the userdata... */

    object = [[class alloc] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)generic_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    /* Add a weak reference. */
    
    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    return 1;
}

static int network_index (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!xstrcmp(k, "http")) {
	if (http) {
	    lua_pushnumber(L, port);
	} else {
	    lua_pushnil (L);
	}
    } else if (!xstrcmp(k, "block")) {
	lua_pushboolean (L, block);
    } else if (!xstrcmp(k, "pages")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, pages);
    } else if (!xstrcmp(k, "mime")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, mime);
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int network_newindex (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!xstrcmp(k, "http")) {
	/* Close the current web server. */

	if (http) {
	    MHD_stop_daemon (http);
	}

	/* And set up a new instance listening to the
	   specified port. */
	
	port = lua_tonumber (L, 3);

	if (port > 0) {
	    http = MHD_start_daemon (MHD_USE_DEBUG,
				     port,
				     NULL, NULL, &respond, NULL,
				     MHD_OPTION_NOTIFY_COMPLETED,
				     finish, NULL, MHD_OPTION_END);
    	}

	if (http) {
	    printf ("Listening for HTTP requests on port %d.\n", port);
	} else {
	    printf ("Could not set up port %d to listen "
		    "for HTTP requests.\n", port);
	}
    } else if (!xstrcmp(k, "block")) {
	block = lua_toboolean (L, 3);
    } else if (!xstrcmp(k, "pages")) {
	luaL_unref (L, LUA_REGISTRYINDEX, pages);
	pages = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "mime")) {
	luaL_unref (L, LUA_REGISTRYINDEX, mime);
	mime = luaL_ref (L, LUA_REGISTRYINDEX);
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

static int constructpeer(lua_State *L)
{
    struct addrinfo *info;
    const struct addrinfo hints = {
	0,
	AF_INET,
	SOCK_DGRAM,
	0, 0, NULL, NULL, NULL
    };
    int file, flags, bound, resolved;
	
    id object;
    Class class;
    
    luaL_checktype(L, 1, LUA_TSTRING);
    luaL_checktype(L, 2, LUA_TNUMBER);

    /* Resolve the hostname. */

    resolved = getaddrinfo (lua_tostring (L, 1),
			    lua_tostring (L, 2),
			    &hints, &info);

    if (resolved == 0) {
	printf ("Resolved %s:%s to %s:%d.\n",
		lua_tostring (L, 1),
		lua_tostring (L, 2),
		inet_ntoa(((struct sockaddr_in *)info->ai_addr)->sin_addr),
		ntohs(((struct sockaddr_in *)info->ai_addr)->sin_port));
    } else {
	printf ("Could not resolve hostname %s (%s).\n",
		lua_tostring (L, 1), gai_strerror(resolved));
    }
    
    /* Create a new datagram socket. */
	    
    file = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	    
    if (file < 0) {
	printf ("Could not open a socket for outgoing peer traffic. (%s).\n",
		strerror (errno));
   }
    
    /* And bind it.  */

    bound = bind(file, info->ai_addr, sizeof (struct sockaddr));

    if (bound < 0) {
	printf ("Could not bind a socket for incoming peer traffic (%s).\n",
		strerror (errno));
    }
    
    flags = fcntl(file, F_GETFL, 0);
    fcntl(file, F_SETFL, flags | O_NONBLOCK);

    if (bound >=0) {
	printf ("Bound a socket to local interface %s at port %d.\n",
		inet_ntoa(((struct sockaddr_in *)info->ai_addr)->sin_addr),
		ntohs(((struct sockaddr_in *)info->ai_addr)->sin_port));

    
	/* Create and initialize the peer userdata. */

	lua_pushvalue (L, lua_upvalueindex (1));
	class = (Class)lua_touserdata(L, -1);
	lua_pop(L, 1);

	object = [[class alloc] initWithSocket: file];
    
	*(id *)lua_newuserdata(L, sizeof(id)) = object;

	lua_newtable(L);
	lua_pushstring(L, "__call");
	lua_pushcfunction(L, peer_call);
	lua_settable(L, -3);
	lua_pushstring(L, "__tostring");
	lua_pushcfunction(L, peer_tostring);
	lua_settable(L, -3);
	lua_pushstring(L, "__gc");
	lua_pushcfunction(L, peer_gc);
	lua_settable(L, -3);
	lua_pushstring(L, "__index");
	lua_pushcfunction(L, dummy_index);
	lua_settable(L, -3);
	lua_pushstring(L, "__newindex");
	lua_pushcfunction(L, dummy_newindex);
	lua_settable(L, -3);
	lua_setmetatable(L, -2);

	lua_pushstring (L, "userdata");
	lua_gettable (L, LUA_REGISTRYINDEX);
	lua_pushlightuserdata (L, object);
	lua_pushvalue (L, -3);
	lua_settable (L, -3);
	lua_pop(L, 1);
    } else {
	printf ("Could not bind a socket to local "
		"interface %s at port %d (%s).\n",
		inet_ntoa(((struct sockaddr_in *)info->ai_addr)->sin_addr),
		ntohs(((struct sockaddr_in *)info->ai_addr)->sin_port),
		strerror (errno));

	close (file);
	lua_pushnil (L);
    }

    freeaddrinfo (info);
    
    return 1;
}

@implementation Network

-(void) begin
{
    run();

    [super begin];
}

-(void) finish
{
    run();

    [super finish];
}

-(void) stepBy: (double)h at: (double)t
{
    run();
    
    [super stepBy: h at: t];
}

@end

int luaopen_network (lua_State *L)
{
    Class nodes[] = {
    };	

    id object;
    struct sigaction ignore;
    int i;

    /* Install a handler to ignore broken pipes. */
    
    ignore.sa_handler = SIG_IGN;
    ignore.sa_flags = 0;
    ignore.sa_restorer = NULL;

    sigemptyset(&ignore.sa_mask);
    sigaction(SIGPIPE, &ignore, NULL);

    /* Create the page and mimetype tables. */
    
    lua_newtable (_L);
    pages = luaL_ref (_L, LUA_REGISTRYINDEX);
    
    lua_newtable (_L);
    mime = luaL_ref (_L, LUA_REGISTRYINDEX);

    /* Create and link a network node. */

    object = [[Network alloc] init];

    lua_getglobal (L, "graph");
    lua_pushlightuserdata (L, object);

    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)generic_newindex);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    /* Add a weak reference. */
    
    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, -3);
    lua_settable (L, -3);
    lua_pop(L, 1);
 
    lua_settable (L, -3);
    lua_pop (L, 1);

    /* Create the network table. */

    {
	char *list[] = {"block", "http", "mime", "pages"};
    
	lua_newtable (_L);    
	lua_newtable (_L);

	lua_pushinteger (_L, sizeof (list) / sizeof (char *));
	lua_setfield (_L, -2, "__bloat");

	lua_pushlightuserdata (_L, list);
	lua_setfield (_L, -2, "__properties");
 
	lua_pushcfunction (_L, network_index);
	lua_setfield (_L, -2, "__index");

	lua_pushcfunction (_L, network_newindex);
	lua_setfield (_L, -2, "__newindex");

	lua_setmetatable (_L, -2);
    
	lua_setglobal (_L, "network");
    }

    /* Create the library table. */
    
    lua_newtable (L);

    lua_pushlightuserdata (L, [Peer class]);
    lua_pushcclosure (L, constructpeer, 1);
    lua_setfield(L, -2, "peer");
    
    for (i = 0 ; i < sizeof(nodes) / sizeof(nodes[0]) ; i += 1) {
	lua_pushlightuserdata (L, nodes[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([nodes[i] name])));
    }

    lua_setglobal (L, "transmission");
    
    return 0;
}
