/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <getopt.h>
#include <setjmp.h>
#include <signal.h>
#include <unistd.h>

#include <ode/ode.h>

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include <AL/al.h>
#include <AL/alc.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include <gdk/gdk.h>
#include <gdk/gdkgl.h>

#include "transform.h"
#include "body.h"
#include "joint.h"

static id root;

static jmp_buf env;

static ALCdevice *device;

static int iterate = 1, frames, width, height, decorate;
static int focus = LUA_REFNIL, defocus = LUA_REFNIL, delete = LUA_REFNIL;
static float canvas[3];
static char *title, *phase;
static const char *devicename;

static double now, then, intervals[21], userinterval, interval = -1;
static struct timespec zero;

static double timescale = 1, stepsize = 0.01, ceiling = 1.0 / 0.0;
static int hide = 1, cursor = 1, iterations = 0, engineering, mute;
static int collision = LUA_REFNIL;

static void callback (void *data, dGeomID a, dGeomID b)
{
    dContactGeom geoms[16];
    dContact contacts[16];
    dReal mu = 0, epsilon = 0, sigma = 0, tau = 1;
    int i, j, n;
    
    if (dGeomIsSpace (a) || dGeomIsSpace (b)) {
	/* Recurse into subspaces as needed. */
	
	dSpaceCollide2 (a, b, data, &callback);

	if (dGeomIsSpace (a)) {
	    dSpaceCollide ((dSpaceID)a, data, &callback);
	}
      
	if (dGeomIsSpace (b)) {
	    dSpaceCollide ((dSpaceID)b, data, &callback);
	}
    } else {
	if (dGeomGetBody (a) || dGeomGetBody (b)) {	    
	    n = dCollide (a, b, 16, geoms, sizeof(dContactGeom));

	    for (i = 0, j = 0 ; i < n ; i += 1) {
		dBodyID p, q;
		dVector3 u, v, delta;
		dReal *r, *n, ndotdelta;

		/* Caclualte the relative velocity of the
		   contact point. */
		
		r = geoms[i].pos;
		n = geoms[i].normal;
		p = dGeomGetBody (geoms[i].g1);
		q = dGeomGetBody (geoms[i].g2);
		
		if (p) {
		    dBodyGetPointVel (p, r[0], r[1], r[2], u);
		} else {
		    dSetZero (u, 3);
		}
		
		if (q) {
		    dBodyGetPointVel (q, r[0], r[1], r[2], v);
		} else {
		    dSetZero (v, 3);
		}	

		dOP (delta, -, u, v);		
		ndotdelta = dDOT(n, delta);
		
		/* if (dGeomGetClass (b) == dCapsuleClass || */
		/*     dGeomGetClass (a) == dCapsuleClass) { */
		/*     printf ("(%f, %f, %f), (%f, %f, %f)\n", */
		/* 	    r[0], r[1], r[2], n[0], n[1], n[2]); */
		/* } */

		/* Only consider this contact if the
		   relative velocity points inward. */

		if (ndotdelta <= 0) {
		    contacts[j].geom = geoms[i];

		    contacts[j].fdir1[0] = delta[0] - ndotdelta * n[0];
		    contacts[j].fdir1[1] = delta[1] - ndotdelta * n[1];
		    contacts[j].fdir1[2] = delta[2] - ndotdelta * n[2];

		    dSafeNormalize3(contacts[j].fdir1);
		
		    j += 1;
		}
	    }

	    if (j > 0) {
		/* Create a collision event. */

		if (collision != LUA_REFNIL) {
		    int h_0, h_1;
		    
		    h_0 = lua_gettop (_L);
		    
		    lua_getfield (_L, LUA_REGISTRYINDEX, "userdata");
		    
		    lua_pushlightuserdata (_L, dGeomGetData(a));
		    lua_gettable (_L, -2);

		    lua_pushlightuserdata (_L, dGeomGetData(b));
		    lua_gettable (_L, -3);

		    lua_rawgeti (_L, LUA_REGISTRYINDEX, collision);

		    /* If it's a single function encapsulate it
		       in a table to unify the approach. */

		    if (!lua_istable (_L, -1)) {
			lua_newtable (_L);
			lua_insert (_L, -2);
			lua_rawseti (_L, -2, 1);
		    }

		    /* Call all bound functions. */
		    
		    lua_pushnil (_L);
		    h_1 = lua_gettop (_L);

		    while (lua_next (_L, -2) != 0) {
			lua_pushvalue (_L, -5);
			lua_pushvalue (_L, -5);

			luaX_call (_L, 2, LUA_MULTRET);
		    
			if (lua_type (_L, h_1 + 1) == LUA_TNUMBER) {
			    j = j < lua_tonumber (_L, h_1 + 1) ?
				j : lua_tonumber (_L, h_1 + 1);
			}
			
			if (lua_type (_L, h_1 + 2) == LUA_TNUMBER) {
			    mu = lua_tonumber (_L, h_1 + 2) < dInfinity ?
				 lua_tonumber (_L, h_1 + 2) : dInfinity;
			}

			if (lua_type (_L, h_1 + 3) == LUA_TNUMBER) {
			    epsilon = lua_tonumber (_L, h_1 + 3);
			}

			if (lua_type (_L, h_1 + 4) == LUA_TNUMBER) {
			    sigma = lua_tonumber (_L, h_1 + 4);
			}
			
			if (lua_type (_L, h_1 + 5) == LUA_TNUMBER) {
			    tau = lua_tonumber (_L, h_1 + 5);
			}

			lua_settop (_L, h_1);
		    }
		    
		    lua_settop (_L, h_0);
		}
		    
		/* Create the collision joints. */
	    
		for (i = 0 ; i < j ; i += 1) {
		    dJointID joint;
		    
		    contacts[i].surface.mode = 0;
		    
		    if (mu > 0) {
			contacts[i].surface.mode |= dContactApprox1;
			contacts[i].surface.mode |= dContactFDir1;
			contacts[i].surface.mu = mu;
		    }

		    if (epsilon > 0) {
			contacts[i].surface.mode |= dContactBounce;
			contacts[i].surface.bounce = epsilon;
			contacts[i].surface.bounce_vel = 0.01;
		    }

		    if (sigma > 0) {
			contacts[i].surface.mode |= dContactSoftCFM;
			contacts[i].surface.soft_cfm = sigma;
		    }

		    if (tau < 1) {
			contacts[i].surface.mode |= dContactSoftERP;
			contacts[i].surface.soft_erp = tau;
		    }

		    joint = dJointCreateContact (_WORLD, _GROUP, &contacts[i]);
		    dJointAttach (joint, dGeomGetBody (a), dGeomGetBody (b));
		}
	    }
	}
    }
}

static int test (lua_State *L)
{
    return 0;
}

static int panic(lua_State *L)
{
    printf("Hmmm, I just got this unprotected error from "
	   "the Lua interpreter:\n\"%s\"\n",
           lua_tostring(L, -1));
    
    /* Don't let Lua terminate the process yet.
       There's cleaning up to do. */
    
    /* longjmp (env, 1); */
    
    return 0;
}

static void handler(int signum)
{
    struct sigaction action;
    
    printf("Hmmm, I just received a rather lethal signal "
	   "(number %d if you must know).\n",
	   signum);

    /* Disable the handler for further occurences of this signal. */
    
    action.sa_handler = SIG_DFL;
    action.sa_flags = 0;
    sigemptyset (&action.sa_mask);

    sigaction (SIGHUP, &action, NULL);
    sigaction (SIGINT, &action, NULL);
    sigaction (SIGQUIT, &action, NULL);
    sigaction (SIGILL, &action, NULL);
    sigaction (SIGABRT, &action, NULL);
    sigaction (SIGKILL, &action, NULL);
    sigaction (SIGSEGV, &action, NULL);
    sigaction (SIGPIPE, &action, NULL);
    sigaction (SIGALRM, &action, NULL);
    sigaction (SIGTERM, &action, NULL);
    
    /* Don't terminate the process yet.
       There's cleaning up to do. */
    
    longjmp (env, 1);
}

static void profilehook (lua_State *L, lua_Debug *ar)
{
    static struct timespec now, then;

    if (lua_getstack (L, 1, ar) == 0) {
	if (ar->event == LUA_HOOKCALL) {
	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &then);
	} else if (ar->event == LUA_HOOKRET || ar->event == LUA_HOOKTAILRET) {
	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &now);
	    
	    userinterval +=
		now.tv_sec - then.tv_sec +
		(now.tv_nsec - then.tv_nsec) / 1e9;
	}
    }
}

static int topointer(lua_State *L)
{
    luaL_checkany(L, 1);
    lua_pushlightuserdata (L, (void *)lua_topointer(L, 1));
    
    return 1;
}

static int children_iterator(lua_State *L)
{
    id node;

    if (lua_isnil (L, 2)) {
	/* This is the first invocation.  Get the first child. */
	
	node = *(id *)lua_touserdata(L, 1);
	node = [node children];
    } else {
	/* Get the previous node from the key and then its sibling. */
	
	lua_gettable (L, 1);
	node = [*(id *)lua_touserdata(L, -1) sister];
	lua_pop (L, 1);
    }

    if (node) {
	/* Get the full userdata from the pointer. */
	
	lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	lua_pushlightuserdata (L, node);
	lua_gettable (L, -2);
	lua_replace (L, -2);

	/* Get the key and insert it before the value. */
	
	lua_getmetatable (L, -1);
	lua_getfield (L, -1, "__key");
	lua_insert(L, -3);
	lua_pop(L, 1);
    } else {
	/* We're done. */
	
	lua_pushnil (L);
	lua_pushnil (L);
    }
    
    return 2;
}

static int children(lua_State *L)
{
    lua_pushcfunction (L, children_iterator);
    lua_pushvalue (L, 1);
    lua_pushnil(L);

    return 3;
}

static int properties_iterator(lua_State *L)
{
    id node;
    char **properties;
    int i, bloat;

    if (lua_istable (L, 1)) {
	lua_getmetatable (L, 1);
	
	lua_getfield (L, -1, "__bloat");
	bloat = lua_tointeger (L, -1);
	lua_pop (L, 1);

	lua_getfield (L, -1, "__properties");
	properties = (char **)lua_touserdata (L, -1);
	lua_pop (L, 1);

	lua_pop (L, 1);
    } else {
	node = *(id *)lua_touserdata(L, 1);

	bloat = [node bloat];
	properties = [node properties];
    }	
    
    /* Find the next property on the list. */
    
    if (lua_isnil (L, 2)) {
	i = 0;
    } else {
	for (i = 1;
	     i < bloat && strcmp (lua_tostring(L, 2), properties[i - 1]);
	     i += 1);
    }

    /* Push the key-value pair. */
    
    if (i < bloat) {
	lua_pushstring (L, properties[i]);
	lua_pushvalue (L, -1);
	lua_gettable (L, 1);
    } else {
	lua_pushnil (L);
	lua_pushnil (L);
    }	

    return 2;
}

static int properties(lua_State *L)
{
    lua_pushcfunction (L, properties_iterator);
    lua_pushvalue (L, 1);
    lua_pushnil(L);

    return 3;
}

static int root_tostring(lua_State *L)
{
    lua_pushstring(L, "Root");
   
    return 1;
}

static int root_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    [object get];

    return 1;
}

static int root_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    [object set];

    return 0;
}

static int accoustics_index (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "gain")) {
	float g;
	
	alGetListenerf (AL_GAIN, &g);
	lua_pushnumber (L, g);
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int accoustics_newindex (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "gain")) {
	alListenerf (AL_GAIN, lua_tonumber (L, 3));
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

static void census (lua_State *L, id node)
{
    id child;
    
    /* Skip the subroot. */

    if (node != [root parent]) {
	/* Get the ledger entry for the node's class. */
	
	lua_pushstring (_L, [node name]);
	lua_gettable (_L, -2);

	/* If there isn't one create it. */
	
	if (lua_isnil (_L, -1)) {
	    lua_pop (_L, 1);

	    lua_pushstring (_L, [node name]);
	    lua_newtable (_L);
	    lua_settable (_L, -3);
	    
	    lua_pushstring (_L, [node name]);
	    lua_gettable (_L, -2);
	}

	/* Get the node's userdata. */
	
	lua_getfield (_L, LUA_REGISTRYINDEX, "userdata");
	lua_pushlightuserdata (_L, node);
	lua_gettable (_L, -2);
	lua_replace (_L, -2);

	if (lua_isnil (_L, -1)) {
	    printf ("WARNING: Could not get userdata for node %p of class %s\n", node, [node name]);
	}

	/* Enter the node and pop the entry. */
	
	lua_rawseti (_L, -2, luaX_objlen (_L, -2) + 1);
	lua_pop (_L, 1);
    }
    
    for(child = [node children] ; child ; child = [child sister]) {
	census (L, child);
    }
}

static int configuration_index (lua_State *L)
{
    const char *k;
    int i, j;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "version")) {
	lua_pushnumber (L, strtod(VERSION, NULL));
    } else if (!xstrcmp(k, "graphics")) {
	lua_newtable (L);
	lua_pushstring (L, (const char *)glGetString(GL_VENDOR));
	lua_rawseti (L, -2, 1);
	lua_pushstring (L, (const char *)glGetString(GL_RENDERER));
	lua_rawseti (L, -2, 2);
	lua_pushstring (L, (const char *)glGetString(GL_VERSION));
	lua_rawseti (L, -2, 3);
	lua_pushstring (L, (const char *)glGetString(GL_EXTENSIONS));
	lua_rawseti (L, -2, 4);
    } else if (!xstrcmp(k, "accoustics")) {
	lua_newtable (L);
	lua_pushstring (L, (const char *)alGetString(AL_VENDOR));
	lua_rawseti (L, -2, 1);
	lua_pushstring (L, (const char *)alGetString(AL_RENDERER));
	lua_rawseti (L, -2, 2);
	lua_pushstring (L, (const char *)alGetString(AL_VERSION));
	lua_rawseti (L, -2, 3);
	lua_pushstring (L, (const char *)alGetString(AL_EXTENSIONS));
	lua_rawseti (L, -2, 4);
    } else if (!xstrcmp(k, "toolkit")) {
	double v;
	
        gdk_gl_query_version(&i, &j);

	for (v = j ; v > 1 ; v /= 10);
	
	lua_pushnumber (L, v + i);
    } else if (!xstrcmp(k, "screen")) {
	lua_newtable (L);
	lua_pushinteger (L, gdk_screen_width());
	lua_rawseti (L, -2, 1);
	lua_pushinteger (L, gdk_screen_height());
	lua_rawseti (L, -2, 2);	
    } else if (!xstrcmp(k, "iteration")) {
	lua_newtable (L);

	for (i = 0;
	     i < sizeof (intervals) / sizeof (intervals[0]);
	     i += 1) {
	    lua_pushnumber (L, intervals[i]);
	    lua_rawseti (L, -2, i + 1);
	}
    } else if (!xstrcmp(k, "time")) {
	struct timespec now;

	lua_newtable (L);

	/* Simulation time. */
	
	lua_pushnumber (L, then);
	lua_rawseti (L, -2, 1);

	/* Wall clock time. */
	
	clock_gettime (CLOCK_REALTIME, &now);
	lua_pushnumber (L,
			now.tv_sec - zero.tv_sec +
			(now.tv_nsec - zero.tv_nsec) / 1e9);
	lua_rawseti (L, -2, 2);
	
	/* CPU time for process. */
	
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &now);
	lua_pushnumber (L, now.tv_sec + now.tv_nsec / 1e9);
	lua_rawseti (L, -2, 3);
    } else if (!xstrcmp(k, "nodes")) {
	lua_newtable (L);
	census (L, [root parent]);	
    } else if (!xstrcmp(k, "phase")) {
	lua_pushstring (L, phase);
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int configuration_newindex (lua_State *L)
{
    const char *k;
    
    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "graphics")) {
    } else if (!xstrcmp(k, "accoustics")) {
    } else if (!xstrcmp(k, "toolkit")) {
    } else if (!xstrcmp(k, "screen")) {
    } else if (!xstrcmp(k, "iteration")) {
    } else if (!xstrcmp(k, "time")) {
    } else if (!xstrcmp(k, "nodes")) {
    } else if (!xstrcmp(k, "phase")) {
   } else {
	lua_rawset (L, 1);
    }

    return 0;
}

static int common_index (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "iterate")) {
        lua_pushboolean(L, iterate);
    } else if (!xstrcmp(k, "interval")) {
	if (interval < 0) {
	    lua_pushnil (L);
	} else {
	    lua_pushnumber (L, interval);
	}
    } else if (!xstrcmp(k, "engineering")) {
	lua_pushboolean (L, engineering);
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int common_newindex (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!xstrcmp(k, "iterate")) {
        iterate = lua_toboolean (L, 3);
    } else if (!xstrcmp(k, "interval")) {
	if (lua_isnumber (L, 3)) {
	    interval = lua_tonumber (L, 3);
	} else {
	    interval = -1;
	}
    } else if (!xstrcmp(k, "engineering")) {
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

static int dynamics_index (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "stepsize")) {
        lua_pushnumber(L, stepsize);
    } else if (!xstrcmp(k, "ceiling")) {
        lua_pushnumber(L, ceiling);
    } else if (!xstrcmp(k, "timescale")) {
        lua_pushnumber(L, timescale);
    } else if (!xstrcmp(k, "iterations")) {
        lua_pushnumber(L, iterations);
    } else if (!xstrcmp(k, "collision")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, collision);
    } else if (!xstrcmp(k, "tolerance")) {
	lua_newtable (L);
	lua_pushnumber (L, dWorldGetCFM (_WORLD));
	lua_rawseti (L, -2, 1);
	lua_pushnumber (L, dWorldGetERP (_WORLD));
	lua_rawseti (L, -2, 2);
    } else if (!xstrcmp(k, "popvelocity")) {
	lua_pushnumber(L, dWorldGetContactMaxCorrectingVel (_WORLD));
    } else if (!xstrcmp(k, "surfacelayer")) {
	lua_pushnumber(L, dWorldGetContactSurfaceLayer (_WORLD));
    } else if (!xstrcmp(k, "gravity")) {
	dVector3 gravity;

	dWorldGetGravity (_WORLD, gravity);
	
        lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, gravity[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int dynamics_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "stepsize")) {
        stepsize = lua_tonumber(L, 3);
    } else if (!xstrcmp(k, "ceiling")) {
        ceiling = lua_tonumber(L, 3);
    } else if (!xstrcmp(k, "timescale")) {
        timescale = lua_tonumber(L, 3);
    } else if (!xstrcmp(k, "iterations")) {
        iterations = lua_tonumber(L, 3);

	dWorldSetQuickStepNumIterations (_WORLD, iterations);
    } else if (!xstrcmp(k, "collision")) {
        luaL_unref(L, LUA_REGISTRYINDEX, collision);
        collision = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "tolerance")) {
	lua_rawgeti (L, 3, 1);
	dWorldSetCFM (_WORLD, lua_tonumber (L, -1));
	lua_rawgeti (L, 3, 2);
	dWorldSetERP (_WORLD, lua_tonumber (L, -1));
	lua_pop(L, 2);
    } else if (!xstrcmp(k, "popvelocity")) {
	dWorldSetContactMaxCorrectingVel (_WORLD, lua_tonumber(L, 3));
    } else if (!xstrcmp(k, "surfacelayer")) {
	dWorldSetContactSurfaceLayer (_WORLD, lua_tonumber(L, 3));
    } else if (!xstrcmp(k, "gravity")) {
	dReal gravity[3];
	
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                gravity[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }

	dWorldSetGravity (_WORLD,
			  gravity[0],
			  gravity[1],
			  gravity[2]);
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

static int graphics_index (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "window")) {
	lua_newtable (L);
	lua_pushinteger (L, width);
	lua_rawseti (L, -2, 1);
	lua_pushinteger (L, height);
	lua_rawseti (L, -2, 2);
    } else if (!xstrcmp(k, "hide")) {
	lua_pushboolean (L, hide);
    } else if (!xstrcmp(k, "title")) {
	lua_pushstring (L, title);
    } else if (!xstrcmp(k, "decorate")) {
	lua_pushboolean (L, decorate);
    } else if (!xstrcmp(k, "cursor")) {
	lua_pushboolean (L, cursor);
    } else if (!xstrcmp(k, "grabinput")) {
	lua_pushboolean (L, gdk_pointer_is_grabbed ());
    } else if (!xstrcmp(k, "canvas")) {
	lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, canvas[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "perspective") ||
	!xstrcmp(k, "orthographic")) {
	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_gettable (L, 1);
    } else if (!xstrcmp(k, "frames")) {
	lua_pushnumber (L, frames);
    } else if (!xstrcmp(k, "colorbuffer")) {
	char *pixels;
	int v[4];

	glGetIntegerv (GL_VIEWPORT, v);
	pixels = malloc(v[2] * v[3] * 4);

	/* glReadBuffer (GL_FRONT); */
	glReadPixels(v[0], v[1], v[2], v[3], GL_RGBA,
		     GL_UNSIGNED_BYTE, pixels);

	lua_pushlstring(L, pixels, v[2] * v[3] * 4);
	luaX_pushunsignedbytes(L);

	free(pixels);
    } else if (!xstrcmp(k, "depthbuffer")) {
	char *pixels;
	int v[4];

	glGetIntegerv (GL_VIEWPORT, v);
	pixels = malloc(v[2] * v[3] * sizeof (unsigned short));

	/* glReadBuffer (GL_FRONT); */
	glReadPixels(v[0], v[1], v[2], v[3], GL_DEPTH_COMPONENT,
		     GL_UNSIGNED_SHORT, pixels);

	lua_pushlstring(L, pixels, v[2] * v[3] * sizeof (unsigned short));
	luaX_pushunsignedshorts(L);

	free(pixels);
    } else if (!xstrcmp(k, "stencilbuffer")) {
	char *pixels;
	int v[4];

	glGetIntegerv (GL_VIEWPORT, v);
	pixels = malloc(v[2] * v[3] * sizeof (unsigned short));

	/* glReadBuffer (GL_FRONT); */
	glReadPixels(v[0], v[1], v[2], v[3], GL_STENCIL_INDEX,
		     GL_UNSIGNED_SHORT, pixels);

	lua_pushlstring(L, pixels, v[2] * v[3] * sizeof (unsigned short));
	luaX_pushunsignedshorts(L);

	free(pixels);
    } else if (!xstrcmp(k, "focus")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, focus);
    } else if (!xstrcmp(k, "defocus")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, defocus);
    } else if (!xstrcmp(k, "close")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, delete);
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int graphics_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);
    
    if (!xstrcmp(k, "window")) {
	GdkGeometry geometry;
	
	if (lua_istable (L, 3)) {
	    gdk_window_hide(_WINDOW);
	    gdk_window_unfullscreen (_WINDOW);
	    gdk_window_set_geometry_hints (_WINDOW, NULL, 0);
	    
	    lua_rawgeti (L, 3, 1);
	    width = lua_tointeger(L, -1);

	    lua_rawgeti (L, 3, 2);
	    height = lua_tointeger(L, -1);

	    lua_pop (L, 2);

	    gdk_window_resize (_WINDOW, width, height);
	    
	    geometry.min_width = width;
	    geometry.min_height = height;
	    geometry.max_width = width;
	    geometry.max_height = height;

	    gdk_window_set_geometry_hints(_WINDOW, &geometry,
	    				  GDK_HINT_MIN_SIZE |
	    				  GDK_HINT_MAX_SIZE);

	    if (!hide) {
		gdk_window_clear (_WINDOW);
		gdk_window_show (_WINDOW);
		gdk_window_raise (_WINDOW);
		/* gdk_window_focus (_WINDOW, GDK_CURRENT_TIME); */

		if (width == gdk_screen_width() &&
		    height == gdk_screen_height()) {
		    gdk_window_fullscreen (_WINDOW);
		}
	    }
	}

	/* Always flush when you're done. */
	
	gdk_flush ();

	/* Clear the window's color buffers to the
	   canvas color. */
		
	glDrawBuffer (GL_FRONT_AND_BACK);
	glClear (GL_COLOR_BUFFER_BIT);
	glFlush();
	glDrawBuffer (GL_BACK);
    } else if (!xstrcmp(k, "hide")) {
	/* Update the window if the state has toggled. */

	if (_WINDOW) {
	    if (!hide && lua_toboolean (L, 3)) {
		gdk_window_hide (_WINDOW);
		gdk_flush();
	    }

	    if (hide && !lua_toboolean (L, 3)) {
		gdk_window_show (_WINDOW);
		gdk_window_raise (_WINDOW);
		/* gdk_window_focus (_WINDOW, GDK_CURRENT_TIME); */

		if (width == gdk_screen_width() &&
		    height == gdk_screen_height()) {
		    gdk_window_fullscreen (_WINDOW);
		}

		gdk_flush();

		/* Clear the window's color buffers to the
		   canvas color. */
		
		glDrawBuffer (GL_FRONT_AND_BACK);
		glClear (GL_COLOR_BUFFER_BIT);
		glFlush();
		glDrawBuffer (GL_BACK);
	    }
	}

	hide = lua_toboolean (L, 3);
    } else if (!xstrcmp(k, "title")) {
	gdk_window_set_title (_WINDOW, (char *) lua_tostring (L, 3));
    } else if (!xstrcmp(k, "decorate")) {
	decorate = lua_toboolean (L, 3);

	gdk_window_set_override_redirect (_WINDOW, !decorate);
    } else if (!xstrcmp(k, "grabinput")) {
	if (lua_toboolean (_L, 3)) {
	    gdk_pointer_grab (_WINDOW, TRUE,
			      GDK_BUTTON_PRESS_MASK |
			      GDK_BUTTON_RELEASE_MASK |
			      GDK_POINTER_MOTION_MASK,
			      NULL, NULL, GDK_CURRENT_TIME);
	} else {
	    gdk_pointer_ungrab (GDK_CURRENT_TIME);
	}
    } else if (!xstrcmp(k, "cursor")) {
	GdkCursor *new;

	cursor = lua_toboolean (L, 3);

	new = gdk_cursor_new (cursor ? GDK_ARROW : GDK_BLANK_CURSOR);
	gdk_window_set_cursor (_WINDOW, new);
	gdk_cursor_destroy(new);
    } else if (!xstrcmp(k, "perspective")) {
	GLdouble p[6];

	for (i = 0 ; i < 6 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    p[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
 	glFrustum (p[0], p[1], p[2], p[3], p[4], p[5]);

	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_settable (L, 1);

	lua_pushnil (L);
	lua_setfield (L, 1, "orthographic");
    } else if (!xstrcmp(k, "orthographic")) {
	GLdouble p[6];

	for (i = 0 ; i < 6 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    p[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
 	glOrtho (p[0], p[1], p[2], p[3], p[4], p[5]);
	
	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_settable (L, 1);

	lua_pushnil (L);
	lua_setfield (L, 1, "perspective");
    } else if (!xstrcmp(k, "canvas")) {
	for (i = 0 ; i < 3 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    canvas[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

 	glClearColor (canvas[0], canvas[1], canvas[2], 0);
    } else if (!xstrcmp(k, "focus")) {
	luaL_unref (L, LUA_REGISTRYINDEX, focus);
	focus = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "defocus")) {
	luaL_unref (L, LUA_REGISTRYINDEX, defocus);
	defocus = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "close")) {
	luaL_unref (L, LUA_REGISTRYINDEX, delete);
	delete = luaL_ref (L, LUA_REGISTRYINDEX);
    } else {
	lua_rawset (L, 1);
    }
    
    return 0;
}

int main(int argc, char **argv)
{
    GdkGLConfig *config;
    GdkGLContext *context;
    GdkGLDrawable *drawable = NULL;
    int configuration[] = {GDK_GL_RGBA,
			   GDK_GL_DOUBLEBUFFER,
			   GDK_GL_RED_SIZE, 1,
			   GDK_GL_GREEN_SIZE, 1,
			   GDK_GL_BLUE_SIZE, 1,
			   GDK_GL_ALPHA_SIZE, 1,
			   GDK_GL_DEPTH_SIZE, 12,
			   GDK_GL_STENCIL_SIZE, 1,
			   GDK_GL_ATTRIB_LIST_NONE};
    
    ALCcontext *audio = NULL;
    ALenum error;

    struct sigaction action;
    
    struct timespec once, time, timers[3];    
    double delta;

    char **names = NULL;
    int i, j, option, scripts = 0;
    
    int parameters[] = {0, 0, 0, 0, 0};

    static struct option options[] = {
	{"option", 1, 0, 'O'},
	{"engineering", 0, 0, 'e'},
	{"device", 1, 0, 'd'},
	{"mute", 0, 0, 'm'},
	{"red", 1, 0, 'r'},
	{"green", 1, 0, 'g'},
	{"blue", 1, 0, 'b'},
	{"alpha", 1, 0, 'a'},
	{"depth", 1, 0, 'z'},
	{"stencil", 1, 0, 's'},
	{"frequency", 1, 0, 'f'},
	{"interval", 1, 0, 'i'},
	{"configuration", 1, 0, 'c'},
	{"help", 0, 0, 'h'},
	{0, 0, 0, 0}
    };

    /* Get the current real time. */
    
    clock_gettime (CLOCK_REALTIME, &once);
    clock_gettime (CLOCK_REALTIME, &zero);    

    /* Create an ODE world. */

    dInitODE();
    
    _WORLD = dWorldCreate();
    _GROUP = dJointGroupCreate (0);
    _SPACE = dSimpleSpaceCreate (NULL);

    dWorldSetAutoDisableFlag (_WORLD, 0);

    printf ("Greetings fellow user, this is Techne %s.\n", VERSION);
    
    /* Create the Lua state. */

    _L = luaL_newstate();

    lua_sethook (_L, profilehook, LUA_MASKCALL | LUA_MASKRET, 0);
    lua_atpanic (_L, panic);

    luaL_openlibs (_L);
    lua_settop (_L, 0);

    /* Leave the traceback function first on
       the stack for use by luaX_call. */
    
    lua_getglobal (_L, "debug");
    lua_getfield (_L, -1, "traceback");
    lua_remove (_L, -2);
    
    lua_pushcfunction (_L, test);
    lua_setfield (_L, LUA_GLOBALSINDEX, "test");

    lua_pushcfunction (_L, topointer);
    lua_setfield (_L, LUA_GLOBALSINDEX, "topointer");

    lua_pushcfunction (_L, children);
    lua_setfield (_L, LUA_GLOBALSINDEX, "children");

    lua_pushcfunction (_L, properties);
    lua_setfield (_L, LUA_GLOBALSINDEX, "properties");
    
    /* Create the userdata reference table. */

    lua_newtable (_L);
    lua_newtable (_L);

    lua_pushstring (_L, "__mode");
    lua_pushstring (_L, "v");
    lua_settable (_L, -3);
    lua_setmetatable (_L, -2);
	    
    lua_pushvalue (_L, -1);
    lua_setfield (_L, LUA_REGISTRYINDEX, "userdata");
    
    /* Modify the Lua path and cpath. */

    lua_getglobal (_L, "package");
    lua_getfield (_L, -1, "path");
    lua_pushstring (_L, ";"PKGDATADIR"/modules/?.lua;"PKGDATADIR"/modules/?.lc");
    lua_concat (_L, 2);
    lua_setfield (_L, -2, "path");
    lua_pop (_L, 1);

    lua_getglobal (_L, "package");
    lua_getfield (_L, -1, "cpath");
    lua_pushstring (_L, ";"PKGLIBDIR"/?.so");
    lua_concat (_L, 2);
    lua_setfield (_L, -2, "cpath");
    lua_pop (_L, 1);
    
    /* Create the options table. */

    lua_newtable (_L);
    
    /* Parse the command line. */
    
    while ((option = getopt_long (argc, argv,
				  "O:r:g:b:a:z:s:f:i:c:d:hem",
				  options, 0)) != -1) {
	if (option == 'O') {
	    char *equal;
	    
	    equal = strchr (optarg, '=');

	    if (equal) {
		*equal = '\0';
	    }

	    lua_pushstring (_L, optarg);

	    /* First get the existing option. */
	    
	    lua_pushstring (_L, optarg);
	    lua_gettable (_L, -3);

	    /* If there is only one encapsulate it. */
	    
	    if (lua_type (_L, -1) == LUA_TNIL) {
		lua_pop (_L, 1);
	    } else if (lua_type (_L, -1) != LUA_TTABLE) {
		lua_newtable (_L);
		lua_insert (_L, -2);
		lua_rawseti (_L, -2, 1);
	    }
	    
	    /* Now push the specified value. */

	    if (equal) {
		lua_pushstring (_L, equal + 1);

		/* Convert to number if possible. */
		
		if (lua_isnumber (_L, -1)) {
		    lua_pushnumber (_L, lua_tonumber (_L, -1));
		    lua_replace (_L, -2);
		}
	    } else {
		lua_pushboolean (_L, 1);
	    }

	    /* If this is a duplicate option
	       insert it into the table. */
	    
	    if (lua_type (_L, -2) == LUA_TTABLE) {
		i = lua_objlen (_L, -2);
		lua_rawseti (_L, -2, i + 1);
	    }

	    /* Finally set the option value. */
	    
	    lua_settable (_L, -3);
	} else if (option == 'e') {
	    engineering = 1;
	} else if (option == 'm') {
	    mute = 1;
	} else if (option == 'd') {
	    const char *list, *name;
	    int i;

	    list = alcGetString (NULL, ALC_DEVICE_SPECIFIER);

	    if (!strcmp (optarg, "list")) {
		printf ("\nValid audio devices are:\n");
	    }
	    
	    for (i = 0, name = list;
		 strlen(name);
		 i += 1, name += strlen(name) + 1) {
		if (!strcmp (optarg, "list")) {
		    printf ("  %d: %s\n", i, name);
		} else if (atoi(optarg) == i) {
		    devicename = name;
		}
	    }
	} else if (option == 'r') {
	    configuration[3] = atoi(optarg);
	} else if (option == 'g') {
	    configuration[5] = atoi(optarg);
	} else if (option == 'b') {
	    configuration[7] = atoi(optarg);
	} else if (option == 'a') {
	    configuration[9] = atoi(optarg);
	} else if (option == 'z') {
	    configuration[11] = atoi(optarg);
	} else if (option == 's') {
	    configuration[13] = atoi(optarg);
	} else if (option == 'f') {
	    if (parameters[0] == 0) {
		parameters[0] = ALC_FREQUENCY;
		parameters[1] = atoi(optarg);
	    } else {
		parameters[2] = ALC_FREQUENCY;
		parameters[3] = atoi(optarg);
	    }
	} else if (option == 'i') {
	    if (parameters[0] == 0) {
		parameters[0] = ALC_REFRESH;
		parameters[1] = atoi(optarg);
	    } else {
		parameters[2] = ALC_REFRESH;
		parameters[3] = atoi(optarg);
	    }
	} else if (option == 'c') {
	    names = (char **)realloc (names, (scripts + 1) * sizeof (char *));

	    if (!strcmp (optarg, "-")) {
		names[scripts] = NULL;
	    } else {
		names[scripts] = optarg;
	    }
	    
	    scripts += 1;
	} else if (option == 'h') {
	    printf ("Usage: %s [OPTION...]\n\n"
		    "Options:\n"
		    "  -h, --help                        "
		    "Display this help message.\n\n"
		    "General options:\n"
                    "  -c FILE, --configuration=FILE     "
		    "Specify a configuration script.\n"
		    "  -O OPTION[=VALUE], --option OPTION[=VALUE]\n"
		    "                                    "
		    "Set the option OPTION with value VALUE.\n\n"
		    "  -e, --engineering                 "
		    "Disable rendering and simulate only.\n\n"
		    "Audio related options:\n"
                    "  -m, --mute                        "
		    "Disable audio rendering.\n"
                    "  -d DEVICE, --device=DEVICE        "
		    "The index of the audio device to open.\n"
                    "  -f FREQ, --frequency=FREQ         "
		    "Specify the audio sampling rate in Hz.\n"
                    "  -i INT, --interval=INT            "
		    "Specify the audio refresh interval in Hz.\n\n"
		    "Video related options:\n"
                    "  -r BITS, --red=BITS               "
		    "Set the red color component depth.\n"
                    "  -g BITS, --green=BITS             "
		    "Set the green color component depth.\n"
                    "  -b BITS, --blue=BITS              "
		    "Set the blue color component depth.\n"
                    "  -a BITS, --alpha=BITS             "
		    "Set the alpha color component depth.\n"
                    "  -s BITS, --stencil=BITS           "
		    "Set the stencil buffer depth.\n"
                    "  -z BITS, --depth=BITS             "
		    "Set the Z-buffer depth.\n", argv[0]);

	    exit(1);
	} else {
	    exit(1);
	}
    }

    lua_setglobal (_L, "options");

    /* Create the root node. */

    root = [[Transform alloc] init];
    *(id *)lua_newuserdata (_L, sizeof(id)) = root;
    
    lua_newtable (_L);
    lua_pushstring (_L, "__tostring");
    lua_pushcfunction (_L, (lua_CFunction)root_tostring);
    lua_settable (_L, -3);
    lua_pushstring (_L, "__index");
    lua_pushcfunction (_L, (lua_CFunction)root_index);
    lua_settable (_L, -3);
    lua_pushstring (_L, "__newindex");
    lua_pushcfunction (_L, (lua_CFunction)root_newindex);
    lua_settable (_L, -3);
    lua_setmetatable (_L, -2);
    
    lua_pushstring (_L, "userdata");
    lua_gettable (_L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (_L, root);
    lua_pushvalue (_L, -3);
    lua_settable (_L, -3);
    lua_pop(_L, 1);

    [root toggle];

    lua_setglobal (_L, "graph");

    /* Create the common table. */
    
    lua_newtable (_L);    
    lua_newtable (_L);

    lua_pushcfunction (_L, common_index);
    lua_setfield (_L, -2, "__index");

    lua_pushcfunction (_L, common_newindex);
    lua_setfield (_L, -2, "__newindex");

    lua_setmetatable (_L, -2);
    lua_setglobal (_L, "common");

    /* Create the accoustics table. */
    
    lua_newtable (_L);
    
    {
	char *list[] = {
	    "gain"
	};

	lua_newtable (_L);

	lua_pushinteger (_L, sizeof (list) / sizeof (char *));
	lua_setfield (_L, -2, "__bloat");

	lua_pushlightuserdata (_L, list);
	lua_setfield (_L, -2, "__properties");
	
	lua_pushcfunction (_L, accoustics_index);
	lua_setfield (_L, -2, "__index");

	lua_pushcfunction (_L, accoustics_newindex);
	lua_setfield (_L, -2, "__newindex");

	lua_setmetatable (_L, -2);
    }
    
    lua_setglobal (_L, "accoustics");

    /* Create the graphics table. */
    
    lua_newtable (_L);

    /* Don't set a metatable if we're in
       engineering mode.*/
    
    if (!engineering) {
	char *list[] = {
	    "canvas", "close", "colorbuffer", "defocus", "extensions",
	    "focus", "frames", "hide", "version", "window",
	    "orthographic", "perspective", "renderer", "title", "cursor",
	    "vendor", "grabinput"
	};

	lua_newtable (_L);

	lua_pushinteger (_L, sizeof (list) / sizeof (char *));
	lua_setfield (_L, -2, "__bloat");

	lua_pushlightuserdata (_L, list);
	lua_setfield (_L, -2, "__properties");
	
	lua_pushcfunction (_L, graphics_index);
	lua_setfield (_L, -2, "__index");

	lua_pushcfunction (_L, graphics_newindex);
	lua_setfield (_L, -2, "__newindex");

	lua_setmetatable (_L, -2);
    }
    
    lua_setglobal (_L, "graphics");

    /* Create the configuration table. */

    lua_newtable (_L);    

    {
	char *list[] = {
	    "version", "graphics", "accoustics", "toolkit", "screen"
	    "time", "iteration", "nodes", "phase"
	};
    
	lua_newtable (_L);

	lua_pushinteger (_L, sizeof (list) / sizeof (char *));
	lua_setfield (_L, -2, "__bloat");

	lua_pushlightuserdata (_L, list);
	lua_setfield (_L, -2, "__properties");
    
	lua_pushcfunction (_L, configuration_index);
	lua_setfield (_L, -2, "__index");

	lua_pushcfunction (_L, configuration_newindex);
	lua_setfield (_L, -2, "__newindex");
	
	lua_setmetatable (_L, -2);
    }

    lua_setglobal (_L, "configuration");

    /* Create the dynamics table. */

    lua_newtable (_L);    

    {
	char *list[] = {
	    "ceiling", "collision", "gravity", "iterations",
	    "popvelocity", "stepsize", "surfacelayer",
	    "timescale", "tolerance"
	};
    
	lua_newtable (_L);

	lua_pushinteger (_L, sizeof (list) / sizeof (char *));
	lua_setfield (_L, -2, "__bloat");

	lua_pushlightuserdata (_L, list);
	lua_setfield (_L, -2, "__properties");
    
	lua_pushcfunction (_L, dynamics_index);
	lua_setfield (_L, -2, "__index");

	lua_pushcfunction (_L, dynamics_newindex);
	lua_setfield (_L, -2, "__newindex");
	
	lua_setmetatable (_L, -2);
    }

    lua_setglobal (_L, "dynamics");
    
    if (!engineering) {
        GdkWindowAttr attributes;
	int r, g, b, a, z, s;

	gdk_init(&argc, &argv);
	gdk_gl_init(&argc, &argv);
	
	/* Create the rendering context and
	   associated visual. */

	config = gdk_gl_config_new(configuration);

	if (!config) {
	    fprintf (stderr, "I could not find a suitable "
		             "visual configuration.\n");
	    exit(1);
	}

	/* Create the window. */

	attributes.window_type = GDK_WINDOW_TOPLEVEL;
        attributes.wclass = GDK_INPUT_OUTPUT;
        attributes.colormap = gdk_gl_config_get_colormap(config);
        attributes.visual = gdk_gl_config_get_visual(config);
        attributes.width = 640;
        attributes.height = 480;
        attributes.event_mask = GDK_STRUCTURE_MASK |
	                        GDK_FOCUS_CHANGE_MASK |
	                        GDK_BUTTON_PRESS_MASK |
	                        GDK_BUTTON_RELEASE_MASK |
	                        GDK_KEY_PRESS_MASK |
	                        GDK_KEY_RELEASE_MASK |
	                        GDK_POINTER_MOTION_MASK |
	                        GDK_SCROLL_MASK;

        _WINDOW = gdk_window_new(gdk_get_default_root_window(),
                                &attributes, 
                                GDK_WA_COLORMAP | GDK_WA_VISUAL);
	
	gdk_window_set_gl_capability(_WINDOW, config, NULL);
	
	/* Create the context. */
	
        drawable = gdk_window_get_gl_drawable(_WINDOW);
        context = gdk_gl_context_new(drawable, NULL, TRUE, GDK_GL_RGBA_TYPE);

        gdk_gl_drawable_make_current(drawable, context);
	
	if (!context) {
	    fprintf (stderr, "I could not create a rendering context.\n");
	    exit(1);
	}
	
	/* Create an openal context. */

	if (!mute) {
	    alcGetError(NULL);

	    device = alcOpenDevice (devicename);
	    error = alcGetError(NULL);

	    if (error != ALC_NO_ERROR) {
		fprintf (stderr,
			 "I could not open the audio device (%s).\n",
			 alcGetString(NULL, error));
	    }
    
	    audio = alcCreateContext (device, parameters);
	    error = alcGetError(device);

	    if (error != ALC_NO_ERROR) {
		fprintf (stderr,
			 "I could not create the audio context (%s).\n",
			 alcGetString(device, error));
	    }

	    alcMakeContextCurrent(audio);
	    alcProcessContext(audio);
	}

	/* Print useful debug information. */    

        gdk_gl_query_version(&i, &j);
        gdk_gl_config_get_attrib(config, GDK_GL_RED_SIZE, &r);
        gdk_gl_config_get_attrib(config, GDK_GL_GREEN_SIZE, &g);
        gdk_gl_config_get_attrib(config, GDK_GL_BLUE_SIZE, &b);
        gdk_gl_config_get_attrib(config, GDK_GL_ALPHA_SIZE, &a);
        gdk_gl_config_get_attrib(config, GDK_GL_DEPTH_SIZE, &z);
        gdk_gl_config_get_attrib(config, GDK_GL_STENCIL_SIZE, &s);

	printf ("\nLet's see what kind of junk we're dealing with here.\n");
        
        printf("The GdkGL extension version is %d.%d.\n", i, j);
	printf ("Graphics rendering is provided courtesy of:\n  '%s %s'.\n",
		glGetString(GL_RENDERER), glGetString(GL_VERSION));

	printf ("The framebuffer configuration I managed to get is "
		"[%d, %d, %d, %d, %d, %d].\n", r, g, b, a, z, s);

	printf ("The rendering context is%sdirect.\n",
		gdk_gl_context_is_direct (context) == TRUE ? " " : " *not* ");

	if (!mute) {
	    printf ("Audio spatialization and renderering has been "
		    "subcontracted to:\n  '%s %s'.\n",
		    alGetString(AL_RENDERER), alGetString(AL_VERSION));
	} else {
	    printf ("Audio output has been muted.\n");
	}
    } else {
	printf ("Engineering mode selected, don't expect any visual "
		"hocus-pocus.\n");
    }

    clock_getres(CLOCK_REALTIME, &time);
    printf ("The resolution of the realitme clock is %ld ns\n", time.tv_nsec);
    clock_getres(CLOCK_PROCESS_CPUTIME_ID, &time);
    printf ("The resolution of the process clock is %ld ns\n", time.tv_nsec);

    /* Set up signal handlers for all
       terminal signals so that we at
       least attempt to clean up first. */

    action.sa_handler = handler;
    action.sa_flags = 0;
    sigemptyset (&action.sa_mask);
    
    sigaction (SIGHUP, &action, NULL);
    sigaction (SIGINT, &action, NULL);
    sigaction (SIGQUIT, &action, NULL);
    sigaction (SIGILL, &action, NULL);
    sigaction (SIGABRT, &action, NULL);
    sigaction (SIGKILL, &action, NULL);
    sigaction (SIGSEGV, &action, NULL);
    sigaction (SIGPIPE, &action, NULL);
    sigaction (SIGALRM, &action, NULL);
    sigaction (SIGTERM, &action, NULL);

    /* Save the stack at this point so that we can return
       here to clean up in case something goes wrong. */
 
    if (scripts == 0) {
	printf ("\nDone configuring, but I will now exit since you didn't "
		"bother to specify\nany scripts.  All that work for "
		"nothing...\n");
	
	iterate = 0;
    } else if (setjmp(env)) {
	printf ("\nAttempting to go down gracefully...\n");
	
	iterate = 0;
    } else {
	printf ("\nReading your scripts and hoping for the best.\n");
  
	/* Keep the motor running. */
	
	iterate = 1;
  
	/* Run the boot scripts. */

	for (i = 0 ; i < scripts ; i += 1) {
	    const char *prefixes[] = {"./", PKGDATADIR"/scripts/", ""};
	    char *script = NULL;
	    int j;
	    
	    if (names[i] == NULL && !feof (stdin)) {
		printf ("Waiting for input on the command line...\n\n");
	    } else {
		/* Try to find the specified script in a bunch
		   of predetermined directories. */
		
		for (j = 0;
		     j < sizeof(prefixes) / sizeof (char *);
		     j += 1) {
		    script = realloc (script,
				      strlen (prefixes[j]) +
				      strlen (names[i]) + 1);

		    strcpy (script, prefixes[j]);
		    strcat (script, names[i]);

		    if (!access (script, F_OK)) {
			break;
		    }
		}
	    }

	    if(!luaL_loadfile (_L, script)) {
		luaX_call (_L, 0, 0);
	    } else {
		lua_error (_L);
	    }

	    if (script) {
		free (script);
	    }
	}
    }

    if (iterate) {
	struct timespec now;

	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &now);
	
	printf ("Spent %f CPU seconds initializing.\nNow entering the main loop.\n",
		now.tv_sec + now.tv_nsec / 1e9);
	
    }

    /* Enter the main loop. */

    while (iterate) {
	/* Reset the userspace timer for this frame. */

	userinterval = 0;
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[0]);

	phase = "input";
	
	if (_WINDOW) {
	    GdkEvent *event;

	    while ((event = gdk_event_get()) != NULL) {
		int h;

		/* Process the next event if it's any of
		   our busyness. */
	    
		assert(event);
	    
		h = lua_gettop (_L);

		if(event->type == GDK_CONFIGURE) {
		    glViewport (0, 0,
				((GdkEventConfigure *)event)->width,
				((GdkEventConfigure *)event)->height);
		} else if (event->type == GDK_FOCUS_CHANGE) {
		    if (((GdkEventFocus *)event)->in == TRUE) {
			lua_rawgeti (_L, LUA_REGISTRYINDEX, focus);
		    } else {
			lua_rawgeti (_L, LUA_REGISTRYINDEX, defocus);
		    }
		} else if (event->type == GDK_DELETE) {
		    lua_rawgeti (_L, LUA_REGISTRYINDEX, delete);
		}

		if (lua_gettop (_L) > h) {
		    if (lua_isfunction (_L, -1)) {
			luaX_call (_L, 0, 0);
		    } else if (lua_istable (_L, -1)) {
			lua_pushnil (_L);
			while (lua_next (_L, -2) != 0) {
			    luaX_call (_L, 0, 0);
			}
		    
			lua_pop (_L, 1);
		    } else {
			lua_pop (_L, 1);
		    }
		}

		/* Put it back for others to see. */

		gdk_event_put (event);
		gdk_event_free (event);

		[root input];

		/* Remove the event from the queue. */
	    
		gdk_event_free (gdk_event_get ());
	    }
	}

	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);
	intervals[11] = userinterval;
	intervals[0] = (timers[1].tv_sec - timers[0].tv_sec +
			(timers[1].tv_nsec - timers[0].tv_nsec) / 1e9);

	/* Begin a new frame. */

	userinterval = 0;
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);
	phase = "begin";
	[root begin];
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	intervals[12] = userinterval;
	intervals[1] = (timers[2].tv_sec - timers[1].tv_sec +
			(timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);

	clock_gettime (CLOCK_REALTIME, &time);

	if (interval < 0) {
	    delta = timescale * (time.tv_sec - once.tv_sec +
				 (time.tv_nsec - once.tv_nsec) / 1e9);
	} else {
	    delta = timescale * interval;
	}

	/* Calculate the time to advance for this iteration. */

	if (delta > ceiling) {
	    /* printf ("Skewing the simulation clock by %.3fs\n", */
	    /* 	    ceiling - delta); */
	    now += ceiling;
	} else {
	    now += delta;
	}

	/* Reset the simulation time intervals for this frame. */
	
	intervals[2] = 0;
	intervals[3] = 0;
	intervals[4] = 0;
	intervals[5] = 0;
	intervals[13] = 0;
	intervals[14] = 0;
	intervals[15] = 0;
	
	/* Simulate the system forward. */
	for (delta = now - then;
	     delta >= stepsize;
	     then += stepsize, delta -= stepsize) {
	    /* Transform the tree to update absolute positions
	       and orientations before calling the collision and
	       stepping methods and callbacks. */
	    
	    userinterval = 0;
	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);	

	    phase = "transform";

	    [root transform];

	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	    intervals[13] += userinterval;
	    intervals[2] += (timers[2].tv_sec - timers[1].tv_sec +
			    (timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);

	    /* Empty the contact group. */
	    
	    userinterval = 0;
	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);	

	    dJointGroupEmpty (_GROUP);

	    if (dSpaceGetNumGeoms (_SPACE) > 1) {
		dSpaceCollide (_SPACE, NULL, callback);
	    } else if (dSpaceGetNumGeoms (_SPACE) > 0) {
		dGeomID geom;

		geom = dSpaceGetGeom (_SPACE, 0);

		if (dGeomIsSpace (geom)) {
		    dSpaceCollide ((dSpaceID)geom, NULL, callback);
		}
	    }

	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	    intervals[14] += userinterval;
	    intervals[3] += (timers[2].tv_sec - timers[1].tv_sec +
			    (timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);

	    userinterval = 0;
	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);	
	    phase = "step";
	    [root stepBy: stepsize at: then];
	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	    intervals[15] += userinterval;
	    intervals[4] += (timers[2].tv_sec - timers[1].tv_sec +
			    (timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);

	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);

	    if (iterations > 0) {
		dWorldQuickStep (_WORLD, stepsize);
	    } else {
		dWorldStep (_WORLD, stepsize);
	    }

	    clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	    intervals[5] += (timers[2].tv_sec - timers[1].tv_sec +
			    (timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);
	}

	/* Advance the real-world time. */
	
	memcpy (&once, &time, sizeof (struct timespec));

	/* Traverse the scene. */

	glClear(GL_DEPTH_BUFFER_BIT |
		GL_COLOR_BUFFER_BIT |
		GL_STENCIL_BUFFER_BIT);

	userinterval = 0;
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);
	phase = "prepare";
	[root prepare];
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	intervals[16] = userinterval;
	intervals[6] = (timers[2].tv_sec - timers[1].tv_sec +
			(timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);
	
	userinterval = 0;
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);
	phase = "pass";
	[root traversePass: 1];
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	intervals[17] = userinterval;
	intervals[7] = (timers[2].tv_sec - timers[1].tv_sec +
			(timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);
	
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);
	phase = "repass";
	[root traversePass: 2];
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	intervals[18] = userinterval;
	intervals[8] = (timers[2].tv_sec - timers[1].tv_sec +
			(timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);
	
	userinterval = 0;
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);
	phase = "finish";
	[root finish];
	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[2]);
	intervals[19] = userinterval;
	intervals[9] = (timers[2].tv_sec - timers[1].tv_sec +
			(timers[2].tv_nsec - timers[1].tv_nsec) / 1e9);

	/* Swap the buffers. */
	
	if (_WINDOW) {
	    gdk_gl_drawable_swap_buffers (drawable);
	}

	clock_gettime (CLOCK_PROCESS_CPUTIME_ID, &timers[1]);
	intervals[10] = (timers[1].tv_sec - timers[0].tv_sec +
			(timers[1].tv_nsec - timers[0].tv_nsec) / 1e9);

	for (intervals[20] = 0, i = 11;
	     i < 20;
	     intervals[20] += intervals[i], i += 1);
	
	frames += 1;
    }

    /* Unlink the whole scene. */

    [root toggle];

    /* Finish cleaning up. */
    
    lua_close (_L);

    /* Destroy the audio context. */
    
    if (audio) {
	alcDestroyContext (audio);
    }

    if (device) {
	alcCloseDevice (device);
    }
    
    puts ("Adieu!");
    return 0;
}
