/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "texture.h"
#include "light.h"

#include "surface.h"

@implementation Light

-(Light *) init
{
    char *list[] = {
	"attenuation", "intensity", "offset", "size", "volume"
    };

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];
    
    self->size[0] = 0;
    self->size[1] = 0;

    self->volume[0] = 0;
    self->volume[1] = 0;
    self->volume[2] = 0;
    self->volume[3] = 0;
    self->volume[4] = 0;
    self->volume[5] = 0;

    self->intensityConstant[0] = 1;
    self->intensityConstant[1] = 1;
    self->intensityConstant[2] = 1;

    self->attenuation[0] = 1;
    self->attenuation[1] = 0;
    self->attenuation[2] = 0;

    self->buffer = 0;
    self->shadowMap = 0;
    self->intensityMap = nil;
    
    return self;
}

-(void) loadPositionInto: (unsigned int)this
      andOrientationInto: (unsigned int)that
{
    float T[16], R[9], r[3];
		    
    glMatrixMode (GL_MODELVIEW);
    glPushMatrix ();
    glMultMatrixd([self matrix]);
    glGetFloatv (GL_MODELVIEW_MATRIX, T);

    r[0] = T[12]; r[1] = T[13]; r[2] = T[14];

    R[0] = T[0]; R[1] = T[4]; R[2] = T[8];
    R[3] = T[1]; R[4] = T[5]; R[5] = T[9];
    R[6] = T[2]; R[7] = T[6]; R[8] = T[10];
		
    glPopMatrix();
			
    glUniform3fvARB(this, 1, r);
    glUniformMatrix3fvARB(that, 1, GL_TRUE, R);
}

-(void) loadIntesityConstantInto: (unsigned int)this
                      andMapInto: (unsigned int)that
{
    int unit;

    glGetIntegerv(GL_ACTIVE_TEXTURE, &unit);
    glActiveTexture(unit + 1);
    glBindTexture(GL_TEXTURE_2D, [self->intensityMap index]);
    
    glUniform3fvARB(this, 1, self->intensityConstant);
    glUniform1iARB (that, unit + 1 - GL_TEXTURE0);
}

-(void) loadAttenuationInto: (unsigned int)this
{
    glUniform3fvARB(this, 1, self->attenuation);
}

-(void) loadShadowMatrixInto: (unsigned int)this
		  andMapInto: (unsigned int)that
		 andFlagInto: (unsigned int)there
{		    
    int unit;

    if (self->shadowMap) {
	glGetIntegerv(GL_ACTIVE_TEXTURE, &unit);
	glActiveTexture(unit + 1);
	glBindTexture(GL_TEXTURE_2D, self->shadowMap);
	
	glUniformMatrix4fvARB(this, 1, GL_FALSE, self->shadowMatrix);
	glUniform1iARB (that, unit + 1 - GL_TEXTURE0);
	glUniform1iARB (there, 1);
    } else {
	glUniform1iARB (there, 0);
    }
    
}

-(void) resetBuffers
{
    glDeleteTextures(1, &self->shadowMap);
    glDeleteFramebuffersEXT(1, &self->buffer);

    if (self->size[0] * self->size[1] > 0) {
	/* Create the shadow map texture. */

	glGenTextures(1, &self->shadowMap);
	glBindTexture(GL_TEXTURE_2D, self->shadowMap);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT,
		     self->size[0], self->size[1], 0,
		     GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC, GL_LEQUAL);
	glTexParameteri(GL_TEXTURE_2D, GL_DEPTH_TEXTURE_MODE, GL_INTENSITY);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE,
			GL_COMPARE_R_TO_TEXTURE);

	/* Create the frame buffer object. */

	glGenFramebuffersEXT (1, &self->buffer);
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, self->buffer);
	glFramebufferTexture2DEXT (GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT,
				   GL_TEXTURE_2D, self->shadowMap, 0);
	glDrawBuffer (GL_NONE);
	glReadBuffer (GL_NONE);

	if(glCheckFramebufferStatusEXT (GL_FRAMEBUFFER_EXT) !=
	   GL_FRAMEBUFFER_COMPLETE_EXT) {
	    printf ("Could not create the framebuffer "
		    "object for the shadow map (%x).\n",
		    glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT));
	}
    
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, 0);
    }
}

-(void) transform
{
    double *r, *R;
    
    [super transform];
    
    R = [self rotation];
    r = [self translation];

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();

    gluLookAt(r[0], r[1], r[2], 
	      r[0] + R[2], r[1] + R[5], r[2] + R[8],
	      R[1], R[4], R[7]);
    
    glGetFloatv(GL_MODELVIEW_MATRIX, self->modelview);
    glPopMatrix();
}

-(void) prepare
{
    id root;
    double T[16];

    if (self->size[0] * self->size[1] > 0) {
	/* Set up the viewport and transforms. */
    
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, self->buffer);
	glPushAttrib (GL_VIEWPORT_BIT);
	glViewport (0, 0, self->size[0], self->size[1]);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadMatrixf (self->projection);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf (self->modelview);

	/* Configure and render the scene. */
    
	glCullFace (GL_FRONT);
	glPolygonOffset (self->offset[0], self->offset[1]);
	glEnable (GL_POLYGON_OFFSET_FILL);
	glClear(GL_DEPTH_BUFFER_BIT);
    
	glUseProgramObjectARB(0);

	root = [self root];
	
	[[self parent] traversePass: 0];
    
	/* Restore the state. */
  
	glCullFace(GL_BACK);
	glDisable(GL_POLYGON_OFFSET_FILL);

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	glPopAttrib();
	glBindFramebufferEXT (GL_FRAMEBUFFER_EXT, 0);

	glGetDoublev (GL_TRANSPOSE_MODELVIEW_MATRIX, T);

	T[12] = -T[0] * T[3] - T[4] * T[7] - T[8] * T[11];
	T[13] = -T[1] * T[3] - T[5] * T[7] - T[9] * T[11];
	T[14] = -T[2] * T[3] - T[6] * T[7] - T[10] * T[11];
	T[3] = 0; T[7] = 0; T[11] = 0;
	
	/* Copy the depth buffer into the texture. */

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glTranslatef (0.5, 0.5, 0.5);
	glScalef (0.5, 0.5, 0.5);
	glMultMatrixf(self->projection);
	glMultMatrixf(self->modelview);
	glMultMatrixd(T);
	glGetFloatv(GL_MODELVIEW_MATRIX, self->shadowMatrix);
	glPopMatrix();
    } else {
	self->shadowMap = 0;
    }
    
    [super prepare];
}

-(void) get
{
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "intensity")) {
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_gettable (_L, 1);
    } else if (!xstrcmp(k, "attenuation")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->attenuation[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "size")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->size[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "offset")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->offset[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "volume")) {
        lua_newtable (_L);
        
        for(i = 0; i < 6; i += 1) {
            lua_pushnumber (_L, self->volume[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "intensity")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->intensityConstant[i] = lua_tonumber (_L, -1);
                lua_pop (_L, 1);
            }
	    
	    self->intensityMap = nil;
        } else if(lua_isuserdata (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                self->intensityConstant[i] = 0;
            }
	    
	    self->intensityMap = *(id *)lua_touserdata (_L, 3);
	}

	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_settable (_L, 1);
    } else if (!xstrcmp(k, "attenuation")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->attenuation[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "size")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->size[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    [self resetBuffers];
        }
    } else if (!xstrcmp(k, "offset")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->offset[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "volume")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 6 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->volume[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
    
	    glMatrixMode(GL_PROJECTION);
	    glPushMatrix();
	    glLoadIdentity();
	    glFrustum (self->volume[0], self->volume[1],
		       self->volume[2], self->volume[3],
		       self->volume[4], self->volume[5]);
	    
	    glGetFloatv(GL_PROJECTION_MATRIX, self->projection);
	    glPopMatrix();
        }
    } else {
	[super set];
    }
}

/* -(void) traversePass: (int)pass */
/* { */
/*     if (pass == 1) { */
/* 	glUseProgramObjectARB(0); */

/* 	glColor3f(1, 1, 0); */
    
/* 	glBegin(GL_LINES); */
/* 	glVertex3d(0, 0, 0); */
/* 	glVertex3d(0, 0, 1); */
/* 	glEnd(); */

/* 	glColor3f(1, 0, 0); */
    
/* 	glBegin(GL_POINTS); */
/* 	glVertex3d(0, 0, 1); */
/* 	glEnd(); */
/*     } */

/*     [super traversePass: pass]; */
/* } */

@end
