/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include "item.h"

/* This is a node that keeps track of all allocated
   Nodes by adopting them (and their descendants)
   when they become orphans. */

static Item *subroot = nil;

@implementation Item

-forward: (SEL) sel : args
{
    return nil;
}

+(void) initialize
{
    subroot = [[Item alloc] init];
}

-(Item *) init
{
    [super init];

    self->bloat = 0;
    self->properties = NULL;
    
    self->up = NULL;
    self->down = NULL;
    self->left = NULL;
    self->right = NULL;

    if (subroot) {
	[subroot adopt: self named: [self name]];
    }
    
    return self;
}

-(void) free
{
    free (self->properties);

    /* Remove dead children from the subroot.
       Note that dead notes may belong to other
       parents when whole subtrees are collected. */
    
    if (self->up == subroot) {
    	[subroot renounce: self];
    }

    [super free];
}

-(char *) key
{
    return self->key;
}

-(id) parent
{
    return self->up;
}

-(id) children
{
    return self->down;
}

-(id) brother
{
    return self->left;
}

-(id) sister
{
    return self->right;
}

-(id) root
{
    id root;

    /* Return the proper graph root, not the subroot. */
    
    for (root = self;
	 [root parent] && [root parent] != subroot;
	 root = [root parent]);

    return (id)root;
}

-(int) bloat
{
    return self->bloat;
}

-(char **) properties
{
    return self->properties;
}

-(void) add: (int) n Properties: (char*[]) list
{
    int i;

    self->properties = realloc (self->properties,
    				(self->bloat + n) * sizeof (char *));

    for (i = 0 ; i < n ; i += 1) {
    	self->properties[self->bloat + i] = list[i];
    }

    self->bloat += n;
}

-(void) adoptedBy: (Item *)parent
{
}

-(void) adopt: (Item *)child named: (char *)name
{
    Item *sibling;

    /* Before the child can get adopted by a proper
       parent the subroot needs to let go of it. */
    
    assert (subroot);
    if (self != subroot) {
    	[subroot renounce: child];
    }
    
    child->key = name;
    child->up = self;

    if (!self->down) {
	self->down = child;
	assert (!child->left && !child->right);
    } else {
	for (sibling = self->down;
	     sibling->right && xstrcmp(sibling->right->key, [child key]) < 0;
	     sibling = sibling->right);

	if(xstrcmp(sibling->key, [child key]) < 0) {
	    child->left = sibling;
	    child->right = sibling->right;
	    sibling->right = child;
		
	    if (child->right) {
		child->right->left = child;
	    }
	} else {
	    assert(self->down == sibling);

	    self->down = child;
	    child->right = sibling;
	    sibling->left = child;
	}

	assert (!child->left || xstrcmp(child->left->key, [child key]) <= 0);
	assert (!child->right || xstrcmp(child->right->key, [child key]) >= 0);
    }    
}

-(void) renounce: (Item *)child
{
    if (child->right) {
	child->right->left = child->left;
    }
	    
    if (child->left){
	child->left->right = child->right;
    } else {
	self->down = child->right;
    }

    child->left = NULL;
    child->right = NULL;
    child->up = NULL;

    /* Put any foster nodes under the care of the
       subroot. */
    
    assert (subroot);
    if (self != subroot) {
    	[subroot adopt: child named: [self name]];
    }
}

@end
