/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <limits.h>
#include <wand/MagickWand.h>

static MagickWand *towand (lua_State *L)
{
    MagickBooleanType result = MagickFalse;
    MagickWand *wand;
    int i;

    wand = NewMagickWand();
    MagickSetSize (wand, lua_tonumber (L, 2), lua_tonumber (L, 3));

    /* Choose a format based on the bytes per sample. */
    
    i = luaX_objlen (L, 1) / (lua_tonumber (L, 2) * lua_tonumber (L, 3));

    if (i == 1) {
	MagickSetFormat (wand, "GRAY");
    } else if (i == 3) {
	MagickSetFormat (wand, "RGB");
    } else if (i == 4) {
	MagickSetFormat (wand, "RGBA");
    } else {
	DestroyMagickWand (wand);

	return NULL;
    }

    /* Read the image. */
    
    if (!lua_getmetatable (L, 1)) {
	lua_newtable(L);
    }

    /* Only one of these should be defined. */
    
    lua_getfield (L, -1, "unsigned char");

    if (lua_isstring (L, -1)) {
	MagickSetDepth (wand, 8);
	result = MagickReadImageBlob(wand,
				     (void *)lua_tostring (L, -1),
				     lua_strlen (L, -1));
    }

    lua_pop(L, 1);
    
    lua_getfield (L, -1, "unsigned short");

    if (lua_isstring (L, -1)) {
	MagickSetDepth (wand, 16);
	result = MagickReadImageBlob(wand,
				     (void *)lua_tostring (L, -1),
				     lua_strlen (L, -1));
    }

    lua_pop(L, 1);

    /* If we couldn't read any binary data
       read the table itself. */

    if (!result) {
	unsigned short *pixels;
	
	/* Read the table into a buffer first. */

	MagickSetDepth (wand, 16);
	pixels = (unsigned short *)malloc(luaX_objlen (L, 1) *
					sizeof(unsigned short));

	for(i = 0 ; i < luaX_objlen (L, 1) ; i += 1) {
	    lua_pushinteger (L, i + 1);
	    lua_gettable (L, 1);

	    pixels[i] = (unsigned short)(USHRT_MAX * lua_tonumber(L, -1));

	    lua_pop(L, 1);
	}

	/* Load it into the wand. */
	
	result = MagickReadImageBlob(wand, (void *)pixels,
				     luaX_objlen (L, 1) * sizeof(unsigned short));

	if (result == MagickFalse) {
	    DestroyMagickWand (wand);
	    
	    return NULL;
	}

	free (pixels);
    }
    
    lua_pop(L, 1);

    return wand;
}

static void pushwand (lua_State *L, MagickWand *wand)
{
    unsigned char *pixels;
    size_t length, depth;

    pixels = MagickGetImageBlob (wand, &length);
    depth = MagickGetImageDepth (wand);

    assert (depth == 8 || depth == 16);

    lua_pushlstring (L, (const char *)pixels, length);
    if (depth == 8) {
	luaX_pushunsignedbytes (L);
    } else if (depth == 16) {
	luaX_pushunsignedshorts (L);
    }

    lua_remove (L, -2);
    
    MagickRelinquishMemory (pixels);
}

static int resize (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);
    luaL_checktype (L, 4, LUA_TNUMBER);
    luaL_checktype (L, 5, LUA_TNUMBER);

    wand = towand (L);

    if (!wand) {
	lua_pushnil (L);
    } else {
	MagickResizeImage(wand,
			  lua_tonumber (L, 4), lua_tonumber (L, 5),
			  LanczosFilter, luaL_optnumber (L, 6, 1.0));

	pushwand (L, wand);
	
	DestroyMagickWand (wand);
    }
    
    return 1;
}

static int modulate (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);
    luaL_checktype (L, 4, LUA_TNUMBER);
    luaL_checktype (L, 5, LUA_TNUMBER);
    luaL_checktype (L, 6, LUA_TNUMBER);

    wand = towand (L);

    if (!wand) {
	lua_pushnil (L);
    } else {
	MagickModulateImage(wand,
			    100 * lua_tonumber (L, 4),
			    100 * lua_tonumber (L, 5),
			    100 * lua_tonumber (L, 6));

	pushwand (L, wand);
	
	DestroyMagickWand (wand);
    }
    
    return 1;
}

static int quantize (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);
    luaL_checktype (L, 4, LUA_TNUMBER);

    wand = towand (L);

    if (!wand) {
    	lua_pushnil (L);
    } else {
	char *format;
	ColorspaceType colorspace;
	
	format = MagickGetFormat(wand);

	if (!strcmp (format, "GRAY")) {
	    colorspace = GRAYColorspace;
	} else {
	    colorspace = RGBColorspace;
	}
	
    	MagickQuantizeImage(wand, lua_tonumber (L, 4),
    			    colorspace, 0,
    			    lua_toboolean (L, 5), 0);

	pushwand (L, wand);
	
    	DestroyMagickWand (wand);
    }
    
    return 1;
}

static int normalize (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);

    wand = towand (L);

    if (!wand) {
    	lua_pushnil (L);
    } else {
    	MagickNormalizeImage(wand);
	pushwand (L, wand);
    	DestroyMagickWand (wand);
    }
    
    return 1;
}

static int autolevel (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);

    wand = towand (L);

    if (!wand) {
    	lua_pushnil (L);
    } else {
    	MagickAutoLevelImage(wand);
	pushwand (L, wand);
    	DestroyMagickWand (wand);
    }
    
    return 1;
}

static int flip (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);

    wand = towand (L);

    if (!wand) {
    	lua_pushnil (L);
    } else {
    	MagickFlipImage(wand);
	pushwand (L, wand);
    	DestroyMagickWand (wand);
    }
    
    return 1;
}

static int flop (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);

    wand = towand (L);

    if (!wand) {
    	lua_pushnil (L);
    } else {
    	MagickFlopImage(wand);
	pushwand (L, wand);
    	DestroyMagickWand (wand);
    }
    
    return 1;
}

static int writeformat (lua_State *L)
{
    MagickWand *wand;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TNUMBER);
    luaL_checktype (L, 3, LUA_TNUMBER);
    luaL_checktype (L, 4, LUA_TSTRING);

    wand = towand (L);

    if (!wand) {
    	lua_pushnil (L);
    } else {
	unsigned char *pixels;
	size_t length;

    	MagickSetFormat(wand, lua_tostring (L, 4));
	
	pixels = MagickGetImageBlob (wand, &length);
	lua_pushlstring (L, (char *)pixels, length);
	
    	DestroyMagickWand (wand);
    }
    
    return 1;
}

int luaopen_imaging (lua_State *L)
{
    const luaL_Reg imaging[] = {
	{"resize", resize},
	{"modulate", modulate},
	{"quantize", quantize},
	{"normalize", normalize},
	{"autolevel", autolevel},
	{"flip", flip},
	{"flop", flop},
	{"writeformat", writeformat},
	{NULL, NULL}
    };
    
    MagickWandGenesis();

    luaL_register (L, "imaging", imaging);

    return 0;
}
