/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>

#include "frame.h"

@implementation Frame

-(Frame *)init
{
    char *list[] = {"radius", "thickness"};

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->thickness = 1;
    self->radius = 0.003;

    return self;
}

-(double)thickness
{
    return self->thickness;
}

-(double)radius
{
    return self->radius;
}

-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "thickness")) {
	lua_pushnumber (_L, self->thickness);
    } else if (!xstrcmp(k, "radius")) {
	lua_pushnumber (_L, self->radius);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "thickness")) {
	self->thickness = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "radius")) {
	self->radius = lua_tonumber (_L, 3);
    } else {
	[super set];
    }
}

-(double) measureWidth
{
    id child;

    for (child = [self children];
	 child && ![child isKindOf: [Widget class]];
	 child = [child sister]);
    
    if(child) {
    	self->minimum[0] = [child measureWidth] +
	                   2 * self->padding[0];
    } else {
    	self->minimum[0] = 2 * self->padding[0];
    }

    return self->minimum[0];
}

-(double) measureHeight
{
    id child;

    for (child = [self children];
	 child && ![child isKindOf: [Widget class]];
	 child = [child sister]);
    
    if(child) {
    	self->minimum[1] = [child measureHeight] +
	                   2 * self->padding[1];
    } else {
    	self->minimum[1] = 2 * self->padding[1];
    }

    return self->minimum[1];
}

-(void) transform
{
    id child;

    for (child = [self children];
	 child && ![child isKindOf: [Widget class]];
	 child = [child sister]);
    
    if (self->align[0] < 0) {
    	self->position[0] += 0.5 * (self->minimum[0] - self->allocated[0]);
    } else if (self->align[0] > 0) {
    	self->position[0] += 0.5 * (self->allocated[0] - self->minimum[0]);
    }

    if (self->align[1] < 0) {
    	self->position[1] += 0.5 * (self->minimum[1] - self->allocated[1]);
    } else if (self->align[1] > 0) {
    	self->position[1] += 0.5 * (self->allocated[1] - self->minimum[1]);
    }

    if(child) {
    	[child position][0] = 0;
    	[child position][1] = 0;
    	[child position][2] = 0;
	 
	 [child allocated][0] = self->minimum[0];
	 [child allocated][1] = self->minimum[1];
    }	

    [super transform];
}

-(void) traversePass: (int)pass
{
    if (pass == 2) {
	double w, h, rho;

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([self matrix]);

	rho = self->radius;
	w = self->minimum[0] / 2;
	h = self->minimum[1] / 2;

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glLineWidth (self->thickness);

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);

	glDepthMask (GL_FALSE);

	glColor3dv(self->color);

	glBegin(GL_LINE_STRIP);
	glVertex2d(-w + rho, -h);
	glVertex2d(w - rho, -h);

	glVertex2d(w, -h + rho);
	glVertex2d(w, h - rho);
    
	glVertex2d(w - rho, h);
	glVertex2d(-w + rho, h);

	glVertex2d(-w, h - rho);
	glVertex2d(-w, -h + rho);
    
	glVertex2d(-w + rho, -h);
	glEnd();

	glColor4dv (self->background);
    
	glBegin(GL_POLYGON);
	glVertex2d(-w + rho, -h);
	glVertex2d(w - rho, -h);

	glVertex2d(w, -h + rho);
	glVertex2d(w, h - rho);
    
	glVertex2d(w - rho, h);
	glVertex2d(-w + rho, h);

	glVertex2d(-w, h - rho);
	glVertex2d(-w, -h + rho);
    
	glVertex2d(-w + rho, -h);
	glEnd();

	glDepthMask (GL_TRUE);

	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
    
	glMatrixMode (GL_MODELVIEW);
	glPopMatrix();
    }
    
    [super traversePass: pass];
}

@end
