/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "body.h"
#include "meteorology.h"
#include "fourstroke.h"

@implementation Fourstroke

-(Fourstroke *) init
{
    char *list[] = {
      "anchor", "axis", "throttle", "inertia", "peak",
      "tolerance", "state", "cylinders", "gearbox", "gears", "primary"
    };

    self->joint = dJointCreateHinge (_WORLD, NULL);
    
    self->anchor[0] = 0;
    self->anchor[1] = 0;
    self->anchor[2] = 0;

    self->axis[0] = 0;
    self->axis[1] = 0;
    self->axis[2] = 1;

    self->flywheel = dBodyCreate (_WORLD);
    self->transmission = dJointCreateAMotor (_WORLD, NULL);
    
    dBodySetData (self->flywheel, self);
    dBodySetAngularVel (self->flywheel, 0, 800 * 2 * M_PI / 60, 0);
    
    dJointSetFeedback (self->transmission, &self->output);

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->benchspeed = -1;
    self->peak = 1.0 / 0;
    self->throttle = 0;
    self->displacement = 0.000250;
    self->cylinders = 4;
    self->bypass = 0.01;
    self->gearbox = NULL;
    self->gears = 0;
    self->primary = 1;
    self->idle = 1;
    self->inertia = 0.02;
    
    return self;
}

-(void) free
{  
    dBodyDestroy (self->flywheel);
    dJointDestroy (self->transmission);
  
    [super free];
}

-(void) update
{
    dBodyID chassis, load;
    dMatrix3 R;

    [super update];

    load = dJointGetBody (self->joint, self->inverted ? 0 : 1);
    chassis = dJointGetBody (self->joint, self->inverted ? 1 : 0);

    /* Position the flywheel correctly. */

    dRFrom2Axes(R,
		self->axis[1], -self->axis[0], 0,
		self->axis[0], self->axis[1], self->axis[2]);

    dBodySetRotation (self->flywheel, R);
    dBodySetPosition (self->flywheel,
		      self->anchor[0],
		      self->anchor[1],
		      self->anchor[2]);

    dJointAttach (self->joint, self->flywheel, chassis);
    dJointAttach (self->transmission, self->flywheel, load);

    dJointSetAMotorMode (self->transmission, dAMotorUser);
    dJointSetAMotorNumAxes (self->transmission, 1);
    dJointSetAMotorAxis (self->transmission, 0, 1,
			 self->axis[0],
			 self->axis[1],
			 self->axis[2]);
    
    /* Axis and anchor should be set after the
       joint has been attached. */
	
    dJointSetHingeAxis ([self joint],
			self->axis[0],
			self->axis[1],
			self->axis[2]);

    dJointSetHingeAnchor ([self joint],
			  self->anchor[0],
			  self->anchor[1],
			  self->anchor[2]);
}

-(void) cycle
{
    const double H_0 = 45e6, R_a = 286.9, phi_0 = 5 * M_PI / 180;

    double Q_atr, Q_atr0, Q_ac, Q_ac0, gamma_t, A;
    double omega, theta, M, M_ind, M_loss, q;
    double p_me0g, p_me0f, p_m0, p_m1, p_mm;
    double T_m = 273 + 30, T_0 = 273 + 20, p_0 = 1e5;
    double phi = (85 * (self->throttle + self->bypass)) * M_PI / 180;
    double eta_t, eta_v, eta_v0, z = self->cylinders;
    int i;

/*   printf ("%f, %f\n", eta_v, eta_t); */
    
    if (self->benchspeed >= 0) {
	omega = self->benchspeed;

	self->benchspeed = -1;
    } else {
	theta = dJointGetHingeAngle (self->joint);
	omega = dJointGetHingeAngleRate (self->joint);
    }
    
    /* Calculate a few constant values. */

    A = M_PI_4 * self->intake[0] * self->intake[0] *
	(1 - cos (phi + phi_0) / cos(phi_0));
    Q_atr0 = self->intake[1] * z * A * p_0 / sqrt (R_a * T_0) * 0.68473;
    Q_ac0 = 0.5 * z * self->displacement * omega / (2 * M_PI) / (R_a * T_m);

    eta_v0 = self->volumetric[0] +
	self->volumetric[1] * omega +
	self->volumetric[2] * omega * omega;

    eta_t = self->thermal[0] +
	self->thermal[1] * omega +
	self->thermal[2] * omega * omega;

    /* Solve for the manifold pressure that equalises 
       the air mass flows into and out of the manifold.
       The function is monotonically decreasing, therefore:
     
       f(0) * f(p_0) < 1 and f(0) > f(p_0)

       The bisection method is used which is guaranteed to
       converge with a tolerance of 100 Pa after 10 iterations. */
  
    for (i = 0, p_m0 = 0, p_m1 = p_0 ; i < 10 ; i += 1) {
	p_mm = 0.5 * (p_m0 + p_m1);

	eta_v = eta_v0 + self->volumetric[3] * p_mm;
	gamma_t = 1.8929 * p_mm / p_0;

	if (gamma_t > 1) {
	    Q_atr = Q_atr0 * 2.4495 * sqrt(pow(gamma_t, 1.4286) -
					   pow (gamma_t, 1.7143) / 1.2);
	} else {
	    Q_atr = Q_atr0;
	}

	Q_ac = Q_ac0 * eta_v * p_mm;

	if (Q_atr > Q_ac) {
	    p_m0 = p_mm;
	} else {
	    p_m1 = p_mm;
	}
    }

    /* Calculate induced torque. */

    M_ind = eta_t * H_0 * Q_ac / 14.7 / omega;

    /* Losses due to friction and gas-exchange according to
       Introduction to Modeling and Control of IC Engine Systems
       pp. 68. */

    p_me0g = self->exchange[0] * (1 - self->exchange[1] * p_mm / p_0);
    p_me0f = self->friction[0] +
	self->friction[1] * omega +
	self->friction[2] * omega * omega;

    if (omega > 0) {
	M_loss = (p_me0g + p_me0f) * z * self->displacement / (4 * M_PI);
    } else {
	M_loss = 0;
    }

    M = M_ind - M_loss;

    dJointSetHingeParam (self->joint, dParamVel, M / 0);

    /* Caclulate the total reduction from
       crankshaft to countershaft. */

    if (!self->idle) {
	q = self->primary * self->gearbox[self->gear];

	dJointSetHingeParam (self->joint, dParamFMax, q * fabs(M));
	dJointSetAMotorParam (self->transmission, dParamVel,
			      (1 - 1 / q) * omega);
	dJointSetAMotorParam (self->transmission, dParamFMax, self->peak);
    } else {
	dJointSetHingeParam (self->joint, dParamFMax, fabs(M));
	dJointSetAMotorParam (self->transmission, dParamFMax, 0);
    }
  
    self->state[2] = p_mm;
    self->state[3] = eta_v;
    self->state[4] = eta_t;
    self->state[5] = M_ind;
    self->state[6] = p_me0g * z * self->displacement / (4 * M_PI);
    self->state[7] = p_me0f * z * self->displacement / (4 * M_PI);
    self->state[8] = M;

/*   printf ("%f, %f, %f, %f;\n", */
/* 	  omega / 2 / M_PI * 60, */
/* 	  q * M_ind * omega / 745.699872, */
/* 	  q * M_loss * omega / 745.699872, */
/* 	  q * M * omega / 745.699872); */
}

-(void) stepBy: (double) h at: (double) t
{
    [self cycle];
    [super stepBy: h at: t];
}

-(void) get
{
    const char *k;
    int i;
    
    k = lua_tostring(_L, -1);

    if (!xstrcmp(k, "anchor")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->anchor[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "tolerance")) {
	lua_pushnumber (_L, self->tolerance);
    } else if (!xstrcmp(k, "axis")) {
	dVector3 a;
	
	dJointGetHingeAxis ([self joint], a);

	lua_newtable (_L);
       
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, a[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "gearbox")) {
	lua_newtable (_L);
       
        for(i = 0; i < self->gears; i += 1) {
            lua_pushnumber (_L, self->gearbox[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "gear")) {
	if (self->idle) {
	    lua_pushnil(_L);
	} else {
	    lua_pushinteger(_L, self->gear + 1);
	}
    } else if (!xstrcmp(k, "primary")) {
	lua_pushnumber(_L, self->primary);
    } else if (!xstrcmp(k, "throttle")) {
	lua_pushnumber(_L, self->throttle);
    } else if (!xstrcmp(k, "peak")) {
      lua_pushnumber (_L, self->peak);
    } else if (!xstrcmp(k, "inertia")) {
      lua_pushnumber (_L, self->inertia);
    } else if (!xstrcmp(k, "displacement")) {
      lua_pushnumber (_L, self->displacement);
    } else if (!xstrcmp(k, "cylinders")) {
      lua_pushinteger (_L, self->cylinders);
    } else if (!xstrcmp(k, "intake")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->intake[i]);
            lua_rawseti (_L, -2, i + 1);
        }      
    } else if (!xstrcmp(k, "volumetric")) {
        lua_newtable (_L);
        
        for(i = 0; i < 4; i += 1) {
            lua_pushnumber (_L, self->volumetric[i]);
            lua_rawseti (_L, -2, i + 1);
        }      
    } else if (!xstrcmp(k, "thermal")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->thermal[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "friction")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->friction[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "exchange")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->exchange[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "forces")) {
        lua_newtable (_L);

	/* The force applied on the chassis. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->feedback.f2[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 1);

	/* The force applied on the sprocket. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->output.f2[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 2);
    } else if (!xstrcmp(k, "torques")) {
        lua_newtable (_L);

	/* The torque applied on the chassis. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->feedback.t2[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 1);

	/* The torque applied on the sprocket. */
	
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->output.t2[i]);
            lua_rawseti (_L, -2, i + 1);
        }

	lua_rawseti (_L, -2, 2);	
    } else if (!xstrcmp(k, "state")) {
	if ([self joint]) {
	    self->state[0] = dJointGetHingeAngle ([self joint]);
	    self->state[1] = dJointGetHingeAngleRate ([self joint]);

	    lua_newtable (_L);
        
	    for(i = 0 ; i < 9 ; i += 1) {
		lua_pushnumber (_L, self->state[i]);
		lua_rawseti (_L, -2, i + 1);
	    }
	} else {
	    lua_pushnil (_L);
	}
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i;
    
    k = lua_tostring(_L, -2);

    if (!xstrcmp(k, "axis")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->axis[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    dSafeNormalize3 (self->axis);
	    dJointSetHingeAxis ([self joint],
				self->axis[0],
				self->axis[1],
				self->axis[2]);

	    /* Can't set the axis relative to the first
	       body when the motor ain't connected. */
	    
	    if (dJointGetBody (self->transmission, 0)) {
		dJointSetAMotorAxis (self->transmission, 0, 1,
				     self->axis[0],
				     self->axis[1],
				     self->axis[2]);
	    }
        }
    } else if (!xstrcmp(k, "anchor")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->anchor[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    dJointSetHingeAnchor ([self joint],
				  self->anchor[0],
				  self->anchor[1],
				  self->anchor[2]);
        }
    } else if (!xstrcmp(k, "tolerance")) {
	self->tolerance = lua_tonumber (_L, 3);

	dJointSetHingeParam ([self joint], dParamCFM, self->tolerance);
    } else if (!xstrcmp(k, "gearbox")) {
        if(lua_istable (_L, 3)) {
	    self->gears = luaX_objlen (_L, 3);
	    self->gearbox = (double *)realloc (self->gearbox,
					       self->gears * sizeof (double));
	    
            for(i = 0 ; i < self->gears ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->gearbox[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "gear")) {
	if (lua_isnumber (_L, 3) &&
	    lua_tointeger (_L, 3) > 0 &&
	    lua_tointeger (_L, 3) <= self->gears) {
	    self->gear = lua_tointeger (_L, 3) - 1;
	    self->idle = 0;
	} else {
	    self->idle = 1;
	}
    } else if (!xstrcmp(k, "primary")) {
	self->primary = lua_tonumber(_L, 3);
    } else if (!xstrcmp(k, "throttle")) {
	self->throttle = lua_tonumber(_L, 3);

	if (self->throttle < 0) {
	    self->throttle = 0;
	}

	if (self->throttle > 1) {
	    self->throttle = 1;
	}
    } else if (!xstrcmp(k, "peak")) {
	self->peak = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "inertia")) {
	self->inertia = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "displacement")) {
	self->displacement = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "cylinders")) {
	self->cylinders = lua_tointeger (_L, 3);
    } else if (!xstrcmp(k, "state")) {
	if (lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 2);
	    self->benchspeed = lua_tonumber (_L, -1);
	    lua_pop(_L, 1);
	    
	    [self cycle];
	}
    } else if (!xstrcmp(k, "intake")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->intake[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "volumetric")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 4 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->volumetric[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "thermal")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->thermal[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "friction")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->friction[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "exchange")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->exchange[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else {
	[super set];
    }

    if (!xstrcmp(k, "gear") ||
	!xstrcmp(k, "gearbox") ||
        !xstrcmp(k, "inertia")) {
      dMass m;
      
      dMassSetZero (&m);

      m.mass = 1e-2;

      m.I[0] = 1e-6;
      m.I[5] = self->inertia *
	  (self->idle ? 1 : self->primary * self->gearbox[self->gear]);
      m.I[10] = 1e-6;

      dBodySetMass (self->flywheel, &m);
    }
}

-(void) traversePass: (int)pass
{
    if (pass == 2 && self->debug) {
	dBodyID a, b;
	dVector3 p, x;

	a = dJointGetBody ([self joint], 0);
	b = dJointGetBody ([self joint], 1);

	assert (a || b);

	dJointGetHingeAnchor ([self joint], p);
	dJointGetHingeAxis ([self joint], x);

	glUseProgramObjectARB(0);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_BLEND);
	glDepthMask (GL_FALSE);

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glTranslatef(p[0], p[1], p[2]);
	
	/* Draw the crankshaft. */

	glPointSize (5);
	glLineWidth(2);
	
	glColor3f(0, 0, 1);

	glBegin (GL_LINES);
	glVertex3f (0, 0, 0);
	glVertex3f (x[0], x[1], x[2]);

	glEnd ();	

	glColor3f(1, 0, 0);

	glBegin (GL_POINTS);
	glVertex3f (x[0], x[1], x[2]);
	glEnd ();

	glPopMatrix ();

	glDepthMask (GL_TRUE);
	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_POINT_SMOOTH);
	glDisable(GL_DEPTH_TEST);
    }
    
    [super traversePass: pass];
}

@end
