/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include "system.h"
#include "point.h"
#include "ball.h"
#include "box.h"
#include "plane.h"
#include "capsule.h"
#include "environment.h"
#include "ray.h"
#include "polyhedron.h"
#include "heightfield.h"
#include "composite.h"
#include "convex.h"

#include "slider.h"
#include "hinge.h"
#include "universal.h"
#include "spherical.h"
#include "clamp.h"
#include "contact.h"
#include "conical.h"
#include "polar.h"

#include "angular.h"
#include "euler.h"
#include "linear.h"

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int foundry_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object freeData];
    [object free];

    return 0;
}

static int foundry_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [object name]);
    lua_pushstring(L, " foundry");
    lua_concat(L, 2);
    
    return 1;
}

static int dummy_index(lua_State *L)
{
    lua_pushnil (L);
    
    return 1;
}

static int dummy_newindex(lua_State *L)
{
    return 0;
}

static int node_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [object name]);
   
    return 1;
}

static int node_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object get];
    
    return 1;
}

static int node_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object set];

    return 0;
}

static int node_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object free];

    return 0;
}

static int constructnode (lua_State *L)
{
    Class class;
    id object;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    object = [[class alloc] init];
    
    /* Create the userdata... */
    
    *(id *)lua_newuserdata(L, sizeof(id)) = object;
    
    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)node_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int polyhedron_len(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [object vertices]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [object indices] / 3);
    lua_rawseti(L, -2, 2);

    return 1;
}

static int polyhedron_call (lua_State *L)
{
    id parent, child;

    parent = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */

    child = [[parent copy] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = child;
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, polyhedron_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, node_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, child);
    lua_pushvalue (L, -3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int constructpolyhedron(lua_State *L)
{
    id object;
    float *vertices;
    int *indices;
    int i, size[2];
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "vertices");
    lua_gettable(L, 1);
    
    /* Get the vertices. */

    vertices = (float *)malloc(3 * size[0] * sizeof(float));

    for(i = 0 ; i < 3 * size[0] ; i += 1) {
	lua_pushinteger (_L, i + 1);
	lua_gettable (_L, -2);

	vertices[i] = (float)lua_tonumber(L, -1);

	lua_pop(L, 1);
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "indices");
    lua_gettable(L, 1);

    /* And the indices. */

    indices = (int *)malloc(size[1] * sizeof(int));

    for(i = 0 ; i < size[1] ; i += 1) {
	lua_pushinteger (_L, i + 1);
	lua_gettable (_L, -2);

	indices[i] = (int)lua_tonumber(L, -1);

	lua_pop(L, 1);
    }
    
    lua_pop(L, 1);
    
    /* Create and initialize the polyhedron userdata. */
    
    object = [[Polyhedron alloc] initWithVertices: vertices
	                               andIndices: indices
	                                   ofSize: size];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, foundry_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, polyhedron_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, foundry_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, dummy_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, dummy_newindex);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, -3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    free(vertices);
    free(indices);

    return 1;
}

static int heightfield_len(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [object width]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [object height]);
    lua_rawseti(L, -2, 2);

    return 1;
}

static int heightfield_call (lua_State *L)
{
    id parent, child;

    parent = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */

    child = [[parent copy] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = child;
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, heightfield_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, node_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, child);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

int constructheightfield(lua_State *L)
{
    id object;
    float *samples;
    int i, size[2];
    float area[2], scale, offset, thickness;
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "area");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            area[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "scale");
    lua_gettable(L, 1);

    if (lua_isnumber (L, -1)) {
	scale = lua_tonumber (L, -1);
    } else {
	scale = 1;
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "offset");
    lua_gettable(L, 1);

    if (lua_isnumber (L, -1)) {
	offset = lua_tonumber (L, -1);
    } else {
	offset = 0;
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "thickness");
    lua_gettable(L, 1);

    if (lua_isnumber (L, -1)) {
	thickness = lua_tonumber (L, -1);
    } else {
	thickness = 0;
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "samples");
    lua_gettable(L, 1);
    
    samples = (float *)malloc(size[0] * size[1] * sizeof(float));

    for(i = 0 ; i < size[0] * size[1] ; i += 1) {
	lua_pushinteger (_L, i + 1);
	lua_gettable (_L, -2);
	samples[i] = (float)(lua_tonumber(L, -1));
	lua_pop(L, 1);
    }
    
    lua_pop(L, 1);

    /* Create and initialize the heightfield userdata. */

    object = [[Heightfield alloc] initWithFloats: samples
                                     ofSize: size
		                   spanning: area
		                   scaledBy: scale
		                   offsetBy: offset
	                     andThickenedBy: thickness];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, foundry_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, heightfield_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, foundry_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, dummy_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, dummy_newindex);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, -3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int convex_len(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [object planes]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [object points]);
    lua_rawseti(L, -2, 2);

    return 1;
}

static int convex_call (lua_State *L)
{
    id parent, child;

    parent = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */

    child = [[parent copy] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = child;
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, convex_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, node_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, child);
    lua_pushvalue (L, 3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

int constructconvex(lua_State *L)
{
    id object;
    dReal *planes = NULL, *points = NULL;
    unsigned int *polygons = NULL;
    int i, j, n = 0, m = 0, size[2] = {0, 0};

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "points");
    lua_gettable(L, 1);
    
    if (lua_type(L, -1) == LUA_TTABLE) {
	n = luaX_objlen (L, -1);
	points = (dReal *)malloc(3 * n * sizeof(dReal));

	for(i = 0 ; i < 3 * n ; i += 1) {
	    lua_pushinteger (_L, i + 1);
	    lua_gettable (_L, -2);

	    points[i] = (dReal)(lua_tonumber(L, -1));

	    lua_pop(L, 1);
	}
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "polygons");
    lua_gettable(L, 1);
    
    if (lua_type(L, -1) == LUA_TTABLE) {
	m = luaX_objlen (L, -1);
	polygons = (unsigned int *)malloc(m * sizeof(unsigned int));

	for(i = 0 ; i < m ; i += 1) {
	    lua_pushinteger (_L, i + 1);
	    lua_gettable (_L, -2);

	    polygons[i] = (unsigned int)(lua_tonumber(L, -1));

	    lua_pop(L, 1);
	}
    }
    
    lua_pop(L, 1);

    planes = (dReal *)malloc(4 * size[1] * sizeof(dReal));
    
    for (i = 0, j = 0 ; j < size[1] ; i += polygons[i] + 1, j += 1) {
	dReal *p, *q, *r, d;
	dVector3 u, v, n;

	p = &points[3 * polygons[i + 1]];
	q = &points[3 * polygons[i + 2]];
	r = &points[3 * polygons[i + 3]];
	
	u[0] = q[0] - p[0];
	u[1] = q[1] - p[1];
	u[2] = q[2] - p[2];

	v[0] = r[0] - p[0];
	v[1] = r[1] - p[1];
	v[2] = r[2] - p[2];

	dCROSS (n, =, u, v);
	dSafeNormalize3(n);

/* 	printf ("%f, %f, %f\n", p[0], p[1], p[2]); */
/* 	printf ("%f, %f, %f\n", u[0], u[1], u[2]); */
/* 	printf ("%f, %f, %f\n", v[0], v[1], v[2]); */
/* 	printf ("%f, %f, %f\n", n[0], n[1], n[2]); */

	d = dDOT (n, p);

	planes[j * 4] = n[0];
	planes[j * 4 + 1] = n[1];
	planes[j * 4 + 2] = n[2];
	planes[j * 4 + 3] = d;
    }

    /* Create and initialize the convex userdata. */

    object = [[Convex alloc] initWith: size[1] planes: planes
                                  and: size[0] points: points
		          andPolygons: polygons];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, foundry_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, convex_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, foundry_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, dummy_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, dummy_newindex);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, -3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

int luaopen_dynamics (lua_State *L)
{
    int i;
    
    Class bodies[] = {
	[System class], [Point class], [Ball class], [Box class],
	[Environment class], [Capsule class], [Ray class],
	[Plane class], [Composite class]
    };	
    
    Class joints[] = {
	[Slider class], [Universal class], [Hinge class], [Spherical class],
	[Clamp class], [Contact class], [Conical class],
	[Polar class]
    };	
    
    Class motors[] = {
	[Angular class], [Linear class], [Euler class]
    };	

    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(bodies) / sizeof(bodies[0]) ; i += 1) {
	lua_pushlightuserdata (L, bodies[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([bodies[i] name])));
    }

    lua_pushcfunction (L, constructpolyhedron);
    lua_setfield(L, -2, decapitalize(strdupa([Polyhedron name])));

    lua_pushcfunction (L, constructheightfield);
    lua_setfield(L, -2, decapitalize(strdupa([Heightfield name])));

    lua_pushcfunction (L, constructconvex);
    lua_setfield(L, -2, decapitalize(strdupa([Convex name])));
    
    lua_setglobal (L, "bodies");

    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(joints) / sizeof(joints[0]) ; i += 1) {
	lua_pushlightuserdata (L, joints[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([joints[i] name])));
    }

    lua_setglobal (L, "joints");

    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(motors) / sizeof(motors[0]) ; i += 1) {
	lua_pushlightuserdata (L, motors[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([motors[i] name])));
    }

    lua_setglobal (L, "motors");

    return 0;
}
