require "network"

local function stringify (value)
   if type(value) == "string" then
      return "\"" .. value .. "\""
   elseif type(value) == "table" then
      local string = {}

      for i, element in ipairs (value) do
	 string[i] = stringify(element) .. ","
      end

      for k, element in pairs (value) do
	 if type(k) == "string" then
	    table.insert (string, k .. "=" .. stringify(element))
	 end
      end
      
      return "{" .. table.concat(string) .. "}"
   else
      return tostring(value)
   end
end

network.bookmark = function (values)
    local self, meta, path
    local pages, mime = {}, {}

    local function lookup (method, uri, version)
       print (string.format ("Retrieving cache entry for uri %s.", uri))

       network.pages[uri] = self.pages[uri]
       return network.pages[uri]
    end

    meta = {
       pages = {},
       mime = {},

       __index = function (self, key)
		    if key == "path" then
		       return path
		    elseif key == "pages" then
		       return meta.pages
		    elseif key == "mime" then
		       return meta.mime
		    else
		       return meta[key]
		    end
		 end,

       __newindex = function (self, key, value)
           if key == "path" then
	      local pageschunk, mimechunk
	      local n, m = 0, 0

	      -- Flush the current cache.

	      for key in pairs (pages) do
		 pages[key] = nil
		 network.pages[key] = nil
	      end

	      for key in pairs (mime) do
		 mime[key] = nil
		 network.mime[key] = nil
	      end
	      
	      -- Read in any prexisting indices from the
	      -- specified path.

	      pageschunk = loadfile (value .. "/pages.lua")
	      mimechunk = loadfile (value .. "/mime.lua")

	      for key, value in pairs (pageschunk and pageschunk() or {}) do
		 pages[key] = value
		 network.pages[key] = lookup

		 n = n + 1
	      end

	      for key, value in pairs (mimechunk and mimechunk() or {}) do
		 mime[key] = value
		 network.mime[key] = value

		 m = m + 1
	      end

	      print (string.format ("Read %d pages and %d mime types from %s.", n, m, value))
	      
	      path = value
	   elseif key == "pages" then
	   elseif key == "mime" then
	   else
	      meta[key] = value	      
	   end
	end,

       __tostring = function(self)
		       return "bookmark"
		    end
    }			   

    setmetatable (meta.pages, {
       __pairs = function (self)
		    return next, pages, nil
		 end,

        __index = function (self, key)
           if pages[key] then
	      local file, html

	      -- Return the cached entry.
	      
	      file = io.open (pages[key], "r")
	      html = file:read ("*a")
	      file:close()

	      return html
	   else
	      return nil
	   end
	end,
	
        __newindex = function (self, key, value)
	   local file, index
				     
	   if value then
	      -- Lookup the cache first.

	      if not pages[key] then
		 pages[key] = path .. "/html." .. os.time()

		 print (string.format ("Adding bookmark entry %s => %s.",
				       key, pages[key]))
	      else
		 print (string.format ("Modifying bookmark entry %s => %s.",
				       key, pages[key]))
	      end

	      network.pages[key] = lookup
					
	      -- Write the supplied page to a file.
	      
	      file = io.open (pages[key], "w")
	      file:write (value)
	      file:close()
	   elseif pages[key] then
	      print (string.format ("Removing bookmark entry %s => %s.",
				    key, pages[key]))

	      -- Remove the file itself.

	      os.remove (pages[key])

	      -- Remove the http server and cache entries.

	      network.pages[key] = nil
	      pages[key] = nil
	   end

	   -- Update the index.

	   index = "return {\n"
	   for key, name in pairs (pages) do
	      index = index .. string.format ('    ["%s"] = "%s",\n', key, name)
	   end
	   index = index .. "}\n"

	   file, _ = io.open (path .. "/pages.lua", "w")
	   file:write (index)
	   file:close ()
	end
     })

    setmetatable (meta.mime, {
       __pairs = function (self)
		    return next, mime, nil
		 end,

       __index = function (self, key)
		    return mime[key]
		 end,

       __newindex = function (self, key, value)
		       local index

		       mime[key] = value
		       network.mime[key] = value

		       -- Update the index.

		       index = "return {\n"
		       for key, value in pairs (mime) do
			  index = index .. string.format ('    ["%s"] = "%s",\n', key, value)
		       end
		       index = index .. "}\n"

		       file = io.open (path .. "/mime.lua", "w")
		       file:write (index)
		       file:close ()
		    end
    })

    self = setmetatable ({}, meta)

    for key, value in pairs(values) do
       self[key] = value
    end

    return self
end
