/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <GL/glu.h>
#include "transition.h"

@implementation Transition

-(Transition *) init
{
    char *list[] = {"duration", "speed", "progress"};
    
    self = [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->texture = 0;
    self->duration = 0;
    self->progress = 0;
    self->speed = 0;
    self->set = 0;
    
    return self;
}

-(void) toggle
{
    clock_gettime (CLOCK_REALTIME, &self->start);
    clock_gettime (CLOCK_REALTIME, &self->last);
    
    [super toggle];

    if (self->linked) {
	int viewport[4];
	GLenum error;

	glGetIntegerv (GL_VIEWPORT, viewport);
	
	glGetError();
	glActiveTexture(GL_TEXTURE0);
	glGenTextures(1, &self->texture);
	glBindTexture(GL_TEXTURE_2D, self->texture);
    
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

	glReadBuffer (GL_FRONT);
	glCopyTexImage2D (GL_TEXTURE_2D,
			  0, GL_RGB,
			  viewport[0], viewport[1],
			  viewport[2], viewport[3],
			  0);
	glReadBuffer (GL_BACK);
		
	error = glGetError();
	if(error != GL_NO_ERROR) {
	    printf ("Could not create transition texture (%s)\n",
		    (char *)gluErrorString(error));
	}
    } else {
	glDeleteTextures (1, &self->texture);
	self->texture = 0;
    }
}
    
-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "duration")) {
	if (self->set) {
	    lua_pushnumber (_L, self->duration);
	} else {
	    lua_pushnil (_L);
	}
    } else if (!xstrcmp(k, "progress")) {
	lua_pushnumber (_L, self->progress);
    } else if (!xstrcmp(k, "speed")) {
	lua_pushnumber (_L, self->speed);
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "duration")) {
	if (lua_tonumber (_L, 3)) {
	    self->duration = lua_tonumber (_L, -1);
	    self->set = 1;
	} else {
	    self->set = 0;
	}
    } else if (!xstrcmp(k, "progress")) {
	self->progress = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "speed")) {
	self->speed = lua_tonumber (_L, 3);
    } else {
	[super set];
    }
}

-(void) finish
{
    struct timespec time;
    double delta;

    clock_gettime (CLOCK_REALTIME, &time);

    if (self->set) {
	delta = time.tv_sec - self->start.tv_sec +
	        (time.tv_nsec - self->start.tv_nsec) * 1e-9;

	if (self->duration >= 0) {
	    self->progress = delta / self->duration;
	} else {
	    self->progress = 1 + delta / self->duration;
	}
    } else {
	delta = time.tv_sec - self->last.tv_sec +
	        (time.tv_nsec - self->last.tv_nsec) * 1e-9;

	self->progress += delta * self->speed;
    }

    clock_gettime (CLOCK_REALTIME, &self->last);

    [super finish];
}

@end
