-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "meshes"

local function append (a, b)
   local n = #a

   for i, v in ipairs(b) do
      a[n + i] = v
   end

   return a
end

meshes.sphere = function (rho, m, n)
   local dtheta = 1 / n
   local dphi = 1 / m

   local vertices = {}
   local indices = {}

   -- Calculate all vertices except for the poles.
   
   for j = 1, n - 1 do
      local theta = j * math.pi / n

      for i = 0, m do
	 local phi = 2 * i * math.pi / m;

	 append (vertices, {
		    rho * math.cos(phi) * math.sin(theta),
		    rho * math.sin(phi) * math.sin(theta),
		    rho * math.cos(theta),

		    math.cos(phi) * math.sin(theta),
		    math.sin(phi) * math.sin(theta),
		    math.cos(theta),

		    i / m,
		    j / n})
      end
   end
   
   -- Now string' em up into triangles.

   for j = 1, n - 2 do
      for i = 1, m do
	 append(indices, {
		   (j - 1) * (m + 1) + i,
		   (j - 1) * (m + 1) + i - 1 ,
		   j * (m + 1) + i - 1,

		   (j - 1) * (m + 1) + i,
		   j * (m + 1) + i - 1,
		   j * (m + 1) + i
		})
      end
   end

   -- And finally append the poles and cap the mesh.

   append (vertices, {
	      0, 0, rho,  0, 0, 1,  0.5, 0,
	      0, 0, -rho,  0, 0, -1,  0.5, 1
	   })

   for i = 1, m do
      append (indices, {
		 i - 1, i, (n - 1) * (m + 1),
		 (n - 2) * (m + 1) + i,
		 (n - 2) * (m + 1) + i - 1,
		 (n - 1) * (m + 1) + 1
	      })
   end

   return {
      size = {#vertices / 8, #indices};
      vertices = vertices;
      indices = indices;
   }
end

function meshes.box(a, b, c)
   a = a / 2
   b = b / 2
   c = c / 2

   return {
      size = {24, 36};

      vertices = {
	 -a,  b, c,  0, 0, 1,  0, 1,
	 a,  b, c,  0, 0, 1,  1, 1,
	 a, -b, c,  0, 0, 1,  1, 0,
	 -a, -b, c,  0, 0, 1,  0, 0,
	 
	 -a,  b, -c,  0, 0, -1,  0, 1,
	 a,  b, -c,  0, 0, -1,  1, 1,
	 a, -b, -c,  0, 0, -1,  1, 0,
	-a, -b, -c,  0, 0, -1,  0, 0,
	 
	 -a, b, c,  0, 1, 0,  0, 1,
	 a, b, c,  0, 1, 0,  1, 1,
	 a, b, -c,  0, 1, 0,  1, 0,
	 -a, b, -c,  0, 1, 0,  0, 0,
	 
	 -a, -b, c,  0, -1, 0,  0, 1,
	 a, -b, c,  0, -1, 0,  1, 1,
	 a, -b, -c,  0, -1, 0,  1, 0,
	 -a, -b, -c,  0, -1, 0,  0, 0,
	 
	 a, -b, c,  1, 0, 0,  0, 1,
	 a, b, c,  1, 0, 0,  1, 1,
	 a, b, -c,  1, 0, 0,  1, 0,
	 a, -b, -c,  1, 0, 0,  0, 0,
	 
	 -a, -b, c,  -1, 0, 0,  0, 1,
	 -a, b, c,  -1, 0, 0,  1, 1,
	 -a, b, -c,  -1, 0, 0,  1, 0,
	 -a, -b, -c,  -1, 0, 0,  0, 0
      };

      indices = {
	 0, 3, 1,
	 1, 3, 2,

	 4, 5, 7,
	 5, 6, 7,

	 8, 9, 11,
	 9, 10, 11,

	 12, 15, 13,
	 13, 15, 14,

	 16, 19, 17,
	 17, 19, 18,

	 20, 21, 23,
	 21, 22, 23
      };
   }
end

meshes.torus = function (rho, sigma, m, n)
   local vertices = {}
   local indices = {}
   local r, theta, phi

   for i = 0, n do
      for j = 0, m do
	 theta = 2 * math.pi * i / n
	 phi = 2 * math.pi * j / m

	 r = rho + sigma * math.cos(phi)

	 vertices[8 * (i * n + j) + 1] = r * math.cos(theta)
	 vertices[8 * (i * n + j) + 2] = r * math.sin(theta)
	 vertices[8 * (i * n + j) + 3] = sigma * math.sin(phi)

	 vertices[8 * (i * n + j) + 4] = math.cos (theta) * math.cos(phi)
	 vertices[8 * (i * n + j) + 5] = math.sin (theta) * math.cos(phi)
	 vertices[8 * (i * n + j) + 6] = math.sin (phi)

	 vertices[8 * (i * n + j) + 7] = 0
	 vertices[8 * (i * n + j) + 8] = 0

	 indices[6 * (i * n + j) + 1] = i * n + j + 1
	 indices[6 * (i * n + j) + 2] = (i + 1) * n + j + 1
	 indices[6 * (i * n + j) + 3] = (i + 1) * n + j + 2
	 
	 indices[6 * (i * n + j) + 4] = i * n + j + 1
	 indices[6 * (i * n + j) + 5] = (i + 1) * n + j + 2
	 indices[6 * (i * n + j) + 6] = i * n + j + 2
      end
   end

   return {
      size = {#vertices / 8, #indices};
      vertices = vertices;
      indices = indices;
   }
end
