/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "node.h"

static int rotation(lua_State *L)
{
    double theta, a[3], rotation[16];
    int i;

    luaL_checknumber (L, 1);
    luaL_checktype (L, 2, LUA_TTABLE);

    theta = lua_tonumber(L, 1);

    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	a[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    /* We let the GL calculate the inverse
       rotation to save us a transposition. */
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glRotated (-theta, a[0], a[1], a[2]);
    glGetDoublev(GL_MODELVIEW_MATRIX, rotation);
    glPopMatrix();

    lua_newtable(L);
    
    for(i = 0 ; i < 9 ; i += 1) {
	lua_pushnumber(L, rotation[i + (i / 3)]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int euler(lua_State *L)
{
    double x, y, z, rotation[16];
    int i;

    x = luaL_checknumber (L, 1);
    y = luaL_checknumber (L, 2);
    z = luaL_checknumber (L, 3);
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glRotated (-z, 0, 0, 1);
    glRotated (-y, 0, 1, 0);
    glRotated (-x, 1, 0, 0);
    glGetDoublev(GL_MODELVIEW_MATRIX, rotation);
    glPopMatrix();

    lua_newtable(L);
    
    for(i = 0 ; i < 9 ; i += 1) {
	lua_pushnumber(L, rotation[i + (i / 3)]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int relue(lua_State *L)
{
    double x, y, z, rotation[16];
    int i;

    x = luaL_checknumber (L, 1);
    y = luaL_checknumber (L, 2);
    z = luaL_checknumber (L, 3);
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glRotated (-x, 1, 0, 0);
    glRotated (-y, 0, 1, 0);
    glRotated (-z, 0, 0, 1);
    glGetDoublev(GL_MODELVIEW_MATRIX, rotation);
    glPopMatrix();

    lua_newtable(L);
    
    for(i = 0 ; i < 9 ; i += 1) {
	lua_pushnumber(L, rotation[i + (i / 3)]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int fromnode(lua_State *L)
{
    id object;
    float *R, b[3], w[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    R = [object orientation];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    w[0] = R[0] * b[0] + R[1] * b[1] + R[2] * b[2];
    w[1] = R[3] * b[0] + R[4] * b[1] + R[5] * b[2];
    w[2] = R[6] * b[0] + R[7] * b[1] + R[8] * b[2];
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int tonode(lua_State *L)
{
    id object;
    float *R, b[3], w[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    R = [object orientation];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    w[0] = R[0] * b[0] + R[3] * b[1] + R[6] * b[2];
    w[1] = R[1] * b[0] + R[4] * b[1] + R[7] * b[2];
    w[2] = R[2] * b[0] + R[5] * b[1] + R[8] * b[2];

    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int lookthrough(lua_State *L)
{
    id object;
    float *r, *R;
    GLdouble T[16];

    luaL_checktype (L, 1, LUA_TUSERDATA);

    object = *(id *)lua_touserdata (L, 1);

    r = [object translation];
    R = [object rotation];
    
    glMatrixMode (GL_MODELVIEW);
    glPushMatrix ();
    glLoadIdentity ();

    gluLookAt (r[0], r[1], r[2],
	       r[0] - R[2], r[1] - R[5], r[2] - R[8],
	       -R[0], -R[3], -R[6]);

    glGetDoublev (GL_MODELVIEW_MATRIX, T);

    glPopMatrix();    
    
    lua_newtable(L);
    
    lua_pushnumber (L, T[12]); lua_rawseti (L, -2, 1);
    lua_pushnumber (L, T[13]); lua_rawseti (L, -2, 2);
    lua_pushnumber (L, T[14]); lua_rawseti (L, -2, 3);

    lua_newtable(L);

    lua_pushnumber (L, T[0]); lua_rawseti (L, -2, 1);
    lua_pushnumber (L, T[4]); lua_rawseti (L, -2, 2);
    lua_pushnumber (L, T[8]); lua_rawseti (L, -2, 3);

    lua_pushnumber (L, T[1]); lua_rawseti (L, -2, 4);
    lua_pushnumber (L, T[5]); lua_rawseti (L, -2, 5);
    lua_pushnumber (L, T[9]); lua_rawseti (L, -2, 6);

    lua_pushnumber (L, T[2]); lua_rawseti (L, -2, 7);
    lua_pushnumber (L, T[6]); lua_rawseti (L, -2, 8);
    lua_pushnumber (L, T[10]); lua_rawseti (L, -2, 9);
    
    return 2;
}

static int toquaternion(lua_State *L)
{
    double T[9], r;
    int i;

    luaL_checktype (L, 1, LUA_TTABLE);

    for (i = 0 ; i < 9 ; i += 1) {
	lua_rawgeti (L, 1, i + 1);
	T[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    lua_newtable(L);
    
    if(1 + T[0] + T[4] + T[8] > 1e-6) {
	r = 2 * sqrt(1 + T[0] + T[4] + T[8]);

	lua_pushnumber (L, (T[3] - T[1]) / r);
	lua_rawseti(L, -2, 3);
	lua_pushnumber (L, (T[7] - T[5]) / r);
	lua_rawseti(L, -2, 1);
	lua_pushnumber (L, (T[2] - T[6]) / r);
	lua_rawseti(L, -2, 2);
	lua_pushnumber (L, 0.25 * r);
	lua_rawseti(L, -2, 4);
    } else if (T[0] > T[4] && T[0] > T[8]) {
	r = 2 * sqrt(1 + T[0] - T[4] - T[8]);

	lua_pushnumber (L, (T[2] + T[6]) / r);
	lua_rawseti(L, -2, 3);
	lua_pushnumber (L, 0.25 * r);
	lua_rawseti(L, -2, 1);
	lua_pushnumber (L, (T[3] + T[1]) / r);
	lua_rawseti(L, -2, 2);
	lua_pushnumber (L, (T[7] - T[5]) / r);
	lua_rawseti(L, -2, 4);
    } else if (T[4] > T[8]) {
	r = 2 * sqrt(1 - T[0] + T[4] - T[8]);

	lua_pushnumber (L, (T[7] + T[5]) / r);
	lua_rawseti(L, -2, 3);
	lua_pushnumber (L, (T[3] + T[1]) / r);
	lua_rawseti(L, -2, 1);
	lua_pushnumber (L, 0.25 * r);
	lua_rawseti(L, -2, 2);
	lua_pushnumber (L, (T[2] - T[6]) / r);
	lua_rawseti(L, -2, 4);
    }else {	
	r = 2 * sqrt(1 - T[0] - T[4] + T[8]);

	lua_pushnumber (L, 0.25 * r);
	lua_rawseti(L, -2, 3);
	lua_pushnumber (L, (T[2] + T[6]) / r);
	lua_rawseti(L, -2, 1);
	lua_pushnumber (L, (T[7] + T[5]) / r);
	lua_rawseti(L, -2, 2);
	lua_pushnumber (L, (T[3] - T[1]) / r);
	lua_rawseti(L, -2, 4);
    }

    return 1;
}

static int fromquaternion(lua_State *L)
{
    double q[4], xx, xy, xz, xt, yy, yz, yt, zz, zt;
    int i;
    
    luaL_checktype (L, 1, LUA_TTABLE);

    for (i = 0 ; i < 4 ; i += 1) {
	lua_rawgeti (L, 1, i + 1);
	q[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    xx = q[0] * q[0];
    xy = q[0] * q[1];
    xz = q[0] * q[2];
    xt = q[0] * q[3];
    yy = q[1] * q[1];
    yz = q[1] * q[2];
    yt = q[1] * q[3];
    zz = q[2] * q[2];
    zt = q[2] * q[3];

    lua_newtable (L);
    lua_pushnumber (L,  1 - 2 * (yy + zz));
    lua_rawseti(L, -2, 1);
    lua_pushnumber (L,      2 * (xy - zt));
    lua_rawseti(L, -2, 2);
    lua_pushnumber (L,      2 * (xz + yt));
    lua_rawseti(L, -2, 3);
    lua_pushnumber (L,      2 * (xy + zt));
    lua_rawseti(L, -2, 4);
    lua_pushnumber (L,  1 - 2 * (xx + zz));
    lua_rawseti(L, -2, 5);
    lua_pushnumber (L,      2 * (yz - xt));
    lua_rawseti(L, -2, 6);
    lua_pushnumber (L,      2 * (xz - yt));
    lua_rawseti(L, -2, 7);
    lua_pushnumber (L,      2 * (yz + xt));
    lua_rawseti(L, -2, 8);
    lua_pushnumber (L,  1 - 2 * (xx + yy));    
    lua_rawseti(L, -2, 9);

    return 1;
}

static int fromaxes(lua_State *L)
{
    double u[3], v[3];
    int i;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TTABLE);

    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 1, i + 1);
	u[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	v[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }
    
    lua_newtable(L);

    /* Left. */
    
    lua_pushnumber (L, u[1] * v[2] - u[2] * v[1]);
    lua_rawseti (L, -2, 1);

    lua_pushnumber (L, u[2] * v[0] - u[0] * v[2]);
    lua_rawseti (L, -2, 4);

    lua_pushnumber (L, u[0] * v[1] - u[1] * v[0]);
    lua_rawseti (L, -2, 7);

    /* Up. */
    
    lua_pushnumber (L, u[0]);
    lua_rawseti (L, -2, 2);

    lua_pushnumber (L, u[1]);
    lua_rawseti (L, -2, 5);

    lua_pushnumber (L, u[2]);
    lua_rawseti (L, -2, 8);

    /* Viewing. */
    
    lua_pushnumber (L, v[0]);
    lua_rawseti (L, -2, 3);

    lua_pushnumber (L, v[1]);
    lua_rawseti (L, -2, 6);

    lua_pushnumber (L, v[2]);
    lua_rawseti (L, -2, 9);

    return 1;
}

int luaopen_transforms (lua_State *L)
{
    const luaL_Reg transforms[] = {
	{"rotation", rotation},
	{"euler", euler},
	{"relue", relue},
	{"fromnode", fromnode},
	{"tonode", tonode},
	{"lookthrough", lookthrough},
	{"fromaxes", fromaxes},
	{"fromquaternion", fromquaternion},
	{"toquaternion", toquaternion},
	{NULL, NULL}
    };

    luaL_register (L, "transforms", transforms);

    return 0;
}
