/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <AL/al.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#include <time.h>

#include "source.h"

static void callhooks (lua_State *L, void *key, int reference)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    lua_call(L, 1, 0);
	} else if (lua_istable (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushnil(L);
	
	    while(lua_next(L, -3)) {
		lua_pushlightuserdata (L, key);
		lua_gettable (L, -4);
		lua_call (L, 1, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Source

-(Source *) init
{
    [super init];

    self->finished = LUA_REFNIL;

    alGenSources (1, &self->index);
    alSourcef (self->index, AL_BUFFER, 0);
    alSourceQueueBuffers(self->index, 1, &self->name);

    return self;
}

-(Source *) initWithSamples: (ALshort *) samples
                     ofSize: (int) n
	        atFrequency: (int) f;
{
    ALenum error;
    ALuint buffer;

    [super init];
        
    alGetError();
    alGenBuffers(1, &buffer);
    alBufferData (buffer, AL_FORMAT_MONO16, samples, 2 * n, f);

    error = alGetError();
    if(error != AL_NO_ERROR) {
	printf ("Could not create source (%s).\n", alGetString(error));
    }
    
    self->name = buffer;
    self->size = n;
    self->frequency = f;
 
    return self;
}

-(void) freeBuffers
{
    alDeleteBuffers(1, &self->name);

    [super free];
}

-(void) free
{
    alDeleteSources (1, &self->index);
}

-(ALint) index
{
    return self->index;
}

-(int) length
{
    return self->size;
}

-(int) frequency
{
    return self->frequency;
}

-(void) toggle
{
    struct timespec time;
    
    if (!linked) {
	clock_gettime (CLOCK_REALTIME, &time);
    
	self->tick = time.tv_sec + time.tv_nsec / 1e9;
	alSourcePlay (self->index);
    } else {
	alSourcePause (self->index);
    }

    [super toggle];
}

-(void) transform
{
    ALenum newstate;
    ALfloat v[3];
    struct timespec time;
    double now, dt;
    float *r, *r_0;
    
    alGetSourcei (self->index, AL_SOURCE_STATE, &newstate);

    if (newstate != self->state && newstate == AL_STOPPED) {
	callhooks (_L, self, self->finished);
    }

    self->state = newstate;

    clock_gettime (CLOCK_REALTIME, &time);
    now = time.tv_sec + time.tv_nsec / 1e9;
    dt = now - self->tick;
    self->tick = now;
    
    r_0 = [self translation];

    [super transform];

    r = [self translation];
    
    v[0] = (r[0] - r_0[0]) / dt;
    v[1] = (r[1] - r_0[1]) / dt;
    v[2] = (r[2] - r_0[2]) / dt;

    alSourcefv (self->index, AL_POSITION, r);    
    alSourcefv (self->index, AL_VELOCITY, v);    
}

-(void) get
{    
    const char *k;
    
    k = lua_tostring (_L, -1);

    if (!xstrcmp(k, "gain")) {
	ALfloat gain;

	alGetSourcef (self->index, AL_GAIN, &gain);

	lua_pushnumber (_L, gain);
    } else if (!xstrcmp(k, "bounds")) {
	ALfloat gain;
	
        lua_newtable (_L);
	
	alGetSourcef (self->index, AL_MIN_GAIN, &gain);
	lua_pushnumber (_L, gain);
	lua_rawseti (_L, -2, 1);
        
	alGetSourcef (self->index, AL_MAX_GAIN, &gain);
	lua_pushnumber (_L, gain);
	lua_rawseti (_L, -2, 2);
    } else if (!xstrcmp(k, "reference")) {
	ALfloat distance;

	alGetSourcef (self->index, AL_REFERENCE_DISTANCE, &distance);
	lua_pushnumber (_L, distance);
    } else if (!xstrcmp(k, "rolloff")) {
	ALfloat factor;

	alGetSourcef (self->index, AL_ROLLOFF_FACTOR, &factor);
	lua_pushnumber (_L, factor);
    } else if (!xstrcmp(k, "pitch")) {
	ALfloat factor;

	alGetSourcef (self->index, AL_PITCH, &factor);
	lua_pushnumber (_L, factor);
    } else if (!xstrcmp(k, "state")) {
	if (self->state == AL_PLAYING) {
	    lua_pushliteral (_L, "playing");
	} else if (self->state == AL_PAUSED) {
	    lua_pushliteral (_L, "paused");
	} else if (self->state == AL_STOPPED || self->state == AL_INITIAL) {
	    lua_pushliteral (_L, "stopped");
	} else {
	    lua_pushliteral (_L, "huh?");
	}
    } else if (!xstrcmp(k, "finished")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->finished);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, -2);

    if (!xstrcmp(k, "gain")) {
	alSourcef (self->index, AL_GAIN, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "bounds")) {
	if (lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 1);
	    alSourcef (self->index, AL_MIN_GAIN, lua_tonumber (_L, -1));

	    lua_rawgeti (_L, 3, 2);
	    alSourcef (self->index, AL_MAX_GAIN, lua_tonumber (_L, -1));

	    lua_pop (_L, 2);
	}
    } else if (!xstrcmp(k, "reference")) {
	alSourcef (self->index, AL_REFERENCE_DISTANCE, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "rolloff")) {
	alSourcef (self->index, AL_ROLLOFF_FACTOR, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "pitch")) {
	alSourcef (self->index, AL_PITCH, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "finished")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->finished);
        self->finished = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	[super set];
    }
}

@end
