/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "fluid.h"

static int size[3] = {0, 0, 0};
static float *samples, scale = 1;

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int node_len(lua_State *L)
{
    int n;
    
    lua_getmetatable (L, 1);
    n = lua_objlen (L, -1);
    lua_pushnumber (L, n);
   
    return 1;
}

static int node_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [object name]);
   
    return 1;
}

static int node_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object get];

    return 1;
}

static int node_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object set];

    return 0;
}

static int node_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object free];

    return 0;
}

static int constructnode (lua_State *L)
{
    Class class;
    id object;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    /* Create the userdata... */

    object = [[class alloc] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, (lua_CFunction)node_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)node_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    /* Add a weak reference. */
    
    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int turbulence_index (lua_State *L)
{
    const char *k;
    
    k = lua_tostring(L, 2);

    if (!strcmp(k, "size")) {
	lua_newtable (L);

	/* Implement this at some point. */
    } else if (!strcmp(k, "scale")) {
	lua_pushnumber (L, scale);
    } else if (!strcmp(k, "samples")) {
	lua_newtable (L);

	/* Implement this at some point. */
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int turbulence_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "size")) {
	if(lua_istable(L, 3)) {
	    for(i = 0 ; i < 3 ; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		size[i] = lua_tonumber(L, -1);
                
		lua_pop(L, 1);
	    }
	} else {
	    size[0] = 0;
	    size[1] = 0;
	    size[2] = 0;
	}
    } else if (!strcmp(k, "scale")) {
	scale = lua_tonumber (L, 3);
    } else if (!strcmp(k, "samples")) {
	if (lua_istable(L, 3)) {
	    int n;
	    
	    n = lua_objlen(L, 3);
	    samples = (float *)realloc(samples, n * sizeof(float));

	    for(i = 0 ; i < n ; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		samples[i] = (float)lua_tonumber(L, -1);
		lua_pop(L, 1);
	    }
	}
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

void get_turbulence_at (double *r, double *v)
{
    double x_0, x_1, y_0, y_1, z_0, z_1;
    double r_0, r_1, r_2;
    int i, j;

    if (size[0] > 0 && size[1] > 0 && size[2] > 0) {
	r_0 = fmod (fabs(r[0]) / scale, size[0] - 1);
	r_1 = fmod (fabs(r[1]) / scale, size[1] - 1);
	r_2 = fmod (fabs(r[2]) / scale, size[2] - 1);

	x_1 = modf(r_0, &x_0);
	y_1 = modf(r_1, &y_0);
	z_1 = modf(r_2, &z_0);

	i = z_0 * (size[0] * size[1]) + y_0 * size[0] + x_0;

	/* Interpolate the turbulence latice at (r_0, r_1, r_2). */
		
	for (j = 0 ; j < 3 ; j += 1) {
	    double v_000, v_001, v_010, v_100;
	    double v_011, v_101, v_110, v_111;

	    v_000 = samples[(i) * 3 + j];
	    v_100 = samples[(i + 1) * 3 + j];
	    v_010 = samples[(i + size[1]) * 3 + j];
	    v_001 = samples[(i + size[0] * size[1]) * 3 + j];
	    v_110 = samples[(i + size[0] + 1) * 3 + j];
	    v_101 = samples[(i + size[0] * size[1] + 1) * 3 + j];
	    v_011 = samples[(i + size[0] * size[1] + size[0]) * 3 + j];
	    v_111 = samples[(i + size[0] * size[1] + size[0] + 1) * 3 + j];

	    v[j] = v_000 * (1 - x_1) * (1 - y_1) * (1 - z_1) +
		   v_100 * x_1 * (1 - y_1) * (1 - z_1) +
		   v_010 * (1 - x_1) * y_1 * (1 - z_1) +
		   v_001 * (1 - x_1) * (1 - y_1) * z_1 +
		   v_101 * x_1 * (1 - y_1) * z_1 +
		   v_011 * (1 - x_1) * y_1 * z_1 +
		   v_110 * x_1 * y_1 * (1 - z_1) +
		   v_111 * x_1 * y_1 * z_1;
	}
    } else {
	v[0] = 0;
	v[1] = 0;
	v[2] = 0;
    }
}
	    
int luaopen_particulate (lua_State *L)
{
    int i;
    
    Class nodes[] = {
	[Fluid class]
    };	

    /* Create the turbulence table. */
    
    lua_newtable (L);

    lua_newtable (L);
    lua_pushcfunction (L, turbulence_index);
    lua_setfield (L, -2, "__index");
    lua_pushcfunction (L, turbulence_newindex);
    lua_setfield (L, -2, "__newindex");
    lua_setmetatable (L, -2);
 
    lua_setglobal (L, "turbulence");

    /* ...and the main table. */
    
    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(nodes) / sizeof(nodes[0]) ; i += 1) {
	lua_pushlightuserdata (L, nodes[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([nodes[i] name])));
    }
 
    lua_setglobal (L, lua_tostring (L, 1));

    return 0;
}
