/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>

#include "static.h"

static int generic_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [object name]);
   
    return 1;
}

static int generic_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object get];
    
    return 1;
}

static int generic_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    
    [object set];

    return 0;
}

static int generic_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object free];

    return 0;
}

static int static_len(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [object vertices]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [object indices] / 3);
    lua_rawseti(L, -2, 2);

    return 1;
}

static int static_call (lua_State *L)
{
    id parent, child;

    parent = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */
	
    child = [[parent copy] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = child;
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, static_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, generic_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, child);
    lua_pushvalue (L, -3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int static_tostring(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [object name]);
    lua_pushstring(L, " geometry foundry");
    lua_concat(L, 2);
    
    return 1;
}

static int static_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object freeBuffers];
    [object free];

    return 0;
}

static int constructstatic(lua_State *L)
{
    id object;
    GLfloat *vertices;
    GLushort *indices;
    int i, size[2];
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "vertices");
    lua_gettable(L, 1);
    
    /* Get the vertices. */

    vertices = (GLfloat *)malloc(8 * size[0] * sizeof(GLfloat));

    for(i = 0 ; i < 8 * size[0] ; i += 1) {
	lua_rawgeti(L, -1, i + 1);
	    
	vertices[i] = (GLfloat)lua_tonumber(L, -1);
	
	lua_pop(L, 1);
    }

    lua_pushstring(L, "indices");
    lua_gettable(L, 1);

    /* And the indices. */

    indices = (GLushort *)malloc(size[1] * sizeof(GLushort));

    for(i = 0 ; i < size[1] ; i += 1) {
	lua_rawgeti(L, -1, i + 1);
	
	indices[i] = (GLushort)lua_tonumber(L, -1);
	
	lua_pop(L, 1);
    }

    /* Create and initialize the static userdata. */
    object = [[Static alloc] initWithVertices: vertices
	                andIndices: indices
	                ofSize: size];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, static_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, static_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, static_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    free(vertices);
    free(indices);

    return 1;
}

int luaopen_meshes (lua_State *L)
{
    const luaL_Reg meshes[] = {
	{"static", constructstatic},
	{NULL, NULL}
    };

    luaL_register (L, lua_tostring (L, 1), meshes);

    return 0;
}
