/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <time.h>
#include <getopt.h>

#include <ode/ode.h>

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include <AL/al.h>
#include <AL/alc.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include <X11/Xatom.h>

#include "node.h"

static id root;

static Display *display;
static Window window;
static ALCdevice *device;

static int iterate = 1;
static int frames, grabkeyboard, grabpointer, autorepeat;
static int focus, defocus, delete;
static float canvas[3];
static char *title;

static struct timespec once;
static double now, then;

static double timescale = 1, stepsize = 0.01;
static int iterations = 0;
static int collision = LUA_REFNIL;

static void callback (void *data, dGeomID a, dGeomID b)
{
    dContactGeom geoms[16];
    dContact contacts[16];
    dReal mu, epsilon, sigma, tau;
    int i, j, n;

    mu = 0.0;
    epsilon = 0.0;
    sigma = 0.0;
    tau = 1.0;
    
    if (dGeomIsSpace (a) || dGeomIsSpace (b)) {
	/* Recurse into subspaces as needed. */
	
	dSpaceCollide2 (a, b, data, &callback);

	if (dGeomIsSpace (a)) {
	    dSpaceCollide ((dSpaceID)a, data, &callback);
	}
      
	if (dGeomIsSpace (b)) {
	    dSpaceCollide ((dSpaceID)b, data, &callback);
	}
    } else {
	if (dGeomGetBody (a) || dGeomGetBody (b)) {	    
	    n = dCollide (a, b, 16, geoms, sizeof(dContactGeom));

	    for (i = 0, j = 0 ; i < n ; i += 1) {
		dBodyID p, q;
		dVector3 u, v, delta;
		dReal *r, *n, ndotdelta;

		/* Caclualte the relative velocity of the
		   contact point. */
		
		r = geoms[i].pos;
		n = geoms[i].normal;
		p = dGeomGetBody (geoms[i].g1);
		q = dGeomGetBody (geoms[i].g2);
		
		if (p) {
		    dBodyGetPointVel (p, r[0], r[1], r[2], u);
		} else {
		    dSetZero (u, 3);
		}
		
		if (q) {
		    dBodyGetPointVel (q, r[0], r[1], r[2], v);
		} else {
		    dSetZero (v, 3);
		}	

		dOP (delta, -, u, v);		
		ndotdelta = dDOT(n, delta);
		
/* 		if (dGeomGetClass (b) == dTriMeshClass || */
/* 		    dGeomGetClass (a) == dTriMeshClass) { */
/* 		    printf ("%d, %f\n", j, ndotdelta); */
/* 		    printf ("%f, %f, %f\n", n[0], n[1], n[2]); */
/* 		} */

		/* Only consider this contact if the
		   relative velocity points inward. */

		if (ndotdelta <= 0) {
		    contacts[j].geom = geoms[i];

		    contacts[j].fdir1[0] = delta[0] - ndotdelta * n[0];
		    contacts[j].fdir1[1] = delta[1] - ndotdelta * n[1];
		    contacts[j].fdir1[2] = delta[2] - ndotdelta * n[2];

		    dSafeNormalize3(contacts[j].fdir1);
		
		    j += 1;
		}
	    }

	    if (j > 0) {
		/* Create a collision event. */

		if (collision != LUA_REFNIL) {
		    int h_0, h_1;
		    
		    h_0 = lua_gettop (_L);
		    
		    lua_getfield (_L, LUA_REGISTRYINDEX, "userdata");
		    
		    lua_pushlightuserdata (_L, dGeomGetData(a));
		    lua_gettable (_L, -2);

		    lua_pushlightuserdata (_L, dGeomGetData(b));
		    lua_gettable (_L, -3);

		    lua_rawgeti (_L, LUA_REGISTRYINDEX, collision);

		    /* If it's a single function encapsulate it
		       in a table to unify the approach. */

		    if (!lua_istable (_L, -1)) {
			lua_newtable (_L);
			lua_insert (_L, -2);
			lua_rawseti (_L, -2, 1);
		    }

		    /* Call all bound functions. */
		    
		    lua_pushnil (_L);
		    h_1 = lua_gettop (_L);

		    while (lua_next (_L, -2) != 0) {
			lua_pushvalue (_L, -5);
			lua_pushvalue (_L, -5);

			lua_call (_L, 2, LUA_MULTRET);
		    
			if (lua_type (_L, h_1 + 1) == LUA_TNUMBER) {
			    j = j < lua_tonumber (_L, h_1 + 1) ?
				j : lua_tonumber (_L, h_1 + 1);
			}
			
			if (lua_type (_L, h_1 + 2) == LUA_TNUMBER) {
			    mu = lua_tonumber (_L, h_1 + 2) < dInfinity ?
				 lua_tonumber (_L, h_1 + 2) : dInfinity;
			}

			if (lua_type (_L, h_1 + 3) == LUA_TNUMBER) {
			    epsilon = lua_tonumber (_L, h_1 + 3);
			}

			if (lua_type (_L, h_1 + 4) == LUA_TNUMBER) {
			    sigma = lua_tonumber (_L, h_1 + 4);
			}
			
			if (lua_type (_L, h_1 + 5) == LUA_TNUMBER) {
			    tau = lua_tonumber (_L, h_1 + 5);
			}

			lua_settop (_L, h_1);
		    }
		    
		    lua_settop (_L, h_0);
		}
		    
		/* Create the collision joints. */
	    
		for (i = 0 ; i < j ; i += 1) {
		    dJointID joint;
		    
		    contacts[i].surface.mode = 0;
		    
		    if (mu > 0) {
			contacts[i].surface.mode |= dContactApprox1;
			contacts[i].surface.mode |= dContactFDir1;
			contacts[i].surface.mu = mu;
		    }

		    if (epsilon > 0) {
			contacts[i].surface.mode |= dContactBounce;
			contacts[i].surface.bounce = epsilon;
			contacts[i].surface.bounce_vel = 0.01;
		    }

		    if (sigma > 0) {
			contacts[i].surface.mode |= dContactSoftCFM;
			contacts[i].surface.soft_cfm = sigma;
		    }

		    if (tau < 1) {
			contacts[i].surface.mode |= dContactSoftERP;
			contacts[i].surface.soft_erp = tau;
		    }
		    
		    joint = dJointCreateContact (_WORLD, _GROUP, &contacts[i]);
		    dJointAttach (joint, dGeomGetBody (a), dGeomGetBody (b));
		}
	    }
	}
    }
}

static int test (lua_State *L)
{
    return 0;
}

static int panic(lua_State *L)
{
    printf("Hmmm, I just got this unprotected error from "
	   "the Lua interpreter:\n\"%s\"\n"
           "No point in prolonging this dreary existence I suppose...\n",
           lua_tostring(L, -1));

    if (autorepeat) {
	XAutoRepeatOn (display);
    }
    
    return 0;
}

static int topointer(lua_State *L)
{
    luaL_checkany(L, 1);
    lua_pushlightuserdata (L, (void *)lua_topointer(L, 1));
    
    return 1;
}

static Bool wait(Display *d, XEvent *e, XPointer arg) {
    return (e->type == MapNotify) && (e->xmap.window == (Window)arg);
}

static int root_tostring(lua_State *L)
{
    lua_pushstring(L, "root");
   
    return 1;
}

static int root_index(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    [object get];

    return 1;
}

static int root_newindex(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);
    [object set];

    return 0;
}

static int common_index (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "iterate")) {
        lua_pushboolean(L, iterate);
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int common_newindex (lua_State *L)
{
    const char *k;

    k = lua_tostring(L, 2);
    
    if (!strcmp(k, "iterate")) {
        iterate = lua_toboolean (L, 3);
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

static int dynamics_index (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "stepsize")) {
        lua_pushnumber(L, stepsize);
    } else if (!xstrcmp(k, "timescale")) {
        lua_pushnumber(L, timescale);
    } else if (!xstrcmp(k, "iterations")) {
        lua_pushnumber(L, iterations);
    } else if (!xstrcmp(k, "time")) {
	struct timespec now;    

	clock_gettime (CLOCK_REALTIME, &now);
    
        lua_pushnumber(L, 1e-9 * (now.tv_nsec - once.tv_nsec) +
		          now.tv_sec - once.tv_sec);
	
    } else if (!xstrcmp(k, "collision")) {
        lua_rawgeti(L, LUA_REGISTRYINDEX, collision);
    } else if (!xstrcmp(k, "tolerance")) {
	lua_newtable (L);
	lua_pushnumber (L, dWorldGetCFM (_WORLD));
	lua_rawseti (L, -2, 1);
	lua_pushnumber (L, dWorldGetERP (_WORLD));
	lua_rawseti (L, -2, 2);
    } else if (!xstrcmp(k, "popvelocity")) {
	lua_pushnumber(L, dWorldGetContactMaxCorrectingVel (_WORLD));
    } else if (!xstrcmp(k, "surfacelayer")) {
	lua_pushnumber(L, dWorldGetContactSurfaceLayer (_WORLD));
    } else if (!xstrcmp(k, "gravity")) {
	dVector3 gravity;

	dWorldGetGravity (_WORLD, gravity);
	
        lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, gravity[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int dynamics_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "stepsize")) {
        stepsize = lua_tonumber(L, 3);
    } else if (!xstrcmp(k, "timescale")) {
        timescale = lua_tonumber(L, 3);
    } else if (!xstrcmp(k, "iterations")) {
        iterations = lua_tonumber(L, 3);

	dWorldSetQuickStepNumIterations (_WORLD, iterations);
    } else if (!xstrcmp(k, "collision")) {
        luaL_unref(L, LUA_REGISTRYINDEX, collision);
        collision = luaL_ref(L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "time")) {
    } else if (!xstrcmp(k, "tolerance")) {
	lua_rawgeti (L, 3, 1);
	dWorldSetCFM (_WORLD, lua_tonumber (L, -1));
	lua_rawgeti (L, 3, 2);
	dWorldSetERP (_WORLD, lua_tonumber (L, -1));
	lua_pop(L, 2);
    } else if (!xstrcmp(k, "popvelocity")) {
	dWorldSetContactMaxCorrectingVel (_WORLD, lua_tonumber(L, 3));
    } else if (!xstrcmp(k, "surfacelayer")) {
	dWorldSetContactSurfaceLayer (_WORLD, lua_tonumber(L, 3));
    } else if (!xstrcmp(k, "gravity")) {
	dReal gravity[3];
	
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                gravity[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }

	dWorldSetGravity (_WORLD,
			  gravity[0],
			  gravity[1],
			  gravity[2]);
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

static int graphics_index (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!xstrcmp(k, "window")) {
	XWindowAttributes attributes;


	XGetWindowAttributes (display, window, &attributes);
	if (attributes.map_state == IsViewable) {
	    lua_newtable (L);
	    lua_pushnumber (L, attributes.width);
	    lua_rawseti (L, -2, 1);
	    lua_pushnumber (L, attributes.height);
	    lua_rawseti (L, -2, 2);
	} else {
	    lua_pushboolean (L, 0);
	}
    } else if (!xstrcmp(k, "title")) {
	lua_pushstring (L, title);
    } else if (!xstrcmp(k, "canvas")) {
	lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, canvas[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "perspective") ||
	!xstrcmp(k, "orthographic")) {
	lua_gettable (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "frames")) {
	lua_pushnumber (L, frames);
    } else if (!xstrcmp(k, "grabpointer")) {
	lua_pushboolean (L, grabpointer);
    } else if (!xstrcmp(k, "grabkeyboard")) {
	lua_pushboolean (L, grabkeyboard);
/*     } else if (!xstrcmp(k, "pointer")) { */
/* 	Window root, child; */
/* 	unsigned int mask; */
/* 	int x, y, u, v; */

/* 	XQueryPointer (display, window, &root, &child, &u, &v, &x, &y, &mask); */

/* 	lua_newtable (L); */
/* 	lua_pushnumber (L, x); */
/* 	lua_rawseti (L, -2, 1); */
/* 	lua_pushnumber (L, y); */
/* 	lua_rawseti (L, -2, 2);	 */
    } else if (!xstrcmp(k, "colorbuffer")) {
        char *pixels;
        int v[4];

	glGetIntegerv (GL_VIEWPORT, v);
	pixels = malloc(v[2] * v[3] * 3);

	glReadBuffer (GL_BACK);
	glReadPixels(v[0], v[1], v[2], v[3], GL_RGB, GL_UNSIGNED_BYTE, pixels);

        lua_pushlstring(L, pixels, v[2] * v[3] * 3);

        free(pixels);
    } else if (!xstrcmp(k, "focus")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, focus);
    } else if (!xstrcmp(k, "defocus")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, defocus);
    } else if (!xstrcmp(k, "close")) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, delete);
    } else if (!xstrcmp(k, "vendor")) {
	lua_pushstring(L, (const char *)glGetString(GL_VENDOR));
    } else if (!xstrcmp(k, "renderer")) {
	lua_pushstring(L, (const char *)glGetString(GL_RENDERER));
    } else if (!xstrcmp(k, "version")) {
	lua_pushstring(L, (const char *)glGetString(GL_VERSION));
    } else if (!xstrcmp(k, "extensions")) {
	lua_pushstring(L, (const char *)glGetString(GL_EXTENSIONS));
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int graphics_newindex (lua_State *L)
{
    const char *k;
    int i;

    k = lua_tostring(L, 2);
    
    if (!xstrcmp(k, "window")) {
	XSizeHints *hints;
	XEvent event;
	int w[2];
	
	XUnmapWindow (display, window);
	
	if (lua_toboolean (L, -1)) {
	    Screen *screen;
	    Atom state, normal, fullscreen;
	    
	    for (i = 0 ; i < 2 ; i += 1) {
		lua_rawgeti (L, 3, i + 1);
		w[i] = lua_tonumber(L, -1);
		lua_pop (L, 1);
	    }

	    screen = DefaultScreenOfDisplay (display);
	    state = XInternAtom (display, "_NET_WM_STATE", False);
	    normal = XInternAtom (display, "_NET_WM_STATE_NORMAL", False);
	    fullscreen = XInternAtom (display,
				      "_NET_WM_STATE_FULLSCREEN",
				      False);

	    hints = XAllocSizeHints();
	    hints->flags = PMinSize | PMaxSize;
	    hints->min_width = w[0];
	    hints->min_height = w[1];
	    hints->max_width = w[0];
	    hints->max_height = w[1];

	    XSetWMNormalHints (display, window, hints);
	    XFree (hints);

	    XResizeWindow(display, window, w[0], w[1]);

	    XMapWindow (display, window);
	    XIfEvent(display, &event, wait, (XPointer)window);

	    if (w[0] == WidthOfScreen(screen) &&
		w[1] == HeightOfScreen(screen)) {
		XChangeProperty (display, window,
				 state, XA_ATOM, 32,
				 PropModeReplace,
				 (unsigned char *)&fullscreen, 1);
	    } else {
		XChangeProperty (display, window,
				 state, XA_ATOM, 32,
				 PropModeReplace,
				 (unsigned char *)&normal, 1);
	    }

	    if (grabkeyboard) {
		XGrabKeyboard (display, window, True,
			       GrabModeAsync, GrabModeAsync,
			       CurrentTime);
	    }

	    if (grabpointer) {
		XGrabPointer (display, window, True, 0,
			      GrabModeAsync, GrabModeAsync,
			      window, None, CurrentTime);
	    }
	}
	
	lua_settable (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "title")) {
	Atom name;

	title =(char *) lua_tostring (L, 3);
	name = XInternAtom (display, "WM_NAME", False);

	XChangeProperty (display, window,
			 name, XA_STRING, 8,
			 PropModeReplace,
			 (unsigned char *)title, strlen (title));
    } else if (!xstrcmp(k, "perspective")) {
	GLdouble p[6];

	for (i = 0 ; i < 6 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    p[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
 	glFrustum (p[0], p[1], p[2], p[3], p[4], p[5]);

	lua_settable (L, LUA_REGISTRYINDEX);

	lua_pushnil (L);
	lua_setfield (L, LUA_REGISTRYINDEX, "orthographic");
    } else if (!xstrcmp(k, "orthographic")) {
	GLdouble p[4];

	for (i = 0 ; i < 4 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    p[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
 	gluOrtho2D (p[0], p[1], p[2], p[3]);
	
	lua_settable (L, LUA_REGISTRYINDEX);

	lua_pushnil (L);
	lua_setfield (L, LUA_REGISTRYINDEX, "perspective");
    } else if (!xstrcmp(k, "canvas")) {
	for (i = 0 ; i < 3 ; i += 1) {
	    lua_rawgeti (L, 3, i + 1);
	    canvas[i] = lua_tonumber(L, -1);
	    lua_pop (L, 1);
	}

 	glClearColor (canvas[0], canvas[1], canvas[2], 0);
    } else if (!xstrcmp(k, "focus")) {
	luaL_unref (L, LUA_REGISTRYINDEX, focus);
	focus = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "defocus")) {
	luaL_unref (L, LUA_REGISTRYINDEX, defocus);
	defocus = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "close")) {
	luaL_unref (L, LUA_REGISTRYINDEX, delete);
	delete = luaL_ref (L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "grabpointer")) {
	grabpointer = lua_toboolean(L, 3);

	if(window && grabpointer) {
	    XGrabPointer (display, window, True, 0,
			  GrabModeAsync, GrabModeAsync,
			  window, None, CurrentTime);
	} else {
	    XUngrabPointer (display, CurrentTime);
	}
    } else if (!xstrcmp(k, "grabkeyboard")) {
	grabkeyboard = lua_toboolean(L, 3);

	if(window && grabkeyboard) {
	    XGrabKeyboard (display, window, True,
			   GrabModeAsync, GrabModeAsync,
			   CurrentTime);
	} else {
	    XUngrabKeyboard (display, CurrentTime);
	}
/*     } else if (!xstrcmp(k, "pointer")) { */
/* 	/\* Warp the pointer somehow. *\/ */
    } else {
	lua_rawset (L, 1);
    }
    
    return 0;
}

int main(int argc, char **argv)
{
    GLXContext graphics;
    XVisualInfo *info;
    Colormap colormap;
    XSetWindowAttributes attributes;
    XKeyboardState keyboard;
    Atom atom;

    ALCcontext *audio;
    ALenum error;
	
    struct timespec time;    
    double t, dt_0, dt;

    char *script = "~/.techne";
    int option;
    
    int configuration[] = {
	GLX_RGBA,
	GLX_DOUBLEBUFFER,
	GLX_RED_SIZE, 1,
	GLX_GREEN_SIZE, 1,
	GLX_BLUE_SIZE, 1,
	GLX_ALPHA_SIZE, 1,
	GLX_DEPTH_SIZE, 12,
	GLX_STENCIL_SIZE, 1,
	None
    };

    /* Create an ODE world. */

    dInitODE();
    
    _WORLD = dWorldCreate();
    _GROUP = dJointGroupCreate (0);
    _SPACE = dSimpleSpaceCreate (NULL);

    dWorldSetAutoDisableFlag (_WORLD, 0);
    
    int parameters[] = {0, 0, 0, 0, 0};

    static struct option options[] = {
	{"option", 1, 0, 'O'},
	{"red", 1, 0, 'r'},
	{"green", 0, 0, 'g'},
	{"blue", 1, 0, 'b'},
	{"alpha", 1, 0, 'a'},
	{"depth", 1, 0, 'z'},
	{"stencil", 1, 0, 's'},
	{"frequency", 1, 0, 'f'},
	{"interval", 1, 0, 'i'},
	{"configuration", 1, 0, 'c'},
	{"help", 0, 0, 'h'},
	{0, 0, 0, 0}
    };

    printf ("Greetings fellow user, this is Techne %s.\n", VERSION);
    clock_gettime (CLOCK_REALTIME, &once);
    
    /* Create the Lua state. */

    _L = luaL_newstate();
    
    lua_atpanic (_L, panic);

    luaL_openlibs (_L);
    lua_settop (_L, 0);

    lua_pushcfunction (_L, test);
    lua_setfield (_L, LUA_GLOBALSINDEX, "test");

    lua_pushcfunction (_L, topointer);
    lua_setfield (_L, LUA_GLOBALSINDEX, "topointer");

    /* Create the userdata reference table. */

    lua_newtable (_L);
    lua_newtable (_L);

    lua_pushstring (_L, "__mode");
    lua_pushstring (_L, "v");
    lua_settable (_L, -3);
    lua_setmetatable (_L, -2);
	    
    lua_pushvalue (_L, -1);
    lua_setfield (_L, LUA_REGISTRYINDEX, "userdata");
    
    /* Modify the Lua path and cpath. */

    lua_getglobal (_L, "package");
    lua_getfield (_L, -1, "path");
    lua_pushstring (_L, ";"PKGDATADIR"/common/?.lua;"PKGDATADIR"/common/?.lc");
    lua_concat (_L, 2);
    lua_setfield (_L, -2, "path");
    lua_pop (_L, 1);

    lua_getglobal (_L, "package");
    lua_getfield (_L, -1, "cpath");
    lua_pushstring (_L, ";"PKGLIBDIR"/?.so");
    lua_concat (_L, 2);
    lua_setfield (_L, -2, "cpath");
    lua_pop (_L, 1);
    
    /* Create the options table. */

    lua_newtable (_L);
    
    /* Parse the command line. */
    
    while ((option = getopt_long (argc, argv,
				  "O:r:g:b:a:z:s:f:i:c:h",
				  options, 0)) != -1) {
	if (option == 'O') {
	    lua_pushstring (_L, optarg);
	    lua_pushboolean (_L, 1);
	    lua_settable (_L, -3);
	} else if (option == 'r') {
	    configuration[3] = atoi(optarg);
	} else if (option == 'g') {
	    configuration[5] = atoi(optarg);
	} else if (option == 'b') {
	    configuration[7] = atoi(optarg);
	} else if (option == 'a') {
	    configuration[9] = atoi(optarg);
	} else if (option == 'z') {
	    configuration[11] = atoi(optarg);
	} else if (option == 's') {
	    configuration[13] = atoi(optarg);
	} else if (option == 'f') {
	    if (parameters[0] == 0) {
		parameters[0] = ALC_FREQUENCY;
		parameters[1] = atoi(optarg);
	    } else {
		parameters[2] = ALC_FREQUENCY;
		parameters[3] = atoi(optarg);
	    }
	} else if (option == 'i') {
	    if (parameters[0] == 0) {
		parameters[0] = ALC_REFRESH;
		parameters[1] = atoi(optarg);
	    } else {
		parameters[2] = ALC_REFRESH;
		parameters[3] = atoi(optarg);
	    }
	    
	} else if (option == 'c') {
	    script = optarg;
	} else if (option == 'h') {
	    printf ("Usage: %s [OPTION...]\n\n"
		    "Options:\n"
		    "  -h, --help                        "
		    "Display this help message.\n\n"
		    "General options:\n"
                    "  -c FILE, --configuration=FILE     "
		    "Specify the configuration script.\n"
		    "  -O OPTION, --option=OPTION        "
		    "Set the option OPTION.  (Consult the\n"
		    "                                    "
		    "documentation for a list of options\n"
		    "                                    "
		    "and their meaning.)\n\n"
		    "Audio related options:\n"
                    "  -f FREQ, --frequency=FREQ         "
		    "Specify the audio sampling rate in Hz.\n"
                    "  -i INT, --interval=INT            "
		    "Specify the audio refresh interval in Hz.\n\n"
		    "Video related options:\n"
                    "  -r BITS, --red=BITS               "
		    "Set the red color component depth.\n"
                    "  -g BITS, --green=BITS             "
		    "Set the green color component depth.\n"
                    "  -b BITS, --blue=BITS              "
		    "Set the blue color component depth.\n"
                    "  -a BITS, --alpha=BITS             "
		    "Set the alpha color component depth.\n"
                    "  -s BITS, --stencil=BITS           "
		    "Set the stencil buffer depth.\n"
                    "  -z BITS, --depth=BITS             "
		    "Set the Z-buffer depth.\n", argv[0]);

	    exit(1);
	} else {
	    exit(1);
	}
    }

    lua_setglobal (_L, "options");

    /* Create the root node. */

    root = [[Node alloc] init];
    *(id *)lua_newuserdata (_L, sizeof(id)) = root;
    
    lua_newtable (_L);
    lua_pushstring (_L, "__string");
    lua_pushcfunction (_L, (lua_CFunction)root_tostring);
    lua_settable (_L, -3);
    lua_pushstring (_L, "__index");
    lua_pushcfunction (_L, (lua_CFunction)root_index);
    lua_settable (_L, -3);
    lua_pushstring (_L, "__newindex");
    lua_pushcfunction (_L, (lua_CFunction)root_newindex);
    lua_settable (_L, -3);
    lua_setmetatable (_L, -2);
    
    [root toggle];

    lua_setglobal (_L, "graph");

    /* Create the common table. */
    
    lua_newtable (_L);    
    lua_newtable (_L);

    lua_pushcfunction (_L, common_index);
    lua_setfield (_L, -2, "__index");

    lua_pushcfunction (_L, common_newindex);
    lua_setfield (_L, -2, "__newindex");

    lua_setmetatable (_L, -2);
    lua_setglobal (_L, "common");

    /* Create the graphics table. */
    
    lua_newtable (_L);    
    lua_newtable (_L);

    lua_pushcfunction (_L, graphics_index);
    lua_setfield (_L, -2, "__index");

    lua_pushcfunction (_L, graphics_newindex);
    lua_setfield (_L, -2, "__newindex");

    lua_setmetatable (_L, -2);
    lua_setglobal (_L, "graphics");

    /* Create the dynamics table. */
    
    lua_newtable (_L);    
    lua_newtable (_L);
    
    lua_pushcfunction (_L, dynamics_index);
    lua_setfield (_L, -2, "__index");

    lua_pushcfunction (_L, dynamics_newindex);
    lua_setfield (_L, -2, "__newindex");

    lua_setmetatable (_L, -2);
    lua_setglobal (_L, "dynamics");

    /* Create the rendering context and
       associated visual. */

    display = XOpenDisplay(0);

    XGetKeyboardControl (display, &keyboard);
    autorepeat = keyboard.global_auto_repeat;
    
    info = glXChooseVisual(display, DefaultScreen(display), configuration);

    if (!info) {
	fprintf (stderr, "I could not find a suitable visual.\n");
	exit(1);
    }
    
    graphics = glXCreateContext(display, info, 0, GL_TRUE);

    if (!graphics) {
	fprintf (stderr, "I could not create a rendering context.\n");
	exit(1);
    }

    /* Create the window. */

    colormap = XCreateColormap(display, RootWindow(display, info->screen),
			       info->visual, AllocNone);
    
    attributes.colormap = colormap;
    attributes.border_pixel = 0;
    attributes.event_mask = StructureNotifyMask | FocusChangeMask;

    window = XCreateWindow(display,
			   RootWindow(display, info->screen),
			   0, 0, 640, 480,
			   0, info->depth, InputOutput, info->visual,
			   CWBorderPixel | CWColormap | CWEventMask,
			   &attributes);
    
    atom = XInternAtom(display, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(display, window, &atom, 1);
    glXMakeCurrent(display, window, graphics);

    /* Create an openal context. */

    alGetError();

#if 1
    device = alcOpenDevice (NULL);
    error = alGetError();

    if (error) {
	fprintf (stderr, "I could not open the audio device.\n");
    }
    
    audio = alcCreateContext (device, parameters);
    error = alGetError();

    if (error) {
	fprintf (stderr, "I could not create the audio context.\n");
    }

    alcMakeContextCurrent(audio);
    alcProcessContext(audio);
#endif

    /* Print useful debug information. */
    
    {
	int r, g, b, a, z, s;

	glXGetConfig (display, info, GLX_RED_SIZE, &r);
	glXGetConfig (display, info, GLX_GREEN_SIZE, &g);
	glXGetConfig (display, info, GLX_BLUE_SIZE, &b);
	glXGetConfig (display, info, GLX_ALPHA_SIZE, &a);
	glXGetConfig (display, info, GLX_DEPTH_SIZE, &z);
	glXGetConfig (display, info, GLX_STENCIL_SIZE, &s);

	printf ("\nLet's see what kind of junk we're dealing with here.\n");
	printf ("The graphics renderer is a '%s %s'.\n",
		glGetString(GL_RENDERER),
		glGetString(GL_VERSION));

	printf ("The framebuffer configuration is "
		"[%d, %d, %d, %d, %d, %d].\n", r, g, b, a, z, s);

	printf ("The rendering context is%sdirect.\n",
		glXIsDirect (display, graphics) == True ? " " : " *not* ");
    
	printf ("The audio renderer is '%s %s'.\n",
		alGetString(AL_RENDERER),
		alGetString(AL_VERSION));
    }

    printf ("\nWe're up and running!\n"
	    "Reading the configuration script.\n");
    
    /* Run the main configuration script. */
    
    if(!luaL_loadfile (_L, script)) {
	lua_call (_L, 0, 0);
    } else {
	lua_error (_L);
    }

    printf ("\nDone configuring, entering the main loop.\n");
    
    /* Enter the main loop. */
    
    do {
	XEvent event;

	while (XCheckTypedEvent (display, ClientMessage, &event) || 
	       XCheckMaskEvent(display,
			       StructureNotifyMask | FocusChangeMask,
			       &event)) {
	    int h;

	    h = lua_gettop (_L);
	    
	    if(event.type == ConfigureNotify) {
		glViewport (0, 0,
			    event.xconfigure.width,
			    event.xconfigure.height);
	    } else if (event.type == FocusIn) {
		lua_rawgeti (_L, LUA_REGISTRYINDEX, focus);

		XAutoRepeatOff (display);
	    } else if (event.type == FocusOut) {
		lua_rawgeti (_L, LUA_REGISTRYINDEX, defocus);

		if (autorepeat) {
		    XAutoRepeatOn (display);
		}
	    } else if (event.type == ClientMessage &&
		       event.xclient.data.l[0] == atom) {
		lua_rawgeti (_L, LUA_REGISTRYINDEX, delete);
	    }
	

	    if (lua_gettop (_L) > h) {
		if (lua_isfunction (_L, -1)) {
		    lua_call (_L, 0, 0);
		} else if (lua_istable (_L, -1)) {
		    lua_pushnil (_L);
		    while (lua_next (_L, -2) != 0) {
			lua_call (_L, 0, 0);
		    }
		    
		    lua_pop (_L, 1);
		} else {
		    lua_pop (_L, 1);
		}
	    }
	}
	
	glClear(GL_DEPTH_BUFFER_BIT |
	      	GL_COLOR_BUFFER_BIT |
	        GL_STENCIL_BUFFER_BIT);
	
	/* Simulate the system forward. */

	clock_gettime (CLOCK_REALTIME, &time);   
	now = time.tv_sec - once.tv_sec + (time.tv_nsec - once.tv_nsec) / 1e9;

	if (then == 0) {
	    then = now;
	}

	for (t = 0,
	     dt_0 = timescale * (now - then),
	     dt = dt_0;
	     dt > 0;
	     t += stepsize, dt = dt_0 - t) {

	    /* Empty the contact group. */
	    
	    dJointGroupEmpty (_GROUP);

	    if (dSpaceGetNumGeoms (_SPACE) > 1) {
		dSpaceCollide (_SPACE, NULL, callback);
	    } else if (dSpaceGetNumGeoms (_SPACE) > 0) {
		dGeomID geom;

		geom = dSpaceGetGeom (_SPACE, 0);

		if (dGeomIsSpace (geom)) {
		    dSpaceCollide ((dSpaceID)geom, NULL, callback);
		}
	    }
	    
	    [root stepBy: stepsize];

	    if (iterations > 0) {
		dWorldQuickStep (_WORLD, stepsize);
	    } else {
		dWorldStep (_WORLD, stepsize);
	    }
	}

	then = now;

	/* Traverse the scene. */

	[root transform];
	[root prepare];
	[root traversePass: 1];
	[root traversePass: 2];
	[root cleanup];

	glXSwapBuffers(display, window);
	frames += 1;
    } while (iterate);

    /* Destroy the audio context. */
    
    if (audio) {
	alcDestroyContext (audio);
    }

    if (autorepeat) {
	XAutoRepeatOn (display);
    }

    /* Finish cleaning up. */
    
    lua_close (_L);
    XCloseDisplay (display);
    
    puts ("Adieu!");
    return 0;
}
