/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <math.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <assert.h>

#include "atmosphere.h"
#include "land.h"

static GLuint name;
static const GLchar *vertexSource = 
"varying vec3 vertex;   	     	     				  \n"
" 									  \n"
"void main() 			   		     			  \n"
"{		 		   		     			  \n"
"    vertex = vec3(gl_ModelViewMatrix * gl_Vertex);			  \n"
"               							  \n"
"    gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex;		  \n"
"    gl_TexCoord[0].s = dot(gl_ObjectPlaneS[0], gl_Vertex);	   	  \n"
"    gl_TexCoord[0].t = dot(gl_ObjectPlaneT[0], gl_Vertex);	   	  \n"
"}		 		   		     			  \n";

static const GLchar *fragmentSource =
"uniform sampler2D sampler, detailSampler;				  \n"
"uniform float turbidity, factor, beta_p;				  \n"
"uniform vec3 sunDirection, sunColor, beta_r;	  		          \n"
"				   		     			  \n"
"varying vec3 vertex;   	     	     				  \n"
"				   		     			  \n"
"void main()                            	     			  \n"
"{	                                	     			  \n"
"    vec3 texel, tau;				                          \n"
"    float cosine, phase_r, phase_p, rho;		                  \n"
"			   		     				  \n"
"    cosine = dot(normalize(vertex), sunDirection);  			  \n"
"    texel = factor * vec3(texture2D(sampler, gl_TexCoord[0].st));        \n"
"			   		     				  \n"
"    phase_r = 0.059683 * (1.0 + cosine * cosine);		          \n"
"			   		     				  \n"
"    tau = exp(beta_r * vertex.z);  				          \n"
"    rho = exp((0.6544 * turbidity - 0.6510) * beta_p * vertex.z);        \n"
"		   		     				  	  \n"
"    gl_FragColor = vec4(sunColor * (50.0 * phase_r * (1.0 - tau) + texel * tau), rho);\n"
"    //gl_FragColor = vec4((vec3(cosine)), 1.0);  \n"
"}                                      	     			  \n";

@implementation Land

-(Land *) init
{
    if (!name) {
	GLuint vertex, fragment;
	int p, m, n;

	name = glCreateProgramObjectARB();
    
	vertex = glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB);
	glShaderSourceARB(vertex, 1,
			  (const GLchar **)&vertexSource, NULL);
	
	glCompileShaderARB(vertex);
	
	fragment = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);
	glShaderSourceARB(fragment, 1,
			  (const GLchar **)&fragmentSource, NULL);
	
	glCompileShaderARB(fragment);

	glAttachObjectARB(name, vertex);
	glAttachObjectARB(name, fragment);
	glLinkProgramARB (name);
    
	glGetObjectParameterivARB (name,
				   GL_OBJECT_LINK_STATUS_ARB,
				   &p);

	if (p != GL_TRUE) {
	    printf ("\nThe program for %s nodes did not "
		    "build properly.\n", [self name]);
	}

	glGetObjectParameterivARB(vertex,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(vertex, n, &m, buffer);
	    printf ("Info log for the `%s' vertex source follows:\n\n%s\n",
		    [self name], buffer);
	}

	glGetObjectParameterivARB(fragment,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(fragment, n, &m, buffer);
	    printf ("Info log for the `%s' fragment source follows:\n\n%s\n",
		    [self name], buffer);
	}

	glGetObjectParameterivARB(name,
				  GL_OBJECT_INFO_LOG_LENGTH_ARB,
				  &n);

	if (n > 1) {
	    char buffer[n];

	    glGetInfoLogARB(name, n, &m, buffer);
	    printf ("Info log for the `%s' program follows:\n\n%s\n",
		    [self name], buffer);
	}
    }

    [super init];
    
    self->program = name;    

    self->sampler = glGetUniformLocationARB (self->program, "sampler");
    self->factor = glGetUniformLocationARB (self->program, "factor");
    self->detailSampler = glGetUniformLocationARB (self->program,
						   "detailSampler");
    self->turbidity = glGetUniformLocationARB (self->program, "turbidity");
    self->rayleigh = glGetUniformLocationARB (self->program, "beta_r");
    self->mie = glGetUniformLocationARB (self->program, "beta_p");
    self->sunDirection = glGetUniformLocationARB (self->program,
						  "sunDirection");
    self->sunColor = glGetUniformLocationARB (self->program,
					      "sunColor");

   
    self->albedo = 1;
				       
    return self;
}

-(void) get
{
    const char *k;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "albedo")) {
	lua_pushnumber (_L, self->albedo);
    } else {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "albedo")) {
	self->albedo = lua_tonumber (_L, -1);
    } else {
	[super set];
    }
}

-(void) traversePass: (int)pass
{
    id parent, child;

    if (pass == 1) {
	glEnable (GL_DEPTH_TEST);
	glEnable (GL_CULL_FACE);

/*     glPolygonMode (GL_FRONT_AND_BACK, GL_LINE); */
	glUseProgramObjectARB(self->program);
	glUniform1iARB (self->sampler, 0);
	glUniform1iARB (self->detailSampler, 1);
	glUniform1fARB (self->factor, self->albedo);

	for (parent = [self parent] ; parent ; parent = [parent parent]) {
	    for (child = [parent children] ; child ; child = [child sister]) {
		if ([child isMemberOf: [Atmosphere class]]) {
		    glUniform3fvARB (self->sunDirection, 1, [child direction]);
		    glUniform3fvARB (self->sunColor, 1, [child sunlight]);
		    glUniform3fvARB (self->rayleigh, 1, [child rayleigh]);
		    glUniform1fARB (self->mie, [child mie]);
		    glUniform1fARB (self->turbidity, [child turbidity]);
		}
	    }
	}

	[super traversePass: pass];

	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
    } else {
	[super traversePass: pass];
    }
}

@end
