/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "system.h"
#include "body.h"

@implementation Body

-(Body *) init
{
    [super init];
    
    dMassSetZero(&self->mass);
    
    self->mass.mass = 1;
    
    self->mass.I[0] = 1;
    self->mass.I[1] = 0;
    self->mass.I[2] = 0;
    self->mass.I[3] = 0;

    self->mass.I[4] = 0;
    self->mass.I[5] = 1;
    self->mass.I[6] = 0;
    self->mass.I[7] = 0;

    self->mass.I[8] = 0;
    self->mass.I[9] = 0;
    self->mass.I[10] = 1;
    self->mass.I[11] = 0;

    self->body = NULL;

    /* Create and attach the body. */
    
    [self release];
    
    return self;
}

-(void) free
{
    if (self->geom) {
	dGeomSetBody (self->geom, NULL);
	dGeomDestroy (self->geom);
    }
    
    if (self->body) {
	dBodyDestroy (self->body);
    }
}

-(dBodyID) body
{
    return self->body;
}

-(dGeomID) geom
{
    return self->geom;
}
 
-(dSpaceID) space
{
    return self->space;
}

-(void) fasten
{
    assert (self->body);

    dBodyDestroy (self->body);
    self->body = NULL;
    
    if (self->geom) {
	dGeomSetBody (self->geom, NULL);
    }
}

-(void) release
{
    assert (!self->body);

    self->body = dBodyCreate (_WORLD);
    
    dBodySetData (self->body, self);
    dBodySetMass (self->body, &self->mass);
    dBodySetFiniteRotationMode (self->body, 1);
    dBodyDisable (self->body);

    if (self->geom) {
	dGeomSetBody (self->geom, self->body);
    }
}

-(void) insertInto: (dSpaceID) new
{
    assert (!self->linked);
    assert (!self->space || !new);

    if (self->space && self->geom) {
	dSpaceRemove (self->space, self->geom);
    }	
    
    if (new && self->geom) {
	dSpaceAdd (new, self->geom);
    }
    
    self->space = new;
}

-(dReal *) velocity
{
    return self->velocity;
}

-(dReal *) spin
{
    return self->spin;
}

-(void) toggle
{
    if (!linked) {
	if (self->body) {
	    dBodyEnable (self->body);
	}
    } else {
	if (self->body) {
	    dBodyDisable (self->body);
	}
    }

    [super toggle];
}

-(void) stepBy: (double) h
{
    [super stepBy: h];
}

-(void) transform
{
    dVector3 zero = {0, 0, 0};
    dReal *r, *R, *drdt, *dRdt;
    int i, j;
    
    if (self->body) {
	drdt = (dReal *)dBodyGetLinearVel (self->body);
	dRdt = (dReal *)dBodyGetAngularVel (self->body);
    } else {
	drdt = zero;
	dRdt = zero;
    }

    if(self->geom &&
       dGeomGetClass(self->geom) != dPlaneClass) {
	r = (dReal *)dGeomGetPosition (self->geom);
	R = (dReal *)dGeomGetRotation (self->geom);
    } else if (self->body) {
	r = (dReal *)dBodyGetPosition (self->body);
	R = (dReal *)dBodyGetRotation (self->body);
    } else {
	[super transform];
	return;
    }
    
    for(i = 0 ; i < 3 ; i += 1) {
	self->position[i] = r[i];
	self->velocity[i] = drdt[i];
	self->spin[i] = dRdt[i];
    }
	    
    for(i = 0 ; i < 3 ; i += 1) {
	for(j = 0 ; j < 3 ; j += 1) {
	    self->orientation[i * 3 + j] = R[i * 4 + j];
	}
    }

    [super transform];
}

-(void) get
{
    const char *k;
    int i, j;
    
    k = lua_tostring (_L, -1);

    if (!xstrcmp(k, "velocity")) {
        lua_newtable (_L);
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->velocity[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "spin")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->spin[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "mass")) {
        lua_newtable (_L);

        lua_pushnumber (_L, self->mass.mass);
        lua_rawseti (_L, -2, 1);

        lua_newtable (_L);

        for(i = 0 ; i < 3 ; i += 1) {
            lua_pushnumber (_L, self->mass.c[i]);
            lua_rawseti (_L, -2, i + 1);
        }

        lua_rawseti (_L, -2, 2);

        lua_newtable (_L);
	
	for(i = 0 ; i < 3 ; i += 1) {
	    for(j = 0 ; j < 3 ; j += 1) {
		lua_pushnumber (_L, self->mass.I[i * 4 + j]);
		lua_rawseti (_L, -2, i * 3 + j + 1);
	    }
	}

        lua_rawseti (_L, -2, 3);
    } else if (!xstrcmp(k, "classification")) {
	unsigned int i, mask;
	
	mask = dGeomGetCategoryBits (self->geom);
	lua_newtable (_L);
	
	for (i = 0 ; i < 32 ; i += 1) {
	    if (mask & ((unsigned int)0x1 << i)) {
		int n;

		n = lua_objlen (_L, -1);
		lua_pushnumber (_L, i + 1);
		lua_rawseti (_L, -2, n + 1);
	    }
	}

	if (lua_objlen (_L, -1) == 1) {
	    lua_rawgeti (_L, -1, 1);
	}
    } else if (!xstrcmp(k, "lethargy")) {
	if (self->body && dBodyGetAutoDisableFlag (self->body)) {
	    lua_newtable (_L);

	    lua_pushnumber (_L, dBodyGetAutoDisableLinearThreshold(self->body));
	    lua_rawseti (_L, -2, 1);

	    lua_pushnumber (_L, dBodyGetAutoDisableAngularThreshold(self->body));
	    lua_rawseti (_L, -2, 2);

	    lua_pushnumber (_L, dBodyGetAutoDisableSteps(self->body));
	    lua_rawseti (_L, -2, 3);

	    lua_pushnumber (_L, dBodyGetAutoDisableTime(self->body));
	    lua_rawseti (_L, -2, 4);
	} else {
	    lua_pushboolean (_L, 0);
	}
    } else if (!xstrcmp(k, "damping")) {
	if (self->body) {
	    lua_newtable (_L);

/* 	    lua_pushnumber (_L, dBodyGetLinearDampingThreshold(self->body)); */
/* 	    lua_rawseti (_L, -2, 1); */

/* 	    lua_pushnumber (_L, dBodyGetAngularDampingThreshold(self->body)); */
/* 	    lua_rawseti (_L, -2, 2); */

/* 	    lua_pushnumber (_L, dBodyGetLinearDamping(self->body)); */
/* 	    lua_rawseti (_L, -2, 3); */

/* 	    lua_pushnumber (_L, dBodyGetAngularDamping(self->body)); */
/* 	    lua_rawseti (_L, -2, 4); */
	} else {
	    lua_pushboolean (_L, 0);
	}
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i, j;

    k = lua_tostring (_L, -2);

    if (!xstrcmp(k, "velocity")) {
	if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->velocity[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }

	if (self->body) {
	    dBodySetLinearVel (self->body,
			       self->velocity[0],
			       self->velocity[1],
			       self->velocity[2]);
	}
    } else if (!xstrcmp(k, "spin")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->spin[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
	
	if (self->body) {
	    dBodySetAngularVel (self->body,
				self->spin[0],
				self->spin[1],
				self->spin[2]);
	}
    } else if (!xstrcmp(k, "mass")) {
	dMassSetZero(&self->mass);
	
        if(lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 1);
	    
	    self->mass.mass = lua_tonumber (_L, -1);

	    lua_pop (_L, 1);
	
	    lua_rawgeti (_L, 3, 2);

	    for(i = 0 ; i < 3 ; i += 1) {
		lua_rawgeti (_L, -1, i + 1);
		
		self->mass.c[i] = lua_tonumber (_L, -1);
		
		lua_pop (_L, 1);
	    }
	
	    lua_pop (_L, 1);

	    lua_rawgeti (_L, 3, 3);
	
	    for(i = 0 ; i < 3 ; i += 1) {
		for(j = 0 ; j < 3 ; j += 1) {
		    lua_rawgeti (_L, -1, i * 3 + j + 1);

		    self->mass.I[i * 4 + j] = lua_tonumber (_L, -1);

		    lua_pop (_L, 1);
		}
	    }

	    lua_pop (_L, 1);
	}

	if (self->body) {
	    dBodySetMass(self->body, &self->mass);
	}
    } else if (!xstrcmp(k, "classification")) {
	unsigned int i, n, mask;

	if (lua_istable (_L, 3)) {
	    mask = 0;
	    n = lua_objlen (_L, 3);
	    
	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (_L, 3, i + 1);
		mask |= (unsigned int)0x1 << ((int)lua_tonumber (_L, -1) - 1);
		lua_pop (_L, 1);
	    }
	} else if (lua_isnumber (_L, 3)) {
	    mask = 1 << ((int)lua_tonumber (_L, 3) - 1);
	} else if (lua_toboolean (_L, 3)) {
	    mask = ~0;
	} else {
	    mask = 0;
	}
	    
	if (self->geom) {
	    dGeomSetCategoryBits (self->geom, mask);
	    dGeomSetCollideBits (self->geom, mask);
	}
    } else if (!xstrcmp(k, "lethargy")) {
	if (self->body) {
	    if (lua_istable (_L, 3)) {
		dBodySetAutoDisableFlag (self->body, 1);

		lua_rawgeti (_L, 3, 1);
		dBodySetAutoDisableLinearThreshold (self->body,
						    lua_tonumber (_L, -1));
		lua_pop (_L, 1);

		lua_rawgeti (_L, 3, 2);
		dBodySetAutoDisableAngularThreshold (self->body,
						     lua_tonumber (_L, -1));
		lua_pop (_L, 1);

		lua_rawgeti (_L, 3, 3);
		dBodySetAutoDisableSteps (self->body, lua_tonumber (_L, -1));
		lua_pop (_L, 1);

		lua_rawgeti (_L, 3, 4);
		dBodySetAutoDisableTime (self->body, lua_tonumber (_L, -1));
		lua_pop (_L, 1);
	    } else if (lua_isboolean (_L, 3)) {
		dBodySetAutoDisableFlag (self->body, lua_toboolean (_L, 3));
	    } else if (lua_isnil (_L, 3)) {
		dBodySetAutoDisableFlag (self->body, 0);
	    }   
	}
    } else if (!xstrcmp(k, "position")) {
	[super set];

	if (self->body) {
	    dBodySetPosition (self->body,
			      self->position[0],
			      self->position[1],
			      self->position[2]);
	}

	if (self->geom &&
	    dGeomGetClass(self->geom) != dPlaneClass) {
	    dGeomSetPosition (self->geom,
			      self->position[0],
			      self->position[1],
			      self->position[2]);
	    dGeomSetPosition (self->geom,
			      self->position[0],
			      self->position[1],
			      self->position[2]);
	}
    } else if (!xstrcmp(k, "orientation")) {
	dMatrix3 R;
	int i, j;
	    
	[super set];

	dRSetIdentity(R);

	for(i = 0 ; i < 3 ; i += 1) {
	    for(j = 0 ; j < 3 ; j += 1) {
		R[i * 4 + j] = self->orientation[i * 3 + j];
	    }
	}
	    
	if (self->body) {
	    dBodySetRotation (self->body, R);
	}
	    
	if (self->geom &&
	    dGeomGetClass(self->geom) != dPlaneClass) {
	    dGeomSetRotation (self->geom, R);
	}
    } else {
	[super set];
    }
}

@end
