/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include "altimeter.h"

@implementation Altimeter

-(Altimeter *)init
{
    [super init];

    self->reading = 0;
    self->thickness = 1;
   
    self->radius[0] = 0.1;
    self->radius[1] = 0.07;
    
    return self;
}

-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "spread")) {
	lua_newtable (_L);
	lua_pushnumber (_L, 0);
	lua_rawseti (_L, -2, 1);
	lua_pushnumber (_L, 2 * M_PI);
	lua_rawseti (_L, -2, 1);
    } else if (!xstrcmp(k, "spacing")) {
	lua_newtable (_L);
	lua_pushnumber (_L, 2);
	lua_rawseti (_L, -2, 1);
	lua_pushnumber (_L, 5);
	lua_rawseti (_L, -2, 1);
    } else if (!xstrcmp(k, "range")) {
	lua_newtable (_L);
	lua_pushnumber (_L, 0);
	lua_rawseti (_L, -2, 1);
	lua_pushnumber (_L, 100);
	lua_rawseti (_L, -2, 1);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "spread")) {
    } else if (!xstrcmp(k, "spacing")) {
    } else if (!xstrcmp(k, "range")) {
    } else {
	[super set];
    }
}

-(void) transform
{
    id child;
    int i;
    
    self->minimum[0] = 2 * (self->padding[0] + self->radius[0]);
    self->minimum[1] = 2 * (self->padding[1] + self->radius[0]);

    self->allocated[0] = self->minimum[0] > self->requested[0] ?
	                 self->minimum[0] : self->requested[0];
    self->allocated[1] = self->minimum[1] > self->requested[1] ?
	                 self->minimum[1] : self->requested[1];
    
    if (self->align[0] < 0) {
	self->position[0] += 0.5 * (self->minimum[0] - self->allocated[0]);
    } else if (self->align[0] > 0) {
	self->position[0] += 0.5 * (self->allocated[0] - self->minimum[0]);
    }

    if (self->align[1] < 0) {
	self->position[1] += 0.5 * (self->minimum[1] - self->allocated[1]);
    } else if (self->align[1] > 0) {
	self->position[1] += 0.5 * (self->allocated[1] - self->minimum[1]);
    }

    for(child = [self children] ; child ; child = [child sister]) {
	GLfloat delta[3];
	float theta;
	
	i = atoi([child key]);
	
	if (i > 0 && i <= 10) {
	    theta = (i - 1) * 2 * M_PI / 10;

	    delta[0] = self->radius[1] * sin (theta);
	    delta[1] = self->radius[1] * cos (theta);
	    delta[2] = 0;
	
	    [child setPositionTo: delta];
	}
    }

    [super transform];
}

-(void) traversePass: (int)pass
{
    if (pass == 2) {
	int i;

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixf ([self homogenous]);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glLineWidth (self->thickness);

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);

	glDepthMask (GL_FALSE);

	/* The face. */

	self->background[3] = self->color[3];
	glColor4fv(self->background);

	glBegin (GL_TRIANGLE_FAN);

	glVertex2f (0, 0);
	for (i = 0 ; i <= 32 ; i += 1) {
	    glVertex2f(self->radius[0] * cos (2 * i * M_PI / 32),
		       self->radius[0] * sin (2 * i * M_PI / 32));
	}
   
	glEnd();
	
	glColor3fv([self color]);

	glBegin (GL_LINE_STRIP);

	for (i = 0 ; i <= 32 ; i += 1) {
	    glVertex2f(self->radius[0] * cos (2 * i * M_PI / 32),
		       self->radius[0] * sin (2 * i * M_PI / 32));
	}
   
	glEnd();

	/* Subdivisions. */
	
	glBegin (GL_LINES);

	for (i = 0 ; i <= 50 ; i += 1) {
	    float theta, r_0;
	    
	    r_0 = i % 5 ? 0.9 : 0.85;
	    theta = i * 2 * M_PI / 50;
	    
	    glVertex2f(0.95 * self->radius[0] * sin (theta),
		       0.95 * self->radius[0] * cos (theta));
	    
	    glVertex2f(r_0 * self->radius[0] * sin (theta),
		       r_0 * self->radius[0] * cos (theta));
	}

	glEnd();

	/* The hundreds hand. */
	
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();

	glRotatef (self->reading / 100 / 10 * 2 * M_PI / M_PI * 180.0,
		   0, 0, -1);

	glBegin (GL_LINE_STRIP);

	glVertex2f(-0.04 * self->radius[0], 0.6 * self->radius[0]);
	glVertex2f(0, 0.7 * self->radius[0]);
	glVertex2f(0.04 * self->radius[0], 0.6 * self->radius[0]);
	glVertex2f(0.04 * self->radius[0], -0.4 * self->radius[0]);
	glVertex2f(0, -0.42 * self->radius[0]);
	glVertex2f(-0.04 * self->radius[0], -0.4 * self->radius[0]);
	glVertex2f(-0.04 * self->radius[0], 0.6 * self->radius[0]);

	glEnd();

	glPopMatrix();

	/* The thousands hand. */
	
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();

	glRotatef (self->reading / 1000 / 10 * 2 * M_PI / M_PI * 180.0,
		   0, 0, -1);

	glBegin (GL_LINE_STRIP);

	glVertex2f(-0.07 * self->radius[0], 0.25 * self->radius[0]);
	glVertex2f(0, 0.45 * self->radius[0]);
	glVertex2f(0.07 * self->radius[0], 0.25 * self->radius[0]);
	glVertex2f(0.04 * self->radius[0], 0);
	glVertex2f(0.07 * self->radius[0], -0.2 * self->radius[0]);
	glVertex2f(0, -0.22 * self->radius[0]);
	glVertex2f(-0.07 * self->radius[0], -0.2 * self->radius[0]);
	glVertex2f(-0.04 * self->radius[0], 0);
	glVertex2f(-0.07 * self->radius[0], 0.25 * self->radius[0]);

	glEnd();

	glPopMatrix();

	/* The tenthousands hand. */
	
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix ();

	glRotatef (self->reading / 10000 / 10 * 2 * M_PI / M_PI * 180.0,
		   0, 0, -1);

	glBegin (GL_LINE_STRIP);

	glVertex2f(-0.02 * self->radius[0], 0);
	glVertex2f(-0.02 * self->radius[0], 0.8 * self->radius[0]);
	glVertex2f(-0.1 * self->radius[0], 0.9 * self->radius[0]);
	glVertex2f(0, 0.92 * self->radius[0]);
	glVertex2f(0.1 * self->radius[0], 0.9 * self->radius[0]);
	glVertex2f(0.02 * self->radius[0], 0.8 * self->radius[0]);
	glVertex2f(0.02 * self->radius[0], 0);

	glEnd();

	glPopMatrix();
	
	glDepthMask (GL_TRUE);

	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
    
	glMatrixMode (GL_MODELVIEW);
	glPopMatrix();
    }
    
    [super traversePass: pass];
}

@end
