/* This file is part of TCD 2.0.
   tracked.c - Curses track editor module.

   Copyright (C) 1997-98 Tim P. Gerla <timg@rrv.net>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
                   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
                                   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
                                               
   Tim P. Gerla
   RR 1, Box 40
   Climax, MN  56523
   timg@rrv.net
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#if HAVE_NCURSES_H
# include <ncurses.h>
#elif HAVE_CURSES_H
# include <curses.h>
#else
# error No curses header found.
#endif

#include "cddb.h"
#include "tcd.h"
#include "tracked.h"
#include "ui-layout.h"

static struct coords disc_title, tracks, cmds, hbar;

static void update_coords(void)
{
    int maxy, maxx;
    getmaxyx(stdscr, maxy, maxx);
    set_coords(&disc_title, 1, 2, 2, maxx - 2);
    set_coords(&cmds, maxy - 4, 2, maxy - 1, maxx - 2);
    set_coords(&hbar, cmds.top - 1, 0, cmds.top, maxx);
    set_coords(&tracks, disc_title.bottom + 1, 2, hbar.top, maxx - 2);
}

static int int_min(int a, int b)
{
    return (a < b) ? a : b;
}

static int list_offset = 0;
static int list_lineno = 0;

static void update_tracklist(struct tcd_state *state)
{
    const int max_i = int_min(tracks.height, state->cdrom->numtracks);
    int i;
    for (i = 0; i < max_i; i++) {
        if (i == list_lineno) {
            attron(A_REVERSE);
        }
        mvprintw(tracks.top + i, tracks.left, "%02d: %-*.*s",
                 1 + list_offset + i, tracks.width - 4, tracks.width - 4,
                 state->cd_info.trk[list_offset + i].name);
        if (i == list_lineno) {
            attroff(A_REVERSE);
        }
    }
}

static void update_screen(struct tcd_state *state)
{
    erase();
    update_coords();

    attron(C_BLUE);
    box(stdscr, ACS_VLINE, ACS_HLINE);
    mvaddch(hbar.top, hbar.left, ACS_LTEE);
    mvhline(hbar.top, hbar.left + 1, ACS_HLINE, hbar.width - 2);
    mvaddch(hbar.top, hbar.right - 1, ACS_RTEE);
    attroff(C_BLUE);

    attron(C_WHITE | A_BOLD);
    mvprintw(0, 2, "TCD Track List Editor");
    mvprintw(disc_title.top, disc_title.left, "%-*.*s",
             disc_title.width, disc_title.width, state->cd_info.disc_title);
    attroff(C_WHITE | A_BOLD);

    attron(C_YELLOW | A_BOLD);
    mvaddstr(cmds.top + 1, cmds.left +  0, "[E]");
    mvaddstr(cmds.top + 1, cmds.left + 25, "[T]");
    mvaddstr(cmds.top + 2, cmds.left +  0, "[Q]");
    mvaddstr(cmds.top + 2, cmds.left + 25, "[C]");
    attroff(C_YELLOW | A_BOLD);

    mvaddstr(cmds.top + 1, cmds.left +  3, " - Edit track name");
    mvaddstr(cmds.top + 1, cmds.left + 28, " - Change disc title");
    mvaddstr(cmds.top + 2, cmds.left +  3, " - Exit to player");
    mvaddstr(cmds.top + 2, cmds.left + 28, " - Clear all tracks");

    update_tracklist(state);
}

static void handle_edit(struct tcd_state *state)
{
    int local_track = list_offset + list_lineno;
    char tmp[sizeof(state->cd_info.trk[local_track].name)];

    input_box(tmp, sizeof(tmp), "Track name");
    if (strcmp(tmp, "") != 0) {
        strcpy(state->cd_info.trk[local_track].name, tmp);
        state->cd_info.modified = 1;
    }
}

static void handle_down(struct tcd_state *state)
{
    if (list_offset + list_lineno < state->cdrom->numtracks - 1) {
        if (list_lineno < tracks.height - 1) {
            list_lineno++;
        } else {
            list_offset++;
        }
    }
}

static void handle_up(void)
{
    if (list_offset + list_lineno > 0) {
        if (list_lineno > 0) {
            list_lineno--;
        } else {
            list_offset--;
        }
    }
}

static void handle_change_title(struct tcd_state *state)
{
    char tmp[sizeof(state->cd_info.disc_title)];

    input_box(tmp, sizeof(tmp), "Artist / Album");
    if (strcmp(tmp, "") != 0) {
        strcpy(state->cd_info.disc_title, tmp);
        state->cd_info.modified = 1;
    }
}

static void handle_clearall(struct tcd_state *state)
{
    int i;
    for (i = 0; i < state->cdrom->numtracks; i++) {
        sprintf(state->cd_info.trk[i].name, "Track %d", 1 + i);
        state->cd_info.modified = 1;
    }
}

extern void edit_trackdb(struct tcd_state *state)
{
    int stop = FALSE;
    int c;

    curs_set(0);
    while (!stop) {
        update_screen(state);
        switch (c = getch()) {
        case 'q':
        case 'Q':
            tcd_writediskinfo(&state->cd_info, state->cdrom);
            stop = TRUE;
            break;
        case 'E':
        case 'e':
        case KEY_ENTER:
        case 012:
            handle_edit(state);
            break;
        case 'D':
        case 'd':
        case KEY_DOWN:
            handle_down(state);
            break;
        case 'U':
        case 'u':
        case KEY_UP:
            handle_up();
            break;
        case 't':
        case 'T':
            handle_change_title(state);
            break;
        case 'c':
        case 'C':
            handle_clearall(state);
            break;
        case ERR:
        default:
            wrefresh(stdscr);
            break;
        }
    }
    curs_set(1);
}
