/**
 * @file wall.c
 * Write a message to all users
 *
 * Copyright (C) 2004, 2005, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _GNU_SOURCE		/**< strndup() */
#include <argp.h>
#include <pwd.h>
#include <grp.h>
#include <time.h>
#include <utmp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/uio.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "wall"		/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("[FILE]");

/** Program synopsis */
static char doc[] =
	N_("Write a message to all users.\n"
	   "\n"
	   "wall displays the contents of FILE or the input from stdin "
	   "on the terminals of all logged in users.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "no-banner", 'n', 0, 0,
	  N_("Disable the banner (root only)"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *fname;		/**< File to send */
	int banner;			/**< 0 to disable banner */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'n':
		args->banner = 0;
		break;

	case ARGP_KEY_INIT:
		args->fname = NULL;
		args->banner = 1;
		break;

	case ARGP_KEY_ARG:
		if (args->fname)
			argp_usage(state);

		args->fname = arg;
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * Send message to all users
 *
 * @param msg a pointer to the message to send
 * @param size the size of the message
 * @return 0 on success, errno on failure
 */
static int send_message(const char *msg, const size_t size)
{
	struct iovec iov;
	struct utmp *ut;
	error_t status = 0;

	/* Set up iovec */
	iov.iov_base = (char *)msg;
	iov.iov_len = (size_t)size;

	/* Rewind utmp database */
	setutent();

	if (errno == EACCES)
		errno = 0;

	/* Send message to all available tty's */
	while ((ut = getutent())) {
		char *tmp, *dst;

		if (!ut->ut_name[0])
			continue;

		/* use-sessreg in /etc/X11/wdm/ can apparently
		 * produce ut_line entries like `:0'...
		 */
		if (ut->ut_line[0] == ':')
			continue;

		/* Only write to user processes... */
		if (ut->ut_type != USER_PROCESS)
			continue;

		if (!(tmp = strndup(ut->ut_line, sizeof (ut->ut_line)))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strndup", strerror(errno));
			status = errno;
			goto EXIT;
		}

		dst = strconcat("/dev/", tmp, NULL);
		free(tmp);

		if ((status = tty_write(&iov, dst))) {
			if (status == EACCES) {
				status = 0;
			} else {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "tty_write", strerror(errno));
				free(dst);
				goto EXIT;
			}
		}

		free(dst);
	}

	/* Close utmp database */
	endutent();

EXIT:

	return status;
}

/**
 * Read a message from stdin
 *
 * @param banner a pointer to the banner to prepend to the message
 * @return a pointer to the message on success, NULL on failure
 */
static char *read_msg(char *banner)
{
	char *tmp = NULL;
	char *tmp2 = NULL;
	size_t n = 0;

	if (getdelim(&tmp, &n, EOF, stdin) == -1) {
		goto EXIT;
	}

	if (!(tmp2 = strconcat(banner ? banner : "", tmp, NULL))) {
		free(tmp);
		goto EXIT;
	}

EXIT:
	return tmp2;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	struct passwd *pw = NULL;
	time_t now;

	error_t status = 0;

	int isroot = 0;

	char nowbuf[16];
	char *msg = NULL;
	char *banner = NULL;
	char *lttyname = NULL;
	char *username = NULL;
	char *hostname = NULL;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Check if the caller has root privileges */
	if ((status = is_root()) &&
	    ((getuid() != geteuid()) || (getgid() != getegid()))) {

		if (status == EPERM) {
			if (!args.banner) {
				fprintf(stderr,
					_("%s: insufficient privileges\n"
					  "You must be root to %s.\n"),
					progname, _("disable the banner"));
				goto EXIT;
			}

			status = 0;
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_root", strerror(errno));
			goto EXIT;
		}
	} else {
		isroot = 1;
	}

	if (args.fname && !isroot) {
		fprintf(stderr,
			_("%s: you must be root to read from a file; "
			  "use stdin\n"),
			progname);
		status = EPERM;
		goto EXIT;
	}

	/* Try to get the current date */
	if (time(&now) == (time_t)-1) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "time()", strerror(errno));
		status = errno;
		goto EXIT;
	} else {
		struct tm ltm;

		/* Convert the time into what we want to display */
		if (!localtime_r(&now, &ltm)) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "localtime_r()",
				strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* A locale-specific version of %R would be nice here...  */
		if (!strftime(nowbuf, sizeof nowbuf,
			      _(/* this is an ugly hack */ "%H:%M"), &ltm)) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strftime()",
				strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (!(pw = getpwuid(getuid()))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "getpwuid()", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (!(username = strdup(pw->pw_name))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strdup()", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (!(hostname = get_host_name())) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_host_name", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (isatty(STDERR_FILENO)) {
		char *tty;

		if (!(tty = ttyname(STDERR_FILENO))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "ttyname()", strerror(errno));
			status = errno;
			goto EXIT;
		} else if (!(lttyname = strdup(tty))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	} else {
		if (!(lttyname = strdup(_("Somewhere")))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (args.banner) {
		if (asprintf(&banner,
			     _("\n"
			       "Broadcast Message from %s@%s\n"
			       "\t(%s) at %s ...\n"
			       "\n"),
			     username, hostname, lttyname, nowbuf) < 0) {
			banner = NULL;
		}
	}

	if (args.fname) {
		if (!freopen(args.fname, "r", stdin)) {
			fprintf(stderr,
				_("%s: error reopening %s as stdin; %s\n"),
				progname, args.fname, strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (!(msg = read_msg(banner))) {
		status = errno;
		goto EXIT;
	}

	send_message(msg, strlen(msg));

EXIT:
	free(lttyname);
	free(hostname);
	free(username);
	free(banner);
	free(msg);

	return status;
}
